#include "DrateCont.h"
#include "errorMsg.h"
#include "McRateUtils.h"
#include "someUtil.h"

#include <cmath>
#include <cassert>
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


DrateCont::DrateCont(const int bar_num, const MDOUBLE lower_limit, const MDOUBLE upper_limit)
: m_distribution(bar_num), m_minimum_limit(lower_limit), m_maximum_limit(upper_limit)
{
	
}


DrateCont::DrateCont(const DrateCont& other)
: m_distribution(other.m_distribution.size()), m_minimum_limit(other.m_minimum_limit), m_maximum_limit(other.m_maximum_limit)
{
}



DrateCont::DrateCont(const VBar& distribution)
: m_minimum_limit(distribution.front().getLowerLimit()),
m_maximum_limit(distribution.back().getUpperLimit())
{
	m_distribution = distribution;
}



DrateCont::~DrateCont()
{

}


VBar DrateCont::getBars() const 
{
	return m_distribution; 
}


//reset the bar distribution so that it will match pInDist 
void DrateCont::resetDistribution(const gammaDistribution* pInDist)
{
	int catNum = pInDist->categories();
	if (catNum != m_distribution.size())
	{
		m_distribution.resize(catNum);
	}

	int i;
	for (i = 0; i < m_distribution.size() - 1; ++i)
	{
		m_distribution[i].setBounderies(pInDist->getBorder(i), pInDist->getBorder(i+1), false);
	}

	//update last bar according to upper limit instead of infinite
	m_distribution[i].setBounderies(pInDist->getBorder(i), m_maximum_limit, false);
}



//scale: multiply the whole distribution by scaleFacotr
void DrateCont::scale(MDOUBLE scaleFactor)
{
	int i;
	for (i = 0; i < m_distribution.size(); ++i)
	{
		MDOUBLE lowBorder = m_distribution[i].getLowerLimit() * scaleFactor;
		MDOUBLE upperBorder = m_distribution[i].getUpperLimit() * scaleFactor;

		//make sure that upper bound is not changed
		if ((upperBorder > m_maximum_limit) || (i == m_distribution.size() - 1)) 
		{
			if (lowBorder > m_maximum_limit)
			{
				lowBorder = m_maximum_limit;
			}
			upperBorder = m_maximum_limit;
		}
		
		m_distribution[i].setBounderies(lowBorder, upperBorder, true);
	}
}




//getExpectation: gets the expectation of the distribution
MDOUBLE DrateCont::getExpectation() const
{
	MDOUBLE res = 0.0;
	MDOUBLE r, Pr, sigmaPr = 0.0;
	int i;
	for (i = 0; i < m_distribution.size(); ++i)
	{
		r = m_distribution[i].getMidPoint();
		Pr = m_distribution[i].getProb();
		res += r * Pr;
		sigmaPr += Pr;
	}

	if (!DEQUAL(sigmaPr, 1.0))
		errorMsg::reportError("total probability != 1.0 in function DrateCont::getExpectation()");

	return res;
}


void DrateCont::setProb(const Vdouble& probVec, const gammaDistribution* pDist)
{
	resetDistribution(pDist);
	assert(m_distribution.size() == pDist->categories());

	int ri;
	for (ri = 0; ri < m_distribution.size(); ++ri)
	{
		m_distribution[ri].setProb(probVec[ri]);
	}
}
