#include <cassert>
#include <iostream>
#include <cmath>
using namespace std;

#include "MRateCont.h"
#include "DrateCont.h"
#include "McRateUtils.h"

#include "errorMsg.h"
#include "someUtil.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


MRateCont::MRateCont(const int bar_num, const MDOUBLE lower_limit, const MDOUBLE upper_limit)
: m_barNum(bar_num), m_lower_limit(lower_limit), m_upper_limit(upper_limit) 
{
	init();
}

MRateCont::MRateCont(const MRateCont& other)
: m_barNum(other.m_barNum), m_lower_limit(other.m_lower_limit), m_upper_limit(other.m_upper_limit),
  MetaRates(other) 
{
	init();
}



MRateCont::~MRateCont()
{
	m_distribution.clear();
}


void MRateCont::init()
{
	m_distribution.resize(m_barNum);
	m_interval = (m_upper_limit - m_lower_limit) / static_cast<MDOUBLE>(m_barNum);
	MDOUBLE low_bound = m_lower_limit;
	MDOUBLE high_bound = 0;
	int i;
	for (i = 0; i< m_barNum; ++i)
	{
		high_bound = low_bound + m_interval;
		m_distribution[i].setBounderies(low_bound, high_bound, false);
		m_distribution[i].setDensity(0.0);
		low_bound = high_bound;
	}
}



//addDRate: adds a DrateCont (k categories rate distribution) to the cummulative distribution 
//from_XXX refers to the bar in <DrateCont rates>
//to_XXX referes to the bar in m_distribution that is being updated
void MRateCont::addDRate(const Drates * pInRates)
{
	const DrateCont* pRates = static_cast<const DrateCont *>(pInRates);

	if (!DEQUAL(m_lower_limit, pRates->getMinBound()) || !DEQUAL(m_upper_limit, pRates->getMaxBound()))
	{
		errorMsg::reportError("cannot add the two distribution - they don't have the same bounderies. in function MRateCont::addDRate");
	}

	VBar from_bars = pRates->getBars();
	int from_bar = 0;
	MDOUBLE from_upper_bound = from_bars[from_bar].getUpperLimit();
	MDOUBLE from_lower_bound = from_bars[from_bar].getLowerLimit();
	int i;
	for (i = 0; i < m_distribution.size(); ++i)
	{
		//add the value of from_bars to each m_distribution[i].
		//if m_distribution[i] falls in 2 from_bars then add the proportional values
		MDOUBLE to_lower_bound = m_distribution[i].getLowerLimit();
		MDOUBLE to_upper_bound = m_distribution[i].getUpperLimit();		
		if (DSMALL_EQUAL(from_upper_bound, to_lower_bound))
		{
			//in case where from_bar is out of bounds for to_bar
			++from_bar;
			if (from_bar >= from_bars.size())
			{
				cerr << "error in function MRateCont::addDRate" << endl;
				updateBar(i, 0);
				continue;
			}
			from_upper_bound = from_bars[from_bar].getUpperLimit();
			from_lower_bound = from_bars[from_bar].getLowerLimit();
		}

		if (DSMALL_EQUAL(from_lower_bound, to_lower_bound) && DBIG_EQUAL(from_upper_bound, to_upper_bound))
		{
			//the simple case where to_bar falls within from_bar
			MDOUBLE addValue = from_bars[from_bar].getDensity();
			updateBar(i, addValue);
		}
		else if (DSMALL_EQUAL(from_lower_bound, to_lower_bound) && DSMALL_EQUAL(from_upper_bound, to_upper_bound))
		{
			//from_bar falls within to_bar --> there are several DRates that have to be added to  add the proportion 
			MDOUBLE ratio_from_bar1 = (from_upper_bound - to_lower_bound) / m_interval;
			MDOUBLE val_from_bar1 = from_bars[from_bar].getDensity() * ratio_from_bar1;

			//check how many Drates to add
			MDOUBLE val = val_from_bar1;
			MDOUBLE ratioToadd;
			while (from_bars[from_bar].getUpperLimit() < to_upper_bound)
			{
				++from_bar;
				from_upper_bound = from_bars[from_bar].getUpperLimit();
				from_lower_bound = from_bars[from_bar].getLowerLimit();

				if (DSMALL_EQUAL(from_upper_bound, to_upper_bound))
				{
					ratioToadd = (from_upper_bound - from_lower_bound) / m_interval;
				}
				else //(from_upper_bound > to_upper_bound)
				{
					ratioToadd = (to_upper_bound - from_lower_bound) / m_interval;
				}
				val += from_bars[from_bar].getDensity() * ratioToadd;
			}
			updateBar(i, val);
			/*MDOUBLE ratio_from_bar1 = (from_upper_bound - to_lower_bound) / m_interval;
			MDOUBLE ratio_from_bar2 = 1 - ratio_from_bar1;

			MDOUBLE val_from_bar1 = from_bars[from_bar].getValue() * ratio_from_bar1;
			MDOUBLE val_from_bar2 = from_bars[from_bar+1].getValue() * ratio_from_bar2;
			updateBar(i, (val_from_bar1 + val_from_bar2) );
			*/
		}
		else
			errorMsg::reportError("error in function MRateCont::addDRate");
	}
	m_weight++; 

}




//addMetaRate: adds a whole MRateCont to this
//if bSameWeight==FALSE then add according to relative weight of each distributon (how many DrateCont it "contains")
//if bSameWeight==TRUE then make a simple adition. the new m_count is doubled
void MRateCont::addMetaRate(const MetaRates * pMetaOther, bool bSameWeight)
{
	const MRateCont* pOther = static_cast<const MRateCont *>(pMetaOther);

	if ((m_lower_limit != pOther->m_lower_limit) || (m_upper_limit != pOther->m_upper_limit))
	{
		errorMsg::reportError("cannot add the two distribution - they don't have the same bounderies. in function MRateCont::addMetaRate()");
	}
	
	if (m_barNum != pOther->m_barNum)
	{
		errorMsg::reportError("cannot add the two distribution - they don't have the same number of bars. in function MRateCont::addMetaRate()");	
	}

	if (bSameWeight == true)
	{
		int i;
		for (i = 0; i < m_distribution.size(); ++i)
		{
			MDOUBLE sum = m_distribution[i].getDensity() * m_weight;
			MDOUBLE otherSum = pOther->m_distribution[i].getDensity() * pOther->m_weight;
			MDOUBLE newVal = (m_distribution[i].getDensity() + pOther->m_distribution[i].getDensity()) / 2;
			m_distribution[i].setDensity(newVal);		
		}
		m_weight += m_weight; 
	}
	else
	{
		int i;
		for (i = 0; i < m_distribution.size(); ++i)
		{
			MDOUBLE sum = m_distribution[i].getDensity() * m_weight;
			MDOUBLE otherSum = pOther->m_distribution[i].getDensity() * pOther->m_weight;
			MDOUBLE newVal = (sum + otherSum) / (m_weight + pOther->m_weight);
			m_distribution[i].setDensity(newVal);		
		}
		m_weight += pOther->m_weight; 
	}

	//check that total probability is 1.0
	MDOUBLE sum = 0.0;
	int i;
	for (i = 0; i < m_distribution.size(); ++i)
	{
		sum += m_distribution[i].getProb();
	}

	if (!DEQUAL(sum, 1.0))
		errorMsg::reportError("total probability is not 1.0 in function MRateCont::addMetaRate()"); 
}


//add a new value to the current bar. 
//this function is called before m_weight is incremented (and so m_weight+1 is used)
void MRateCont::updateBar(const int barNum, const MDOUBLE addValue)
{
	MDOUBLE curSum = m_weight * m_distribution[barNum].getDensity();
	MDOUBLE newVal = (curSum + addValue) / (m_weight + 1);
	m_distribution[barNum].setDensity(newVal);
}


//getExpectation: gets the expectation of the distribution
MDOUBLE MRateCont::getExpectation() const
{
	MDOUBLE res = 0.0;
	MDOUBLE r, Pr, sigmaPr = 0.0;
	int i;
	for (i = 0; i < m_barNum; ++i)
	{
		r = m_distribution[i].getMidPoint();
		Pr = m_distribution[i].getProb();
		res += r * Pr;
		sigmaPr += Pr;
	}

	if (!DEQUAL(sigmaPr, 1.0))
		errorMsg::reportError("total probability != 1.0 in function MRateCont::getExpectation()");


	return res;
}


//getStd: returns the standard deviation of MRateCont
//std = sqrt(E[x^2] - E[x]^2)
MDOUBLE MRateCont::getStd() const
{
	MDOUBLE r, Pr, Ex = 0.0, Ex2 = 0.0;
	int i;
	for (i = 0; i < m_barNum; ++i)
	{
		r = m_distribution[i].getMidPoint();
		Pr = m_distribution[i].getProb();
		Ex += r * Pr;
		Ex2 += r * r * Pr;
	}

	MDOUBLE var = Ex2 - (Ex * Ex);
	if (var < 0.0)
		errorMsg::reportError("variance is negative in in function MRateCont::getStd()");


	return sqrt(var);
}


void MRateCont::printDistribution(ofstream &outFile)
{
	printTime(outFile);
	int i;
	for (i = 0; i < m_distribution.size(); ++i)
	{
		outFile << i <<"\t" <<m_distribution[i].getLowerLimit()<<" - " << m_distribution[i].getUpperLimit() << "\t" << m_distribution[i].getDensity() <<endl;
	}
	
}
