#ifndef ___CHAIN
#define ___CHAIN

#include <fstream>
using namespace std;

#include "McRate_Definitions.h"
#include "McRateOptions.h" 

#include "sequenceContainer1G.h"
#include "treeInterface.h"
using namespace treeInterface;
#include "ProposeAlpha.h"
#include "ProposeGlobalBranch.h"
#include "ProposeLocalBranch.h"
#include "ProposeNNI.h"
#include "DrateCont.h"
#include "MRateCont.h"
#include "Counters.h"
#include "gammaDistribution.h"
#include "stochasticProcess.h"
#include "alphabet.h"

class Chain  
{
public:
	//LOCAL_BRANCH: change only a single branch at a time
	//GLOBAL_BRANCH: change all branches each step
	//ALPHA: change alpha at each step
	//NNI: Topology change - Nearesr Neighbour Interchange
	enum ProposeName {LOCAL_BRANCH =1, GLOBAL_BRANCH, ALPHA, NNI};
public:
	
	explicit Chain(const tree& t1,
			       const sequenceContainer1G & seqContainer, 
				   const stochasticProcess* pSp,
				   const string resFileStr, 
				   const McRateOptions& options);
	explicit Chain(const sequenceContainer1G & seqContainer, 
				   const stochasticProcess* pSp, 
				   const string resFileStr, 
				   const McRateOptions& options);

	virtual ~Chain();

	//makeBurnInSteps(): stepNum= number of burnin steps to make. calibrateCycle= after how many steps to make parameter calibrations. if bTopology = false then make no topology changes
	void makeBurnInSteps(const int stepsNum, const int calibrateCycle, const bool bTopology = false);
	void makeInferenceSteps(const int stepsNum, bool bInferRates, bool bScaleRates, bool bTopology = false);
	
	//resetCounters: reset all the counters to zero
	void resetCounters();	

	//getTree: get the mcmc tree. if bTopology=false then the branch-lengths are the chain average
	tree getBayesTree(const bool bTopology) const;
	//getWeight: return the number of rate inference steps  
	int getWeight() const {return m_MetaRates[0]->getWeight();}
	//getBayesRates: return the average rate4site over the whole chain 
	Vdouble getBayesRates() const {return m_expRates;}
	//getAlpha: return the average alpha over the whole chain 
	MDOUBLE getAlpha() const {return m_avgAlpha;}

	void printRates(ofstream& outFile, const bool bDrate = false);

	MDOUBLE& operator[] (int index) {return m_expRates[index];}
	const MDOUBLE& operator[] (int index) const {return m_expRates[index];}

private:
	void init(const stochasticProcess* pSp,
			  const string baseStr, 
			  const McRateOptions& options);
	//initialize the proposals probabilities vector
	void initProposalProbs();


	int seqLen() const {return m_seqContainer.seqLen();}
	int getTotalSteps() const {return m_Counters[0].getTotalStepNum();}
	//initialize the starting tree 
	void initTree(const tree& inTree);
	//initialize a tree from one possible NJ tree
	void setRandomTree();

	//propose a new state - decide if to accept or reject it. if bTopology=false then make no topology changes
	bool makeSingleStep(const bool bTopology);
	//calibrate: Calibrates all proposal types according to their current acceptance levels (during burnIn only)
	void calibrate(); 
	//decide whether to accept or reject the proposed state. update the counter of the proposed type 
	bool IsNewStateAccepted(const tree& proptree, const MDOUBLE hastingsRatio, const MDOUBLE priorRatio, const ProposeName proName);
	//decide which proposed type to do
	ProposeName getStepType(const bool bTopology) const;

	void printCurrentState(); 

	void findRates(const bool bScaleRates);
	//compute the rate vector (Drate for each site)
	void computeRate4site();

private:
	const sequenceContainer1G  m_seqContainer; // the multiple alignment
	const stochasticProcess* m_pSp; // the stochastic process matrix
	tree m_tree; //the current tree of the chain
	distribution* m_pPriorDist; //should be same (gamma) distribution as in m_pSp->distr() but can have different number of cateories for a better resolution for calculating rates


	const int m_kThining; // sampling the chain every k step
	ProposeAlpha m_propAlpha;
	ProposeGlobalBranch m_propGlobal;
	ProposeLocalBranch m_propLocal;
	ProposeNNI m_propNni;
	vector<Counters> m_Counters;
	int m_sampledSteps; //number of sampled steps  
	
	vector<ProposeName> m_proposalProb; //each entry represents 10% probability that the specified proposal type will be choosed

	vector<MetaRates*> m_MetaRates; // a vector of MetaRates for each position (amino acid)
	vector<Drates*> m_currentRates; // a vector of Drates for each position.these are the rates of the current state of the chain
	Vdouble m_expRates; //holds the average rate (over the entire chain) for each site
	Vdouble m_currentExpRates; //holds the average rate for the current tree for each site
	Vdouble m_expBranches; //holds the average length (over the entire chain) of each branch - can be calculated only when the topology of the tree does not change
	MDOUBLE m_avgAlpha; //holds the average alpha (over the entire chain)
	MDOUBLE m_curLikelihood;
	bool m_bChanged; // Indicates whether the tree changes since the last tree update
	
	ofstream m_resFile; // For the all the output of each step (likelihood, alpha ..)
	ofstream m_logFile; // For extra output as acceptence rate 
	ofstream m_treesFile;
};

#endif // ___CHAIN