 /* Copyright (C) 2007 the NxOS developers
 *
 * See AUTHORS for a full list of the developers.
 *
 * Redistribution of this file is permitted under
 * the terms of the GNU Public License (GPL) version 2.
 * 
 * Modified by Sivan Toledo.
 */

/* The following are easier mnemonics for the pins used by the
 * tachymeter. Each motor has a tach-pulse pin whose value flips at
 * regular rotation intervals, and a direction pin whose value depends
 * on the direction of rotation.
 */
#define MOTOR_A_TACH AT91C_PIO_PA15
#define MOTOR_A_DIR  AT91C_PIO_PA1
#define MOTOR_B_TACH AT91C_PIO_PA26
#define MOTOR_B_DIR  AT91C_PIO_PA9
#define MOTOR_C_TACH AT91C_PIO_PA0
#define MOTOR_C_DIR  AT91C_PIO_PA8

/* 
 * We often need to manipulate all tach pins and/or all dir pins
 * together. Lets's simplify doing that.
 */
#define MOTORS_TACH (MOTOR_A_TACH | MOTOR_B_TACH | MOTOR_C_TACH)
#define MOTORS_DIR  (MOTOR_A_DIR  | MOTOR_B_DIR  | MOTOR_C_DIR )
#define MOTORS_ALL  (MOTORS_TACH | MOTORS_DIR)

/* The pin mapping for the motor tacho inputs. */
static struct {
  uint32_t tach;
  uint32_t dir;
  int32_t  current;
} _tacho_state[NXT_N_MOTORS] = {
  { MOTOR_A_TACH, MOTOR_A_DIR, 0 },
  { MOTOR_B_TACH, MOTOR_B_DIR, 0 },
  { MOTOR_C_TACH, MOTOR_C_DIR, 0 },
};

/* 
 * Tachometer interrupt handler, triggered by a change 
 * of value of a tachometer pin.
 */
static
void __attribute__ ((interrupt("IRQ"))) _tacho_isr(void) {
  int i;
  uint32_t changes;
  uint32_t pins;
  uint32_t time;

  /* Acknowledge the interrupt and grab the state of the pins. */
  changes = *AT91C_PIOA_ISR;
  pins    = *AT91C_PIOA_PDSR;

  /* Grab the time, as we're going to use it to check for timed
   * rotation end.
   */
  time = systick;

  /* Check each motor's tachymeter. */
  for (i=0; i<NXT_N_MOTORS; i++) {
    if (changes & _tacho_state[i].tach) {
      uint32_t tach = pins & _tacho_state[i].tach;
      uint32_t dir  = pins & _tacho_state[i].dir;

      /* If the tachymeter pin value is the opposite the direction pin
       * value, then the motor is rotating 'forwards' (positive speed
       * value given to start it). Otherwise, it's rotating
       * 'backwards', and we should decrement the tachymeter count
       * instead of incrementing it.
       */
      if ((tach && !dir) || (!tach && dir))
        _tacho_state[i].current++;
      else
        _tacho_state[i].current--;
    }
  }
  tacho_change_event();

  aicUpdatePriority();
}

/*
 * Initialize before interrupts are enabled.
 */
static
void tachoInit(void)
{
  /* Enable the PIO controller. */
  *AT91C_PMC_PCER = (1 << AT91C_ID_PIOA);

  /* Disable all PIO interrupts until we are ready to handle them. */
  *AT91C_PIOA_IDR = ~0;

  /* Configure all tachymeter pins:
   *  - Enable input glitch filtering
   *  - Disable pull-up (externally driven pins)
   *  - Assign pins to the PIO controller
   *  - Set pins to be inputs
   */
  *AT91C_PIOA_IFER  = MOTORS_ALL;
  *AT91C_PIOA_PPUDR = MOTORS_ALL;
  *AT91C_PIOA_PER   = MOTORS_ALL;
  *AT91C_PIOA_ODR   = MOTORS_ALL;

  aicInstallIsr(AT91C_ID_PIOA, AIC_PRIO_MEDIUMLOW, AIC_TRIG_LEVEL, _tacho_isr);

  /* Enable interrupts on changes to the state of the tachy pins. */
  *AT91C_PIOA_IER = MOTORS_TACH;
}

static inline int32_t tachoGetCount(uint32_t motor) {
  return _tacho_state[motor].current;
}
