/*
 * Sivan Toledo, May 2008.
 * 
 * Distributed under GPL 2.
 * 
 * systick maintenance (1ms counter), invocation of the 
 * systick periodic task every 1ms, and busywait routines.
 * 
 */
volatile uint32_t systick;

/*
 * The system interrupt source includes the interval timer,
 * real-time clock, memory controller, and power-management controller.
 */
 
void __attribute__ ((interrupt("IRQ"))) _system_isr(void) {
  if (*AT91C_PITC_PISR & 1) { /* interval timer has expired */
    /* 
     * Read the count field. This acknowledges the interrupt
     * and tells us how many periods we missed, if the ISR
     * was delayed. That way, we maintain systick correctly
     * even if the ISR is not invoked for a few milliseconds.
     */
    int count = (*AT91C_PITC_PIVR >> 20); /* this acknowledges the interrupt */
    
    systick += count;
    
    systick_periodic_task();   
  }
  
  aicUpdatePriority(); /* tell AIC we are done */
}

static uint32_t _busywaitItersPerMsec;

/* 
 * wait u microseconds
 * or   u/1024 = u >> 10 milliseconds
 * or   (u>>10) * _busywaitItersPerMsec iterations
 * 
 * We have to be careful of overflows...
 */

static void busywait(uint32_t usec) {
  uint32_t i, iters;
  if (usec > 32768)
    iters = ((usec >> 5)*_busywaitItersPerMsec) >> 5;
  else
    iters = (usec*_busywaitItersPerMsec) >> 10;
  for (i = 0; i<iters; i++)
    asm volatile ("nop");
}

static
void busywaitLong(uint32_t ms) {
  uint32_t target = systick + ms + 1; /* +1 to avoid quick roll over */
  
  while (systick < target);
}

static void _busywaitInit() {
  uint32_t count;
  uint32_t iter_count = 1048576;
  uint8_t  string[17];
  int32_t  i;
    
  _busywaitItersPerMsec = 1;

  *AT91C_PITC_PIMR = AT91C_PITC_PITEN
                   | 1; /* clocked at MCK/16, and a period of 2, so MCK/32 */
  count = *AT91C_PITC_PIVR; /* just to clear the overflow counter */

  busywait(iter_count);
  
  count = (*AT91C_PITC_PIVR >> 20); /* this acknowledges the interrupt */
    
  /*
   * the delay loop performs about
   *   pclk_freq*iter_count / (pclk_count * 1000000)
   * iterations per microsecond
   */
   
  _busywaitItersPerMsec = (MCK_FREQ/32) / count; 
}

/* 
 * interval is a 20-bit number that counts mck/16 periods, with
 * an overflow and reset every interval+1 mck/16 periods.
 */
static
void systickInit(void) {
  uint32_t interval = ((MCK_FREQ/16) / 1000)-1;
  
  /*
   * busywaitInit also uses the PIT to estimate the delay-loop
   * counter, so we run it before setting the PIT.
   */
  _busywaitInit(); 
  
  *AT91C_PITC_PIMR = AT91C_PITC_PITEN
                   | AT91C_PITC_PITIEN
                   | interval;             /* 20-bit interval */

  aicInstallIsr(AT91C_ID_SYS, AIC_PRIO_VERYLOW, AIC_TRIG_LEVEL, _system_isr);
}
