 /* Copyright (C) 2007 the NxOS developers
 *
 * See AUTHORS for a full list of the developers.
 *
 * Redistribution of this file is permitted under
 * the terms of the GNU Public License (GPL) version 2.
 * 
 * Modified by Sivan Toledo.
 * 
 * This is a nearly trivial tacho-based motor controller. It will
 * rotate a motor indefinitely, or for a given period, or to aim
 * for a particular angle. It is driven by tacho change, so even
 * the time-based rotation only works for motors with tacho encoders.
 * Also, because the control is not time dependent, the angle mode
 * never stops, otherwise we will almost certainly overshoot. All
 * in all, this is very naive, much more naive than the controller in
 * the standard firmware. 
 */

/* Definitions of the state of each motor. */
static volatile struct {
  /* The mode this motor is currently in. */
  enum {
    MOTOR_STOP = 0,      /* No rotation. */
    MOTOR_CONFIGURING,   /* Reconfiguration in progress. */
    MOTOR_ON_CONTINUOUS, /* Rotate until stopped. */
    MOTOR_ON_ANGLE,      /* Rotate a given angle. */
    MOTOR_ON_TIME,       /* Rotate a given time. */
  } mode;

  /* If in one of the automatic modes (angle/time rotation), defines
   * whether the motor will brake (TRUE) or coast (FALSE) when
   * stopped.
   */
  bool brake;

  /* The target is a mode-dependent value.
   *  - In stop and continuous mode, it is not used.
   *  - In angle mode, holds the target tachymeter count.
   *  - In time mode, holds the remaining number of milliseconds to
   *    run.
   */
  int32_t target;
} _simplemotorcontrol_state[NXT_N_MOTORS] = {
  { MOTOR_STOP, FALSE, 0 },
  { MOTOR_STOP, FALSE, 0 },
  { MOTOR_STOP, FALSE, 0 },
};

static void simplemotorcontrolStop(uint32_t motor, bool brake) {
  ASSERT(motor < NXT_N_MOTORS);

  _simplemotorcontrol_state[motor].mode = MOTOR_STOP;
  motorSet(motor, 0, brake);
}

static void simplemotorcontrolTachoChangeHandler(void) {
  uint32_t i;
  uint32_t time = systick;
  int32_t  err, power;
  
  for (i=0; i<NXT_N_MOTORS; i++) {
    switch (_simplemotorcontrol_state[i].mode) {
      case MOTOR_ON_TIME:
        if (time >= _simplemotorcontrol_state[i].target) 
          simplemotorcontrolStop(i, _simplemotorcontrol_state[i].brake);
        break;
      case MOTOR_ON_ANGLE:
        err = _simplemotorcontrol_state[i].target - tachoGetCount(i);
        if (err == 0) { motorSet(i, 0, FALSE); break; }
        power = err; 
        if (power > 100) power = 100;
        if (power < -100) power = -100;
        if (power > 0 && power <  50) power =  50;
        if (power < 0 && power > -50) power = -50;
        motorSet(i, power, FALSE);
        break;
      default:
        break;
    }
  }
}  

static
void simplemotorcontrolRotate(uint32_t motor, int32_t speed) {
  ASSERT(motor < NXT_N_MOTORS);
  ASSERT(speed <= 100);
  ASSERT(speed >= -100);

  /* Continuous mode has no target or brake parameter, just set the
   * mode and fire up the motor.
   */
  _simplemotorcontrol_state[motor].mode = MOTOR_ON_CONTINUOUS;
  motorSet(motor, speed, FALSE);
}

static
void simplemotorcontrolRotateAngle(uint32_t motor, int32_t angle) {
  int32_t current = tachoGetCount(motor);
  
  ASSERT(motor < NXT_N_MOTORS);
  ASSERT(speed <= 100);
  ASSERT(speed >= -100);

  /* Set the motor to configuration mode. This way, if we are
   * overriding another intelligent mode, the tachymeter interrupt
   * handler will ignore the motor while we tweak its settings.
   */

  _simplemotorcontrol_state[motor].mode = MOTOR_CONFIGURING;

  /* Set the target tachymeter value based on the rotation
   * direction */
  _simplemotorcontrol_state[motor].target = current + angle;
  _simplemotorcontrol_state[motor].mode   = MOTOR_ON_ANGLE;

  motorSet(motor, angle > 0 ? 100 : -100, FALSE);
}

static
void simplemotorcontrolRotateTime(uint32_t motor, int32_t speed, uint32_t ms, bool brake) {
  ASSERT(motor < NXT_N_MOTORS);
  ASSERT(speed <= 100);
  ASSERT(speed >= -100);

  /* Set the motor to configuration mode. This way, if we are
   * overriding another intelligent mode, the tachymeter interrupt
   * handler will ignore the motor while we tweak its settings.
   */
  _simplemotorcontrol_state[motor].mode = MOTOR_CONFIGURING;

  _simplemotorcontrol_state[motor].target = systick + ms;
  _simplemotorcontrol_state[motor].brake  = brake;
  _simplemotorcontrol_state[motor].mode   = MOTOR_ON_TIME;
  motorSet(motor, speed, FALSE);
}

