/* Copyright (C) 2007 the NxOS developers
 *
 * See AUTHORS for a full list of the developers.
 *
 * Redistribution of this file is permitted under
 * the terms of the GNU Public License (GPL) version 2.
 * 
 * Modified by Sivan Toledo.
 */

/** @defgroup sensors Sensor interface
 *
 * The NXT sensor interface is mostly control mechanisms for analog
 * sensors (digital sensors are handled by the I2C driver). Sensors
 * have to be enabled to return readings.
 *
 * @note For all API calls, the sensor ports are zero-indexed,
 * ie. from 0 through 3, not 1 through 4.
 */
/*@{*/

/** Enumeration defining the types of sensor data pins for analog sensors. */
typedef enum {
  DIGI0 = 0,
  DIGI1,
} sensorports_pin_t;

/** Mapping of PIO ports to I2C busses.
 *
 * Each sensor port has two DIGI pins, whose use varies from sensor to
 * sensor. We remember which two pins each sensor has using the
 * sensor_pins structure.
 */

static const struct {
  uint32_t scl; /* DIGI0 - I2C clock. */
  uint32_t sda; /* DIGI1 - I2C data. */
} _sensorportsPinmap[NXT_N_SENSORS] = {
  { AT91C_PIO_PA23, AT91C_PIO_PA18 },
  { AT91C_PIO_PA28, AT91C_PIO_PA19 },
  { AT91C_PIO_PA29, AT91C_PIO_PA20 },
  { AT91C_PIO_PA30, AT91C_PIO_PA2  },
};

static enum {
  OFF = 0, /* Unused. */
  LEGACY,  /* Legacy RCX sensor support (Not currently supported). */
  ANALOG,  /* NXT sensor in analog mode. */
  DIGITAL, /* NXT sensor in digital (i2c) mode. */
} _sensorportsMode[NXT_N_SENSORS] = {
  /* All sensors start off. */
  OFF, OFF, OFF, OFF
};

static
void sensorportsInit(void) {
  uint32_t pinmask = 0;
  int i;

  for (i=0; i<NXT_N_SENSORS; i++) {
    pinmask |= _sensorportsPinmap[i].sda | _sensorportsPinmap[i].scl;
  }
  /* Disable output on all DIGI0/1 pins, which will set the lines high
   * due to the internal PIO pull-up resistor. We will keep the lines
   * in this idle state until a sensor driver tells us what to do with
   * the lines.
   */
  *AT91C_PIOA_PER = pinmask;
  *AT91C_PIOA_ODR = pinmask;
}

static
void sensorportsEnableDigital(uint32_t sensor) {
  uint32_t pinmask;

  ASSERT(sensor < NXT_N_SENSORS);
  ASSERT(_sensorportsMode[sensor] == OFF);

  _sensorportsMode[sensor] = DIGITAL;

  /* In digital mode, the DIGI outputs (SDA and SCL) are left up, and
   * enabled in multi-drive mode.
   */
  pinmask = _sensorportsPinmap[sensor].sda
          | _sensorportsPinmap[sensor].scl;

  *AT91C_PIOA_OER  = pinmask;
  *AT91C_PIOA_SODR = pinmask;
  *AT91C_PIOA_MDER = pinmask;
}

/*
const nx__sensors_pins *nx__sensors_get_pins(uint32_t sensor) {
  ASSERT(sensor < NXT_N_SENSORS);

  return &_sensorportsPinmap[sensor];
}
*/

static
void sensorportsDisable(uint32_t sensor) {
  ASSERT(sensor < NXT_N_SENSORS);
  //if (sensor >= NXT_N_SENSORS)
  //  return;

  switch (_sensorportsMode[sensor]) {
  case OFF:
  case LEGACY:
    break;
  case ANALOG:
  case DIGITAL:
    /* Disable output on the DIGI pins to return to the idle state. */
    *AT91C_PIOA_SODR = (_sensorportsPinmap[sensor].sda |
                        _sensorportsPinmap[sensor].scl);
    *AT91C_PIOA_ODR = (_sensorportsPinmap[sensor].sda |
                       _sensorportsPinmap[sensor].scl);
    break;
  }

  _sensorportsMode[sensor] = OFF;
}

static
void sensorportsEnableAnalog(uint32_t sensor) {
  uint32_t pinmask;
  uint32_t sodr, codr;

  ASSERT(sensor < NXT_N_SENSORS);
  ASSERT(_sensorportsMode[sensor] == OFF);

  _sensorportsMode[sensor] = ANALOG;

  pinmask = _sensorportsPinmap[sensor].sda
          | _sensorportsPinmap[sensor].scl;

  /* In analog mode, the DIGI outputs are driven low. */
  *AT91C_PIOA_OER  = pinmask;
  *AT91C_PIOA_CODR = pinmask;
}

static
void sensorportsDigiSet(uint32_t sensor, sensorports_pin_t pin) {
  ASSERT(sensor < NXT_N_SENSORS);
  ASSERT(_sensorportsMode[sensor] == ANALOG);

  *AT91C_PIOA_SODR = (pin == DIGI1 ? _sensorportsPinmap[sensor].sda 
                                   : _sensorportsPinmap[sensor].scl);
}

static
void sensorportsDigiClear(uint32_t sensor, sensorports_pin_t pin) {
  ASSERT(sensor < NXT_N_SENSORS);
  ASSERT(_sensorportsMode[sensor] == ANALOG);

  *AT91C_PIOA_CODR = (pin == DIGI1 ? _sensorportsPinmap[sensor].sda 
                                   : _sensorportsPinmap[sensor].scl);
}
