/*
 * nxtfs.h
 * 
 * Functions to access files on the NXT's standard file system.
 * 
 * Copyright Sivan Toledo, 2008.
 * 
 * Distributed under GPL 2.
 * 

/*
 * constrants and structures from the standard firmware.
 */

#define   FILENAME_LENGTH               19 /* zero termination not included */

#define   STARTOFFILETABLE              (0x13FF00L)
#define   STARTOFUSERFLASH              (0x121400L)//(0x11F000L)
#define   SIZEOFUSERFLASH               (STARTOFFILETABLE - STARTOFUSERFLASH)
#define   FLASHOFFSET                   (0x100000L)

#define   SIZEOFFLASH                   262144L
#define   SECTORSIZE                    256L
#define   SECTORSIZESHIFT               8
#define   NOOFSECTORS                   (SIZEOFFLASH/SECTORSIZE)
#define   HEADERFIXEDSIZE               (FILENAME_SIZE + 4 + 4 + 4 + 2 + 2)
#define   FILENAME_SIZE                 (FILENAME_LENGTH + 1)

#define   FULLNAME                      1
#define   NAME                          2
#define   EXTENTION                     3
#define   WILDCARD                      4

/* MAX_FILES is a weird constant. why would it be related ot sector size? */
#define   MAX_FILES                     ((SECTORSIZE/4) - 1)  /* Last file entry is used for file version*/

/* Constants related to filetype */
enum
{
  SYSTEMFILE = 0x01,
  DATAFILE   = 0x02,
  LINEAR     = 0x04,
  NONLINEAR  = 0x08
};

/* 
 * The last field is really just a pointer to the first
 * sector number. It is not an array. 
 * 
 * Let's hope that GCC packs this like IAR... (it seems to).
 */
typedef struct {
  uint8_t    FileName[FILENAME_SIZE];
  uint32_t   FileStartAdr;
  uint32_t   FileSize;
  uint32_t   DataSize;
  uint16_t   CheckSum;
  uint16_t   FileType;
  uint16_t   FileSectorTable[SIZEOFUSERFLASH/SECTORSIZE];
} FILEHEADER;

static FILEHEADER** Files = (FILEHEADER**) STARTOFFILETABLE;

/*
 * That's it for things copied from the standard firmware.
 * From here on, it's my code and structures.
 */

#define NXTFS_LIST_START         -1 /* must be -1 */
#define NXTFS_LIST_NO_MORE_FILES -2
#define NXTFS_FILE_NOT_FOUND     -3

typedef struct {
  int32_t   index; /* file index                        */
  uint32_t  pos;   /* our relative position in the file */
  uint8_t*  ptr;   /* absolute address of file[pos]     */
  uint16_t* nextsector;
} nxtfs_handle_t;

/*
 * Takes a state (iterator) and returns the state for the next file
 * to be enumerated.
 * Clients start the enumeration with NXTFS_LIST_START (that is, -1).
 * The function returns NXTFS_LIST_NO_MORE_FILES if it did not
 * find another file to list.
 * If the returned state is no NXTFS_LIST_NO_MORE_FILES, the
 * file name is copied to the first argument. 
 */ 
static int32_t nxtfsIteratorNext(int32_t index) {
  int j;
  for (index++; index<MAX_FILES; index++) {
    uint32_t header_ptr = (uint32_t) Files[index];
    if (header_ptr == 0 || header_ptr == 0xFFFFFFFF) continue;
    return index;
  }
  return NXTFS_LIST_NO_MORE_FILES;
}

static int32_t nxtfsFindFile(uint8_t* filename) {
  int32_t index;
  for (index=0; index<MAX_FILES; index++) {
    uint32_t header_ptr = (uint32_t) Files[index];
    if (header_ptr == 0 || header_ptr == 0xFFFFFFFF) continue;
    if (strcmp( Files[index]->FileName, filename ) == 0)
      return index;
  }
  return NXTFS_FILE_NOT_FOUND;
}

static inline uint8_t* nxtfsGetName(int32_t index) {
  return Files[index]->FileName;
}

static inline uint32_t nxtfsGetFileSize(int32_t index) {
  return Files[index]->FileSize;
}

static inline uint32_t nxtfsGetDataSize(int32_t index) {
  return Files[index]->DataSize;
}

static inline bool nxtfsIsLinear(int32_t index) {
  return ((Files[index]->FileType & LINEAR) != 0);
}

static inline void* nxtfsGetLinearAddress(int32_t index) {
  ASSERT( nxtfsIsLinear(index) );
  
  return (void*) Files[index]->FileStartAdr;
}

/*
 * This function fills in the handle data structure,
 * which the client code must allocate.
 */
static void nxtfsOpenRead(int32_t index, nxtfs_handle_t* h) {
  h->index = index;
  h->pos   = 0;
  h->ptr   = (uint8_t*) Files[index]->FileStartAdr;
  h->nextsector = Files[index]->FileSectorTable;
}

static void _nxtfsNextByte(nxtfs_handle_t* h) {
  (h->ptr)++;
  (h->pos)++;
  
  /* did we move to a new sector? */
  if (!((uint32_t)(h->ptr) & (SECTORSIZE-1))) {

    /* yes, we need to go to the first byte in the next sector */

    /*
     * The last sector pointer in every sector is not
     * to a sector of file data, but to a sector of pointers
     * to file data. We start by checking for this condition.
     */
    if (!((uint32_t)(h->nextsector + 1) & (SECTORSIZE-1)))
      h->nextsector = (uint16_t*)(((uint32_t)(*(h->nextsector)) << SECTORSIZESHIFT) | FLASHOFFSET);

    /* If pointing at an illegal adr then set it to NULL */
    //if (SIZEOFFLASH < (ULONG)((ULONG)(HandleTable[Handle].pSectorNo) & ~FLASHOFFSET))
    //  pAdr = NULL;
    //else
    h->ptr = (uint8_t*)(((uint32_t)(*(h->nextsector)) << SECTORSIZESHIFT) | FLASHOFFSET);

    (h->nextsector)++;
  }
}

static void nxtfsRead(nxtfs_handle_t* h, void* buffer, uint32_t len) {
  uint8_t* p = buffer;
  while (len && h->pos < Files[ h->index ]->DataSize) {
    *p = *( h->ptr );
    p++;
    _nxtfsNextByte(h);
    len--;
  }
}

/*** Demos ***/

static void nxtfsDemoListFiles() {
  int32_t i = NXTFS_LIST_START; 
  while (1) {
    i = nxtfsIteratorNext(i);
    if (i == NXTFS_LIST_NO_MORE_FILES) break;

    displayClear();

    displayString(nxtfsGetName(i));
    displayString("\n");

    displayString("fs ");
    displayUInt(nxtfsGetFileSize(i));
    displayString("\n");

    displayString("ds ");
    displayUInt(nxtfsGetDataSize(i));
    displayString("\n");
      
    displayString("linear? ");
    displayUInt((uint32_t) nxtfsIsLinear(i));
    displayString("\n");

    busywait(4*1000000);
  }  
}

static void nxtfsDemoDisplayTextFile(uint8_t* filename) {
  int32_t i; 
  nxtfs_handle_t h;

  displayClear();
  
  i = nxtfsFindFile(filename);
  if (i == NXTFS_FILE_NOT_FOUND) {
    displayString("File <");
    displayString(filename);
    displayString("> not found\n");

    busywait(4*1000000);
    return;
  }
  
  nxtfsOpenRead(i,&h);
        
  displayString("pos ");
  displayUInt(h.pos);
  displayString("\n");
            
  displayString("ptr ");
  displayHex((uint32_t) h.ptr);
  displayString("\n");

  while (h.pos < nxtfsGetDataSize(i)) {
    uint8_t c[2];
    c[0] = '?';
    c[1] = 0;
    nxtfsRead(&h,c,1);
    if (c[0] == '\n') { 
      busywait(1*1000000); 
      displayClear(); 

      displayString("pos ");
      displayUInt(h.pos);
      displayString("\n");
            
      displayString("ptr ");
      displayHex((uint32_t) h.ptr);
      displayString("\n");
             
    } else displayString(c);
  }  
}

static void nxtfsDemo(void) {
  nxtfsDemoListFiles();
  busywait(1*1000000);
  nxtfsDemoDisplayTextFile("foo.bar");
  busywait(1*1000000);
  nxtfsDemoDisplayTextFile("data.txt");
  busywait(1*1000000);
}