/*
 * Run DSRs
 */

typedef struct dsr_st {
  void (*function)(void);
  struct dsr_st* next;
  struct dsr_st* prev;
  bool   in_use;
} dsr_t;

static dsr_t _dsrFirst;
static dsr_t _dsrLast;

static
void __attribute__ ((interrupt("IRQ"))) _dsr_isr(void) {
  dsr_t* dsr;
  interrupt_state_t intstate;
  
  while ((dsr = _dsrFirst.next) != &_dsrLast) {
    dsr->next->prev = &_dsrFirst;
    _dsrFirst.next = dsr->next;
    dsr->in_use = FALSE;
  
    /* 
     * before enabling interrupts, we must
     * push spsr on the stack. We use lr
     * as a temporary, since it was already
     * pushed in the beginning of the isr.
     */
    asm ("mrs  lr,spsr;"    
         "stmfd sp!,{lr}");
         
    //interruptsEnableSaveToArg(intstate);
    intstate = interruptsSaveAndEnable();
    
    (*(dsr->function))();
  
    interruptsRestore(intstate);
    
    asm ("ldmfd sp!,{lr};"
         "msr   spsr_cxsf,lr");
  }
  
  aicUpdatePriority(); /* tell AIC we are done */
}

static void dsrInit(void) {
  _dsrFirst.next = &_dsrLast;
  _dsrLast.prev  = &_dsrFirst;
  
  aicInstallIsr(AT91C_ID_IRQ1, AIC_PRIO_LOWEST, AIC_TRIG_EDGE, _dsr_isr);
} 

static void dsrPost( dsr_t* dsr ) {
  if ( dsr->in_use ) return; /* do not post a posted dsr; */
  dsr->in_use = TRUE;
  
  {
    interrupt_state_t intstate = interruptsSaveAndDisable();
  
    dsr_t* before_last = _dsrLast.prev;
    dsr->next = &_dsrLast;
    dsr->prev = before_last;
    before_last->next = dsr;
    _dsrLast.prev      = dsr;
  
    aicSet(AT91C_ID_IRQ1);
  
    interruptsRestore(intstate);
  }
}

static void dsrPostFromIsr( dsr_t* dsr ) {
  if ( dsr->in_use ) return; /* do not post a posted dsr; */
  dsr->in_use = TRUE;

  dsr_t* before_last = _dsrLast.prev;
  dsr->next = &_dsrLast;
  dsr->prev = before_last;
  before_last->next = dsr;
  _dsrLast.prev      = dsr;

  aicSet(AT91C_ID_IRQ1);  
}
