/* Copyright (C) 2007 the NxOS developers
 *
 * See AUTHORS for a full list of the developers.
 *
 * Redistribution of this file is permitted under
 * the terms of the GNU Public License (GPL) version 2.
 * 
 * Heavily modified by Sivan Toledo.
 * The NxOS driver used while loops inside ISRs, to
 * wait for the end of an SPI bus transfer, to signal
 * to the LCD controller whether the next transfer
 * is commands or data. This is not a good idea.
 * 
 * I changed the structure to resemble that of the 
 * driver in the standard firmware, in which a bit
 * of work is done in every systick interrupt: sending
 * the new line's address, or sending the new line bitmap.
 * The standard firmware seems to also send a complete
 * initialization string to the LCD at every screen refresh.
 * I do not know why that is necessary so I left it out.
 */
 
 /** Width of the LCD display, in pixels. */
#define LCD_WIDTH 100
/** Height of the LCD display, in bytes. */
#define LCD_HEIGHT 8 /* == 64 pixels. */
 
/* The display buffer, which is mirrored to the LCD controller's RAM. */
static uint8_t framebuffer[LCD_HEIGHT][LCD_WIDTH];

/******************************************************************************/
/** @name Font data
 *
 * A basic font is embedded into the baseplate, for use by the display
 * driver.
 *
 * @note This data is automatically generated from a "font grid" image
 * at compile time. This makes it easy to tweak the looks of the font by
 * editing an image, and have it nicely embedded in the Baseplate.
 */
/*@{*/

/** The ASCII offset of the first character in the font table.
 *
 * All characters before that one are unprintable.
 */
#define FONT_START 0x20

/** The ASCII offset of the last character in the font table.
 *
 * All characters after that one are unprintable.
 */
#define FONT_END 0x80

/** The width of a font character in pixels. */
#define FONT_WIDTH 5
/** The width of a font cell, which includes a 1-pixel spacer. */
#define CELL_WIDTH (FONT_WIDTH + 1)
/** The number of font cells per line. */
#define DISPLAY_WIDTH_CELLS (LCD_WIDTH / CELL_WIDTH)
/** The number of fond lines on the display. */
#define DISPLAY_HEIGHT_CELLS LCD_HEIGHT

/** The font character data.
 *
 * Each entry in the font array is a subarray of length @b
 * FONT_WIDTH. Each element in that array gives the bitmask for one
 * vertical slice of the character in question.
 */
static const uint8_t font_data[][FONT_WIDTH] = {
  { 0x00, 0x00, 0x00, 0x00, 0x00 },
  { 0x00, 0x00, 0x5F, 0x00, 0x00 },
  { 0x00, 0x07, 0x00, 0x07, 0x00 },
  { 0x14, 0x3E, 0x14, 0x3E, 0x14 },
  { 0x55, 0xAA, 0x55, 0xAA, 0x55 },
  { 0x26, 0x16, 0x08, 0x34, 0x32 },
  { 0xAA, 0x55, 0xAA, 0x55, 0xAA },
  { 0x00, 0x00, 0x07, 0x00, 0x00 },
  { 0x00, 0x1C, 0x22, 0x41, 0x00 },
  { 0x00, 0x41, 0x22, 0x1C, 0x00 },
  { 0x2A, 0x1C, 0x7F, 0x1C, 0x2A },
  { 0x08, 0x08, 0x3E, 0x08, 0x08 },
  { 0x00, 0x50, 0x30, 0x00, 0x00 },
  { 0x08, 0x08, 0x08, 0x08, 0x08 },
  { 0x00, 0x60, 0x60, 0x00, 0x00 },
  { 0x20, 0x10, 0x08, 0x04, 0x02 },
  { 0x3E, 0x51, 0x49, 0x45, 0x3E },
  { 0x00, 0x42, 0x7F, 0x40, 0x00 },
  { 0x42, 0x61, 0x51, 0x49, 0x46 },
  { 0x21, 0x41, 0x45, 0x4B, 0x31 },
  { 0x18, 0x14, 0x12, 0x7F, 0x10 },
  { 0x27, 0x45, 0x45, 0x45, 0x39 },
  { 0x3C, 0x4A, 0x49, 0x49, 0x30 },
  { 0x01, 0x01, 0x79, 0x05, 0x03 },
  { 0x36, 0x49, 0x49, 0x49, 0x36 },
  { 0x06, 0x49, 0x49, 0x29, 0x1E },
  { 0x00, 0x36, 0x36, 0x00, 0x00 },
  { 0x00, 0x56, 0x36, 0x00, 0x00 },
  { 0x08, 0x14, 0x22, 0x41, 0x00 },
  { 0x14, 0x14, 0x14, 0x14, 0x14 },
  { 0x41, 0x22, 0x14, 0x08, 0x00 },
  { 0x02, 0x01, 0x59, 0x05, 0x02 },
  { 0x3E, 0x41, 0x49, 0x55, 0x1E },
  { 0x7E, 0x09, 0x09, 0x09, 0x7E },
  { 0x7F, 0x49, 0x49, 0x49, 0x3E },
  { 0x3E, 0x41, 0x41, 0x41, 0x22 },
  { 0x7F, 0x41, 0x41, 0x22, 0x1C },
  { 0x7F, 0x49, 0x49, 0x49, 0x41 },
  { 0x7F, 0x09, 0x09, 0x09, 0x01 },
  { 0x3E, 0x41, 0x41, 0x49, 0x3A },
  { 0x7F, 0x08, 0x08, 0x08, 0x7F },
  { 0x00, 0x41, 0x7F, 0x41, 0x00 },
  { 0x20, 0x40, 0x41, 0x3F, 0x01 },
  { 0x7F, 0x08, 0x14, 0x22, 0x41 },
  { 0x7F, 0x40, 0x40, 0x40, 0x40 },
  { 0x7F, 0x02, 0x04, 0x02, 0x7F },
  { 0x7F, 0x04, 0x08, 0x10, 0x7F },
  { 0x3E, 0x41, 0x41, 0x41, 0x3E },
  { 0x7F, 0x09, 0x09, 0x09, 0x06 },
  { 0x3E, 0x41, 0x51, 0x21, 0x5E },
  { 0x7F, 0x09, 0x19, 0x29, 0x46 },
  { 0x26, 0x49, 0x49, 0x49, 0x32 },
  { 0x01, 0x01, 0x7F, 0x01, 0x01 },
  { 0x3F, 0x40, 0x40, 0x40, 0x3F },
  { 0x1F, 0x20, 0x40, 0x20, 0x1F },
  { 0x7F, 0x20, 0x18, 0x20, 0x7F },
  { 0x63, 0x14, 0x08, 0x14, 0x63 },
  { 0x03, 0x04, 0x78, 0x04, 0x03 },
  { 0x61, 0x51, 0x49, 0x45, 0x43 },
  { 0x00, 0x7F, 0x41, 0x41, 0x00 },
  { 0x02, 0x04, 0x08, 0x10, 0x20 },
  { 0x00, 0x41, 0x41, 0x7F, 0x00 },
  { 0x04, 0x02, 0x01, 0x02, 0x04 },
  { 0x40, 0x40, 0x40, 0x40, 0x40 },
  { 0x00, 0x01, 0x02, 0x04, 0x00 },
  { 0x20, 0x54, 0x54, 0x54, 0x78 },
  { 0x7F, 0x48, 0x44, 0x44, 0x38 },
  { 0x30, 0x48, 0x48, 0x48, 0x20 },
  { 0x38, 0x44, 0x44, 0x48, 0x7F },
  { 0x38, 0x54, 0x54, 0x54, 0x18 },
  { 0x08, 0x7E, 0x09, 0x09, 0x02 },
  { 0x0C, 0x52, 0x52, 0x52, 0x3E },
  { 0x7F, 0x08, 0x04, 0x04, 0x78 },
  { 0x00, 0x44, 0x7D, 0x40, 0x00 },
  { 0x20, 0x40, 0x40, 0x3D, 0x00 },
  { 0x7F, 0x10, 0x28, 0x44, 0x00 },
  { 0x00, 0x41, 0x7F, 0x40, 0x00 },
  { 0x7C, 0x04, 0x18, 0x04, 0x78 },
  { 0x7C, 0x08, 0x04, 0x04, 0x78 },
  { 0x38, 0x44, 0x44, 0x44, 0x38 },
  { 0xFC, 0x14, 0x14, 0x14, 0x08 },
  { 0x08, 0x14, 0x14, 0x18, 0x7C },
  { 0x7C, 0x08, 0x04, 0x04, 0x08 },
  { 0x48, 0x54, 0x54, 0x54, 0x20 },
  { 0x04, 0x3F, 0x44, 0x40, 0x20 },
  { 0x3C, 0x40, 0x40, 0x20, 0x7C },
  { 0x1C, 0x20, 0x40, 0x20, 0x1C },
  { 0x3C, 0x40, 0x38, 0x40, 0x3C },
  { 0x44, 0x28, 0x10, 0x28, 0x44 },
  { 0x0C, 0x50, 0x50, 0x50, 0x3C },
  { 0x44, 0x64, 0x54, 0x4C, 0x44 },
  { 0x00, 0x08, 0x36, 0x41, 0x00 },
  { 0x00, 0x00, 0x7F, 0x00, 0x00 },
  { 0x00, 0x41, 0x36, 0x08, 0x00 },
  { 0x00, 0x07, 0x00, 0x07, 0x00 },
  { 0x08, 0x1C, 0x2A, 0x08, 0x08 },
};
/******************************************************************************/

/* 
 * Internal command bytes implementing part of the basic commandset of
 * the UC1601.
 */
#define SET_COLUMN_ADDR0(addr) (0x00 | (addr & 0xF))
#define SET_COLUMN_ADDR1(addr) (0x10 | ((addr >> 4) & 0xF))
#define SET_MULTIPLEX_RATE(rate) (0x20 | (rate & 3))
#define SET_SCROLL_LINE(sl) (0x40 | (sl & 0x3F))
#define SET_PAGE_ADDR(page) (0xB0 | (page & 0xF))
#define SET_BIAS_POT0() (0x81)
#define SET_BIAS_POT1(pot) (pot & 0xFF)
#define SET_RAM_ADDR_CONTROL(auto_wrap, page_first, neg_inc, write_only_inc) \
           (0x88 | (auto_wrap << 0) | (page_first << 1) |                    \
            (neg_inc << 2) | (write_only_inc << 3))
#define SET_ALL_PIXELS_ON(on) (0xA4 | (on & 1))
#define SET_INVERSE_DISPLAY(on) (0xA6 | (on & 1))
#define ENABLE(on) (0xAE | (on & 1))
#define SET_MAP_CONTROL(mx, my) (0xC0 | (mx << 1) | (my << 2))
#define RESET() (0xE2)
#define SET_BIAS_RATIO(bias) (0xE8 | (bias & 3))

/*
 * SPI controller driver.
 */
typedef enum {
  LCD_COMMAND,
  LCD_DATA
} spi_mode;

/*
 * Set the data transmission mode.
 */
static void _lcdSetTransmitMode(spi_mode mode) {
  while(!(*AT91C_SPI_SR & AT91C_SPI_TXEMPTY));

  if (mode == LCD_COMMAND) *AT91C_PIOA_CODR = AT91C_PA12_MISO;
  else                     *AT91C_PIOA_SODR = AT91C_PA12_MISO;
}

/*
 * Send a command byte to the LCD controller.
 */
static void _lcdSendCommandByte(uint8_t command) {
  _lcdSetTransmitMode(LCD_COMMAND);

  /* Wait for the transmit register to empty. */
  while (!(*AT91C_SPI_SR & AT91C_SPI_TDRE));

  /* Send the command byte and wait for a reply. */
  *AT91C_SPI_TDR = command;
}

static volatile uint8_t lcd_initialized;
static volatile uint8_t lcd_state;
static volatile uint8_t lcd_line;
static volatile uint8_t lcd_cmd[3] = { SET_COLUMN_ADDR0(0), SET_COLUMN_ADDR1(0), 0 };
  
static void lcdPeriodicTask(void) {
  if (!lcd_initialized) return;
  if (!(*AT91C_SPI_SR & AT91C_SPI_ENDTX))   return;
  if (!(*AT91C_SPI_SR & AT91C_SPI_TDRE))    return;
  if (!(*AT91C_SPI_SR & AT91C_SPI_TXEMPTY)) return;
  switch (lcd_state) {
    case 0:
      *AT91C_PIOA_CODR = AT91C_PA12_MISO; // command
      lcd_cmd[2] = SET_PAGE_ADDR(lcd_line);
      *AT91C_SPI_TPR = (uint32_t) lcd_cmd;
      *AT91C_SPI_TCR = 3;
      *AT91C_SPI_PTCR = AT91C_PDC_TXTEN;
      lcd_state++;
      break; 
    case 1:
      *AT91C_PIOA_SODR = AT91C_PA12_MISO; // data
      *AT91C_SPI_TPR = (uint32_t) &( framebuffer[lcd_line][0] );
      *AT91C_SPI_TCR = 100;
      *AT91C_SPI_PTCR = AT91C_PDC_TXTEN;
      lcd_state = 0;
      lcd_line = (lcd_line+1) % LCD_HEIGHT;
      break;
  }
}

/* Initialize the LCD controller. */
static 
void lcdInit(void) {
  uint32_t i;
  /* This is the command byte sequence that should be sent to the LCD
   * after a reset.
   */
  const uint8_t lcd_init_sequence[] = {
    /* LCD power configuration.
     *
     * The LEGO Hardware Developer Kit documentation specifies that the
     * display should be configured with a multiplex rate (MR) of 1/65,
     * and a bias ratio (BR) of 1/9, and a display voltage V(LCD) of 9V.
     *
     * The specified MR and BR both map to simple command writes. V(LCD)
     * however is determined by an equation that takes into account both
     * the BR and the values of the PM (Potentiometer) and TC
     * (Temperature Compensation) configuration parameters.
     *
     * The equation and calculations required are a little too complex
     * to inline here, but the net result is that we should set a PM
     * value of 92. This will result in a smooth voltage gradient, from
     * 9.01V at -20 degrees Celsius to 8.66V at 60 degrees Celsius
     * (close to the maximum operational range of the LCD display).
     */
    SET_MULTIPLEX_RATE(3),
    SET_BIAS_RATIO(3),
    SET_BIAS_POT0(),
    SET_BIAS_POT1(92),

    /* Set the RAM address control, which defines how the data we send
     * to the LCD controller are placed in its internal video RAM.
     *
     * We want the bytes we send to be written in row-major order (line
     * by line), with no automatic wrapping.
     */
    SET_RAM_ADDR_CONTROL(1, 0, 0, 0),

    /* Set the LCD mapping mode, which defines how the data in video
     * RAM is driven to the display. The display on the NXT is mounted
     * upside down, so we want just Y mirroring.
     */
    SET_MAP_CONTROL(0, 1),

    /* Set the initial position of the video memory cursor. We
     * initialize it to point to the start of the screen.
     */
    SET_COLUMN_ADDR0(0),
    SET_COLUMN_ADDR1(0),
    SET_PAGE_ADDR(0),

    /* Turn the display on. */
    ENABLE(1),
  };

  /* Enable power to the SPI and PIO controllers. */
  *AT91C_PMC_PCER = (1 << AT91C_ID_SPI) | (1 << AT91C_ID_PIOA);

  /* Configure the PIO controller: Hand the MOSI (Master Out, Slave
   * In) and SPI clock pins over to the SPI controller, but keep MISO
   * (Master In, Slave Out) and PA10 (Chip Select in this case) and
   * configure them for manually driven output.
   *
   * The initial configuration is command mode (sending LCD commands)
   * and the LCD controller chip not selected.
   */
  *AT91C_PIOA_PDR = AT91C_PA13_MOSI | AT91C_PA14_SPCK;
  *AT91C_PIOA_ASR = AT91C_PA13_MOSI | AT91C_PA14_SPCK;

  *AT91C_PIOA_PER = AT91C_PA12_MISO | AT91C_PA10_NPCS2;
  *AT91C_PIOA_OER = AT91C_PA12_MISO | AT91C_PA10_NPCS2;
  *AT91C_PIOA_CODR = AT91C_PA12_MISO;
  *AT91C_PIOA_SODR = AT91C_PA10_NPCS2;

  /* Disable all SPI interrupts, then configure the SPI controller in
   * master mode, with the chip select locked to chip 0 (UC1601 LCD
   * controller), communication at 2MHz, 8 bits per transfer and an
   * inactive-high clock signal.
   */
  *AT91C_SPI_CR = AT91C_SPI_SWRST;
  *AT91C_SPI_CR = AT91C_SPI_SPIEN;
  *AT91C_SPI_IDR = ~0;
  *AT91C_SPI_MR = (6 << 24) | AT91C_SPI_MSTR;
  AT91C_SPI_CSR[0] = ((0x18 << 24) | (0x18 << 16) | (0x18 << 8) |
                      AT91C_SPI_BITS_8 | AT91C_SPI_CPOL);

  /* enable DMA transfers */
  *AT91C_SPI_PTCR = AT91C_PDC_TXTEN;


  /* Now that the SPI bus is initialized, pull the Chip Select line
   * low, to select the uc1601. For some reason, letting the SPI
   * controller do this fails. Therefore, we force it once now.
   */
  *AT91C_PIOA_CODR = AT91C_PA10_NPCS2;


  /* Initialize the SPI controller to enable communication, then wait
   * a little bit for the UC1601 to register the new SPI bus state.
   */
  //spi_init();
  busywait(20000);

  /* Issue a reset command, and wait. Normally here we'd check the
   * UC1601 status register, but as noted at the start of the file, we
   * can't read from the LCD controller due to the board setup.
   */
  _lcdSendCommandByte(RESET());
  busywait(20000);

  for (i=0; i<sizeof(lcd_init_sequence); i++)
    _lcdSendCommandByte(lcd_init_sequence[i]);
    
  { 
    interrupt_state_t s = interruptsSaveAndDisable();
    lcd_initialized = 1;
    interruptsRestore(s);
  }
}

static
void lcdShutdown(void) {
  /* When power to the controller goes out, there is the risk that
   * some capacitors mounted around the controller might damage it
   * when discharging in an uncontrolled fashion. To avoid this, the
   * spec recommends putting the controller into reset mode before
   * shutdown, which activates a drain circuit to empty the board
   * capacitors gracefully.
   */
  *AT91C_SPI_IDR = ~0;
  *AT91C_SPI_PTCR = AT91C_PDC_TXTDIS;
  _lcdSendCommandByte(RESET());
  busywait(20000);
}

/*
 * Just for debug/demo purposes now.
 * If we send 132 bytes for every row, the LCD
 * controller wraps around automatically, so
 * we don't need to reposition the address. If
 * we only write 100 bytes, we need to set the
 * address at every line.
 */
static
void lcdSynchronousRefresh() {
  int i, j;

  _lcdSetTransmitMode(LCD_COMMAND);
  _lcdSendCommandByte(SET_COLUMN_ADDR0(0));
  _lcdSendCommandByte(SET_COLUMN_ADDR1(0));
  _lcdSendCommandByte(SET_PAGE_ADDR(0));
  _lcdSetTransmitMode(LCD_DATA);
  /* Start the data transfer. */
  for (i=0; i<8; i++) {
    //spi_set_tx_mode(COMMAND);
    //spi_write_command_byte(SET_COLUMN_ADDR0(0));
    //spi_write_command_byte(SET_COLUMN_ADDR1(0));
    //spi_write_command_byte(SET_PAGE_ADDR(i));
    //spi_set_tx_mode(DATA);

    for (j=0; j<100; j++) {
      /* Wait for the transmit register to empty. */
      while (!(*AT91C_SPI_SR & AT91C_SPI_TDRE));

      /* Send the command byte and wait for a reply. */
      *AT91C_SPI_TDR = framebuffer[i][j];
    }
    for (j=0; j<32; j++) {
      while (!(*AT91C_SPI_SR & AT91C_SPI_TDRE));
      *AT91C_SPI_TDR = 0;
    }
  }
}
/***************************************************************/


static struct {
  uint8_t x;
  uint8_t y;
  bool ignore_lf; /* If the display just wrapped from the right side
                     of the screen, ignore an LF immediately
                     after. */
} cursor;

static
void displayCursorSetPosition(uint8_t x, uint8_t y) {
  ASSERT(is_on_screen(x, y));
  cursor.x = x;
  cursor.y = y;
}

static
void displayClear(void) {
  memset(framebuffer, 0, sizeof(framebuffer));
  displayCursorSetPosition(0, 0);
}


/*
 * Text display functions.
 */
//static inline bool is_on_screen(uint8_t x, uint8_t y) {
//  if (x < DISPLAY_WIDTH_CELLS &&
//      y < DISPLAY_HEIGHT_CELLS)
//    return TRUE;
//  else
//    return FALSE;
//}

static inline const uint8_t *_lcdCharToFont(const char c) {
  if (c >= FONT_START)
    return font_data[c - FONT_START];
  else
    return font_data[0]; /* Unprintable characters become spaces. */
}

static void _displayUpdateCursor(bool inc_y) {
  if (!inc_y) {
    cursor.x++;

    if (cursor.x >= LCD_WIDTH) {
      cursor.x = 0;
      cursor.y++;
      cursor.ignore_lf = TRUE;
    } else {
      cursor.ignore_lf = FALSE;
    }
  } else if (cursor.ignore_lf) {
    cursor.ignore_lf = FALSE;
  } else {
    cursor.x = 0;
    cursor.y++;
  }

  if (cursor.y >= LCD_HEIGHT)
    cursor.y = 0;
}

//inline void nx_display_end_line(void) {
//  update_cursor(TRUE);
//}

static
void displayString(const char *str) {
  while (*str != '\0') {
    if (*str == '\n') {
      _displayUpdateCursor(TRUE);
   } else {
      int x_offset = cursor.x * CELL_WIDTH;
      memcpy(&framebuffer[cursor.y][x_offset],
             _lcdCharToFont(*str), 
             FONT_WIDTH);
      _displayUpdateCursor(FALSE);
    }
    str++;
  }
}

static
void displayHex(uint32_t val) {
  const char hex[16] = "0123456789ABCDEF";
  char buf[9];
  char *ptr = &buf[8];

  if (val == 0) {
    ptr--;
    *ptr = hex[0];
  } else {
    while (val != 0) {
      ptr--;
      *ptr = hex[val & 0xF];
      val >>= 4;
    }
  }

  buf[8] = '\0';

  displayString(ptr);
}

static
void displayUInt(uint32_t val) {
  char buf[11];
  char *ptr = &buf[10];

  if (val == 0) {
    ptr--;
    *ptr = '0';
  } else {

    while (val > 0) {
      ptr--;
      *ptr = val % 10 + '0';
      val /= 10;
    }

  }

  buf[10] = '\0';

  displayString(ptr);
}

static
void displayInt(int32_t val) {
  char buf[11];
  char *ptr = &buf[10];

  if (val == 0) {
    ptr--;
    *ptr = '0';
  } else {
    bool negative = (val < 0);
    if (negative) val = -val;

    while (val > 0) {
      ptr--;
      *ptr = val % 10 + '0';
      val /= 10;
    }
    
    if (negative) {
      ptr--;
      *ptr = '-';
    }
  }

  buf[10] = '\0';

  displayString(ptr);
}

/***********************************************************/
