/* Copyright (C) 2007 the NxOS developers
 *
 * See AUTHORS for a full list of the developers.
 *
 * Redistribution of this file is permitted under
 * the terms of the GNU Public License (GPL) version 2.
 * 
 * Modified by Sivan Toledo.
 * 
 * THIS IS STILL UNDER CONSTRUCTION. DO NOT USE!
 */
 
/*
 * We begin with the UART driver, which is private; only
 * the Bluetooth core uses it.
 * 
 * (But I think that the other UART is connected to the RS485 chip.)
 */

#ifndef uart_read_event
#warning "uart_read_event(ptr,len) not defined"
#define uart_read_event(ptr,len) 
#endif

#define UART_BUFSIZE 128 /* Buffer size for UART messages. */

/** Prototype for the UART read callback.
 *
 * The UART driver fires the callback with a @a buffer of length @a
 * packet_size of data to process.
 */
typedef void (*nx__uart_read_callback_t)(uint8_t *buffer, uint32_t packet_size);

/* Pinmask for all the UART pins. */
#define UART_PIOA_PINS \
   (AT91C_PA21_RXD1 |  \
    AT91C_PA22_TXD1 |  \
    AT91C_PA25_CTS1 |  \
    AT91C_PA24_RTS1 |  \
    AT91C_PA23_SCK1)

/* UART baud rate: 460.8kBaud */
#define UART_BAUD_RATE 460800

/* Value of the UART Clock Divisor to get as close as possible to that
 * value. This divisor actually programs for 461.5kBaud, for 0.15%
 * error.
 */
#define UART_CLOCK_DIVISOR (MCK_FREQ / 8 / UART_BAUD_RATE)

static volatile struct {
  nx__uart_read_callback_t callback;

  uint32_t packet_size;
  uint8_t buf[UART_BUFSIZE];

  /* for manual reading from the bluetooth driver : */
  uint32_t to_read;

} uart_state = {
  NULL, 0, {0}, 0
};

static void __attribute__ ((interrupt("IRQ"))) _uart_isr(void) {
  uint32_t status = *AT91C_US1_CSR;

  /* If we receive a break condition from the Bluecore, send up a NULL
   * packet and reset the controller status.
   */
  if (status & AT91C_US_RXBRK) {
    uart_state.callback(NULL, 0); /* do we really need this? */
    *AT91C_US1_CR = AT91C_US_RSTSTA;
  }

  if (status & AT91C_US_RXRDY) {

    /* we've just read the first byte:
     * it's the packet size */
    /* now we will use the PDC to read the whole packet */

    *AT91C_US1_IDR = AT91C_US_RXRDY;
    while(*AT91C_US1_IMR & AT91C_US_RXRDY);

    uart_state.packet_size = *AT91C_US1_RHR & 0xFF;
    *AT91C_US1_RCR = uart_state.packet_size;

    /* we reenable the receiving with the PDC */

    *AT91C_US1_IER = AT91C_US_ENDRX;
    *AT91C_US1_PTCR = AT91C_PDC_RXTEN;
  }

  if (status & AT91C_US_ENDRX) {
    *AT91C_US1_PTCR = AT91C_PDC_RXTDIS;

    uart_state.callback((uint8_t*)&(uart_state.buf), uart_state.packet_size);
    //uart_read_event((uint8_t*)&(uart_state.buf), uart_state.packet_size);

    /* We must put a size != 0 in the RCR register (even if the PDC is disabled for the receiving) */
    /* else when we try to read manually a value on US1_RHR thanks to the RXRDY interruption
     * the RXRDY of the CSR seems to never be set to 1 (no value read on the UART ?) */
    /* TODO : figure this out */
    *AT91C_US1_RPR = (uint32_t)(&uart_state.buf);
    *AT91C_US1_RCR = UART_BUFSIZE; /* default size */

    /* we've read a packet, so now we will do a manual reading
     * to have the next packet size and adapt the PDC RCR register value */
    *AT91C_US1_IER = AT91C_US_RXRDY;
  }

  aicUpdatePriority(); /* tell AIC we are done */
}

//void _uartInit(nx__uart_read_callback_t callback) {
static void _uartInit(nx__uart_read_callback_t callback) {
  interrupt_state_t intstate = interruptsSaveAndDisable();
  
  uart_state.callback = callback;

  /* Power up the USART. */
  *AT91C_PMC_PCER = (1 << AT91C_ID_US1);

  /* Hand the USART I/O pins over to the controller. */
  *AT91C_PIOA_PDR = UART_PIOA_PINS;
  *AT91C_PIOA_ASR = UART_PIOA_PINS;

  /* Disable both receiver and transmitter, inhibit USART interrupts,
   * and reset all of the controller's components.
   */
  *AT91C_US1_CR = AT91C_US_TXDIS | AT91C_US_RXDIS;
  *AT91C_US1_IDR = ~0;
  *AT91C_US1_CR = (AT91C_US_RSTRX | AT91C_US_RSTTX |
       AT91C_US_RSTSTA | AT91C_US_RSTNACK);

  /* configure/reset the PDC */

  /* We must put a size != 0 in the RCR register (even if the PDC is
   * disabled for the receiving) else when we try to read manually a
   * value on US1_RHR thanks to the RXRDY interruption the RXRDY of the
   * CSR seems to never be set to 1 (no value read on the UART ?)
   *
   * TODO : figure this out
   */
  *AT91C_US1_RPR = (uint32_t)(&uart_state.buf);
  *AT91C_US1_RCR = UART_BUFSIZE;
  *AT91C_US1_TPR = 0;
  *AT91C_US1_TCR = 0;
  *AT91C_US1_RNPR = 0;
  *AT91C_US1_RNCR = 0;
  *AT91C_US1_TNPR = 0;
  *AT91C_US1_TNCR = 0;

  /* Configure the USART for:
   *  - Hardware handshaking
   *  - Master clock as the clock source
   *  - 8-bit characters, 1 stop bit, no parity bit
   *  - Asynchronous communication
   *  - No receive timeout
   */
  *AT91C_US1_MR = (AT91C_US_USMODE_HWHSH | AT91C_US_CLKS_CLOCK |
       AT91C_US_CHRL_8_BITS | AT91C_US_NBSTOP_1_BIT |
       AT91C_US_PAR_NONE | AT91C_US_CHMODE_NORMAL |
       AT91C_US_OVER);
  *AT91C_US1_BRGR = UART_CLOCK_DIVISOR;
  *AT91C_US1_RTOR = 0;

  /* Start listening for a character to receive. Since we programmed no
   * timeout, this will just start listening until something happens.
   */
  *AT91C_US1_CR = AT91C_US_STTTO;

  /* Install an interrupt handler and start listening on interesting
   * interrupt sources.
   */
  aicInstallIsr(AT91C_ID_US1, AIC_PRIO_MEDIUMHIGH, AIC_TRIG_LEVEL, _uart_isr);
  *AT91C_US1_IER = AT91C_US_RXRDY | AT91C_US_RXBRK;

  /* Activate the USART and its associated DMA controller. */
  *AT91C_US1_CR = AT91C_US_TXEN | AT91C_US_RXEN;
  *AT91C_US1_PTCR = AT91C_PDC_TXTEN;

  interruptsRestore(intstate);
}

/* this is blocking! (unless _uartIsReady()==TRUE) */
static void _uartWrite(const uint8_t *data, uint32_t lng) {
  ASSERT(data != NULL);
  ASSERT(lng > 0);

  while (*AT91C_US1_TNCR != 0);

  *AT91C_US1_TNPR = (uint32_t)data;
  *AT91C_US1_TNCR = lng;
}

static bool _uartIsReady(void) {
  return (*AT91C_US1_TNCR == 0);
}

static bool _uartIsWriting(void) {
  return (*AT91C_US1_TCR + *AT91C_US1_TNCR) > 0;
}

void _uartSetCallback(nx__uart_read_callback_t callback) {
  if (callback == NULL) {

    *AT91C_US1_IDR = AT91C_US_RXRDY | AT91C_US_RXBRK | AT91C_US_ENDRX;
    /* we disable the PDC */
    *AT91C_US1_PTCR = AT91C_PDC_RXTDIS;
    uart_state.callback = callback;

    *AT91C_US1_RCR = 0;
    *AT91C_US1_RPR = (uint32_t)NULL;

  } else {
    *AT91C_US1_IDR = AT91C_US_RXRDY | AT91C_US_RXBRK | AT91C_US_ENDRX;

    *AT91C_US1_PTCR = AT91C_PDC_RXTDIS;

    *AT91C_US1_RCR = UART_BUFSIZE;
    *AT91C_US1_RPR = (uint32_t)(&uart_state.buf);

    uart_state.callback = callback;

    /* we reenable the reading of the first char (ie the packet size) */
    *AT91C_US1_IER = AT91C_US_RXRDY | AT91C_US_RXBRK;
  }

}

static void _uartRead(uint8_t *buf, uint32_t length) {
  uart_state.to_read = length;

  *AT91C_US1_RPR = (uint32_t)buf;
  *AT91C_US1_RCR = length;

  if (buf != NULL && length > 0) {
    *AT91C_US1_PTCR = AT91C_PDC_RXTEN;
  } else {
    *AT91C_US1_PTCR = AT91C_PDC_RXTDIS;
  }
}

static uint32_t _uartReadCount(void) {
  return uart_state.to_read - (*AT91C_US1_RCR);
}


/******************************************
 * Here comes the real Bluetooth driver.  *
 ******************************************/


#define BT_ADDR_SIZE  7    /* Size of a bluetooth address (in bytes) */
#define BT_CLASS_SIZE 4    /* Size of a bluetooth class (in bytes) */
#define BT_NAME_MAX_LNG 16 /* Maximum length of bluetooth friendly name (in bytes) */
#define BT_PIN_MAX_LNG  16 /* Maximum length of bluetooth pin code (in bytes) */

typedef struct bt_device {
  uint8_t addr[BT_ADDR_SIZE];
  char name[BT_NAME_MAX_LNG+1];
  uint8_t class[BT_CLASS_SIZE];
} bt_device_t;

typedef enum {
  BT_STATE_WAITING = 0x0,
  BT_STATE_INQUIRING,
  BT_STATE_KNOWN_DEVICES_DUMPING,
  BT_STATE_STREAMING
} bt_state_t;

typedef enum {
  BT_NOTHING = 0x0, /* can mean no answer from the bluecore */
  BT_LR_SUCCESS = 0x50,
  BT_LR_COULD_NOT_SAVE,
  BT_LR_STORE_IS_FULL,
  BT_LR_ENTRY_REMOVED,
  BT_LR_UNKNOWN_ADDR
} bt_return_value_t;

typedef enum {
  BT_DISCONNECTION_SUCCESS,
  BT_DISCONNECTION_LINK_LOSS,
  BT_DISCONNECTION_NO_SLC, /* Service Level Connection */
  BT_DISCONNECTION_TIMEOUT,
  BT_DISCONNECTION_ERROR
} bt_disconnection_status_t;

typedef struct bt_version {
  uint8_t major;
  uint8_t minor;
} bt_version_t;

#define BT_ACK_TIMEOUT 3000
#define BT_ARGS_BUFSIZE (BT_NAME_MAX_LNG+1)

/* to remove : */
/*#define UART_DEBUG*/
#ifdef UART_DEBUG
#define CMDS_BUFSIZE 128
#define USB_SEND(txt) _usbWrite((uint8_t*)txt, strlen(txt))
#else
#define CMDS_BUFSIZE 0
#define USB_SEND(txt)
#endif

#define BT_RST_PIN     AT91C_PIO_PA11
#define BT_ARM_CMD_PIN AT91C_PIO_PA27
/* BT_BC4_CMD is connected to the channel 4 (counting from 0) of the Analog to Digital converter */
#define BT_CS_PIN      AT91C_PIO_PA31

/*** MESSAGES CODES ***/

typedef enum {
  /* AMR7 -> BC4 */
  BT_MSG_BEGIN_INQUIRY = 0x00,
  BT_MSG_CANCEL_INQUIRY = 0x01,
  BT_MSG_CONNECT = 0x02,
  BT_MSG_OPEN_PORT = 0x03,
  BT_MSG_LOOKUP_NAME = 0x04,
  BT_MSG_ADD_DEVICE = 0x05,
  BT_MSG_REMOVE_DEVICE = 0x06,
  BT_MSG_DUMP_LIST = 0x07,
  BT_MSG_CLOSE_CONNECTION = 0x08,
  BT_MSG_ACCEPT_CONNECTION = 0x09,
  BT_MSG_PIN_CODE = 0x0A,
  BT_MSG_OPEN_STREAM = 0x0B,
  BT_MSG_START_HEART = 0x0C,
  BT_MSG_SET_DISCOVERABLE = 0x1C,
  BT_MSG_CLOSE_PORT = 0x1D,
  BT_MSG_SET_FRIENDLY_NAME = 0x21,
  BT_MSG_GET_LINK_QUALITY = 0x23,
  BT_MSG_SET_FACTORY_SETTINGS = 0x25,
  BT_MSG_GET_LOCAL_ADDR = 0x26,
  BT_MSG_GET_FRIENDLY_NAME = 0x29,
  BT_MSG_GET_DISCOVERABLE = 0x2A,
  BT_MSG_GET_PORT_OPEN = 0x2B,
  BT_MSG_GET_VERSION = 0x2F,
  BT_MSG_GET_BRICK_STATUS_BYTE = 0x33,
  BT_MSG_SET_BRICK_STATUS_BYTE = 0x34,
  BT_MSG_GET_OPERATING_MODE = 0x35,
  BT_MSG_SET_OPERATING_MODE = 0x36,
  BT_MSG_GET_CONNECTION_STATUS = 0x38,
  BT_MSG_GOTO_DFU_MODE = 0x3A,

  /* BC4 -> ARM7 */
  BT_MSG_HEARTBEAT = 0x0D,
  BT_MSG_INQUIRY_RUNNING = 0x0E,
  BT_MSG_INQUIRY_RESULT = 0x0F,
  BT_MSG_INQUIRY_STOPPED = 0x10,
  BT_MSG_LOOKUP_NAME_RESULT = 0x11,
  BT_MSG_LOOKUP_NAME_FAILURE = 0x12,
  BT_MSG_CONNECT_RESULT = 0x13,
  BT_MSG_RESET_INDICATION = 0x14,
  BT_MSG_REQUEST_PIN_CODE = 0x15,
  BT_MSG_REQUEST_CONNECTION = 0x16,
  BT_MSG_LIST_RESULT = 0x17,
  BT_MSG_LIST_ITEM = 0x18,
  BT_MSG_LIST_DUMP_STOPPED = 0x19,
  BT_MSG_CLOSE_CONNECTION_RESULT = 0x1A,
  BT_MSG_PORT_OPEN_RESULT = 0x1B,
  BT_MSG_CLOSE_PORT_RESULT = 0x1E,
  BT_MSG_PIN_CODE_ACK = 0x1F,
  BT_MSG_SET_DISCOVERABLE_ACK = 0x20,
  BT_MSG_SET_FRIENDLY_NAME_ACK = 0x22,
  BT_MSG_LINK_QUALITY_RESULT = 0x24,
  BT_MSG_SET_FACTORY_SETTINGS_ACK = 0x26,
  BT_MSG_GET_LOCAL_ADDR_RESULT = 0x28,
  BT_MSG_GET_FRIENDLY_NAME_RESULT = 0x2C,
  BT_MSG_GET_DISCOVERABLE_RESULT = 0x2D,
  BT_MSG_GET_PORT_OPEN_RESULT = 0x2E,
  BT_MSG_GET_VERSION_RESULT = 0X30,
  BT_MSG_GET_BRICK_STATUS_BYTE_RESULT = 0x31,
  BT_MSG_SET_BRICK_STATUS_BYTE_RESULT = 0x32,
  BT_MSG_OPERATING_MODE_RESULT = 0x37,
  BT_MSG_CONNECTION_STATUS_RESULT = 0x39,
} bt_msg_t;

/*** Predefined messages ***/

static const uint8_t bt_msg_start_heart[] = {
  0x03, /* length */
  BT_MSG_START_HEART,
  0xFF, /* checksum (high) */
  0xF4  /* checksum (low) */
};

static const uint8_t bt_msg_get_version[] = {
  0x03, /* length */
  BT_MSG_GET_VERSION,
  0xFF, /* checksum (hi) */
  0xD1  /* checksum (lo) */
};

static const uint8_t bt_msg_set_discoverable_true[] = {
  0x04, /* length */
  BT_MSG_SET_DISCOVERABLE,
  0x01, /* => true */
  0xFF,
  0xE3
};

static const uint8_t bt_msg_set_discoverable_false[] = {
  0x04, /* length */
  BT_MSG_SET_DISCOVERABLE, /* set discoverable */
  0x00, /* => false */
  0xFF,
  0xE4
};

static const uint8_t bt_msg_cancel_inquiry[] = {
  0x03, /* length */
  BT_MSG_CANCEL_INQUIRY,
  0xFF,
  0xFF
};

/* dump list of known devices */
static const uint8_t bt_msg_dump_list[] = {
  0x03,
  BT_MSG_DUMP_LIST,
  0xFF,
  0xF9
};

static const uint8_t bt_msg_get_friendly_name[] = {
  0x03,
  BT_MSG_GET_FRIENDLY_NAME,
  0xFF,
  0xD7
};

static const uint8_t bt_msg_open_port[] = {
  0x03,
  BT_MSG_OPEN_PORT,
  0xFF,
  0xFD
};

static const uint8_t bt_msg_accept_connection[] = {
  0x04,
  BT_MSG_ACCEPT_CONNECTION,
  0x01,
  0xFF,
  0xF6
};

static const uint8_t bt_msg_refuse_connection[] = {
  0x04,
  BT_MSG_ACCEPT_CONNECTION,
  0x00,
  0xFF,
  0xF7
};

static volatile struct {
  bt_state_t state;

  /* see nx_systick_get_ms() */
  uint32_t last_heartbeat;

  /* used for inquiring */
  uint32_t last_checked_id, remote_id; /* used to know when a new device is found */
  bt_device_t remote_device;

  /* all bytes to 0 if no device is waiting a pin code */
  uint8_t dev_waiting_for_pin[BT_ADDR_SIZE];
  /* all bytes to 0 if no device is waiting to connect */
  uint8_t dev_waiting_for_connection[BT_ADDR_SIZE];
  /* set when a new connection has been established. */
  int new_handle;

  uint8_t last_msg;

  /* use to return various return value (version, etc) */
  uint8_t args[BT_ARGS_BUFSIZE];

  int nmb_checksum_errors;

#ifdef UART_DEBUG
  /* to remove: */
  uint8_t cmds[CMDS_BUFSIZE];
  uint32_t cmds_pos;
#endif

} bt_state;

/* len => checksum included
 */
static uint32_t bt_get_checksum(uint8_t *msg, uint32_t len, bool count_len)
{
  uint32_t i;
  uint32_t checksum;

  /* from the second byte of the message
   * to the last byte before the checksum
   * 
   * Sivan: this was the original comment,
   * but the code includes the first byte
   * as well. 
   */
  checksum = 0;
  for (i = 0 ; i < len-2 ; i++) {
    checksum += msg[i];
  }

  if (count_len)
    checksum += len;

  checksum = -checksum;

  return checksum;
}

/* len => length excepted, but checksum included
 * two last bytes will be set
 */
static void bt_set_checksum(uint8_t *msg, uint32_t len) {
  uint32_t checksum = bt_get_checksum(msg, len, FALSE);

  msg[len-2] = ((checksum >> 8) & 0xFF);
  msg[len-1] = checksum & 0xFF;
}

/* len => length excepted, but checksum included */
static bool bt_check_checksum(uint8_t *msg, uint32_t len) {

  /* Strangeness: Must include the packet length in the checksum ?! */
  uint32_t checksum = bt_get_checksum(msg, len, TRUE);
  uint32_t hi, lo;

  hi = ((checksum >> 8) & 0xFF);
  lo = checksum & 0xFF;

  return (hi == msg[len-2] && lo == msg[len-1]);
}

static bool bt_wait_msg(uint8_t msg)
{
  uint32_t start = systick;

  while(bt_state.last_msg != msg && start+BT_ACK_TIMEOUT > systick);

  return bt_state.last_msg == msg;
}

static void bt_reseted(void)
{
  bt_state.state = BT_STATE_WAITING;
  //nx__uart_write(bt_msg_start_heart, sizeof(bt_msg_start_heart));
}




static void btUartCommandCallback(uint8_t *msg, uint32_t len)
{
  uint32_t i;

  /* if it's a break from the nxt */
  if (msg == NULL || len == 0) {
    bt_state.nmb_checksum_errors++;
    return;
  }

  /* we check first the checksum and ignore the message if the checksum is invalid */
  if (len < 1 || !bt_check_checksum(msg, len)) {
    bt_state.nmb_checksum_errors++;
    return;
  }

#ifdef UART_DEBUG
  if (bt_state.cmds_pos < CMDS_BUFSIZE) {
    bt_state.cmds[bt_state.cmds_pos] = msg[0];
    bt_state.cmds_pos++;
  }
#endif

  bt_state.last_msg = msg[0];

  /* we copy the arguments */
  for (i = 0 ; i < len-3 && i < BT_ARGS_BUFSIZE; i++) {
    bt_state.args[i] = msg[i+1];
  }

  for (; i < BT_ARGS_BUFSIZE ; i++) {
    bt_state.args[i] = 0;
  }


  if (msg[0] == BT_MSG_HEARTBEAT) {
    bt_state.last_heartbeat = systick;
    return;
  }


  if (msg[0] == BT_MSG_INQUIRY_RESULT) {
    if (bt_state.state != BT_STATE_INQUIRING)
      return;

    /* if the last device written wasn't fetched in the user
     * space, we don't erase it to avoid corrupted structures */
    if (bt_state.last_checked_id != bt_state.remote_id) {
      return;
    }

    /* we've found a device => we extract the fields */

    for (i = 0 ; i < BT_ADDR_SIZE ; i++)
      bt_state.remote_device.addr[i] = msg[1+i];

    for (i = 0 ; i < BT_NAME_MAX_LNG ; i++)
      bt_state.remote_device.name[i] = msg[1+BT_ADDR_SIZE+i];
    bt_state.remote_device.name[BT_NAME_MAX_LNG+1] = '\0';

    for (i = 0 ; i < BT_CLASS_SIZE ; i++)
      bt_state.remote_device.class[i] = msg[1+BT_ADDR_SIZE+BT_NAME_MAX_LNG+i];
      
    //bt_inquiry_result_event( bt_state.remote_device.name );

    bt_state.remote_id++;
    return;
  }

  if (msg[0] == BT_MSG_LIST_ITEM) {
    if (bt_state.state != BT_STATE_KNOWN_DEVICES_DUMPING)
      return;

    /* if the last device written wasn't fetched in the user
     * space, we don't erase it to avoid corrupted structures */
    if (bt_state.last_checked_id != bt_state.remote_id) {
      return;
    }

    for (i = 0 ; i < BT_ADDR_SIZE ; i++)
      bt_state.remote_device.addr[i] = msg[1+i];

    for (i = 0 ; i < BT_NAME_MAX_LNG ; i++)
      bt_state.remote_device.name[i] = msg[1+BT_ADDR_SIZE+i];
    bt_state.remote_device.name[BT_NAME_MAX_LNG+1] = '\0';

    for (i = 0 ; i < BT_CLASS_SIZE ; i++)
      bt_state.remote_device.class[i] = msg[1+BT_ADDR_SIZE+BT_NAME_MAX_LNG+i];

    bt_state.remote_id++;

    return;
  }


  if (msg[0] == BT_MSG_INQUIRY_STOPPED) {
    if (bt_state.state == BT_STATE_INQUIRING)
      bt_state.state = BT_STATE_WAITING;
    return;
  }

  if (msg[0] == BT_MSG_LIST_DUMP_STOPPED) {
    if (bt_state.state == BT_STATE_KNOWN_DEVICES_DUMPING)
      bt_state.state = BT_STATE_WAITING;
    return;
  }

  if (msg[0] == BT_MSG_RESET_INDICATION) {
    bt_reseted();
    return;
  }

  if (msg[0] == BT_MSG_REQUEST_PIN_CODE) {
    for (i = 0 ; i < BT_ADDR_SIZE ; i++) {
      bt_state.dev_waiting_for_pin[i] = bt_state.args[i];
    }

    return;
  }

  if (msg[0] == BT_MSG_REQUEST_CONNECTION) {
    for (i = 0 ; i < BT_ADDR_SIZE ; i++) {
      bt_state.dev_waiting_for_connection[i] = bt_state.args[i];
    }

    return;
  }

  if (msg[0] == BT_MSG_CONNECT_RESULT) {
    if (bt_state.args[0] >= 1)
      bt_state.new_handle = bt_state.args[1];
    return;
  }
}

static
void btInit(void)
{
  memset((void*)&bt_state, 0, sizeof(bt_state));
  USB_SEND("nx_bt_init()");

  bt_state.new_handle = -1;

  /* we put the ARM CMD pin to 0 => command mode */
  /* and we put the RST PIN to 1 => Will release the reset on the bluecore */
  *AT91C_PIOA_PER   = BT_RST_PIN | BT_ARM_CMD_PIN;
  *AT91C_PIOA_PPUDR = BT_ARM_CMD_PIN;
  *AT91C_PIOA_CODR  = BT_ARM_CMD_PIN;
  *AT91C_PIOA_SODR  = BT_RST_PIN;
  *AT91C_PIOA_OER   = BT_RST_PIN | BT_ARM_CMD_PIN;

  busywait(100*1000);
  
  _uartInit(btUartCommandCallback);

  bt_wait_msg(BT_MSG_RESET_INDICATION);
  /* the function bt_uart_callback() should start the heart after receiving the reset indication */

  USB_SEND("nx_bt_init() finished");
}


void nx_bt_set_friendly_name(char *name)
{
  int i;
  uint8_t packet[20] = { 0 };

  USB_SEND("set_friendly_name()");

  packet[0] = 19; /* length */
  packet[1] = BT_MSG_SET_FRIENDLY_NAME;

  for (i = 0 ; i < 16 && name[i] != '\0' ; i++)
    packet[i+2] = name[i];

  for (; i < 16 ; i++)
    packet[i+2] = '\0';

  bt_set_checksum(packet+1, 19);

  do {
    _uartWrite(packet, 20);
  } while(!bt_wait_msg(BT_MSG_SET_FRIENDLY_NAME_ACK));
}


void nx_bt_set_discoverable(bool d)
{
  USB_SEND("set_discoverable()");

  do {
    if (d)
      _uartWrite(bt_msg_set_discoverable_true, sizeof(bt_msg_set_discoverable_true));
    else
      _uartWrite(bt_msg_set_discoverable_false, sizeof(bt_msg_set_discoverable_false));
  } while(!bt_wait_msg(BT_MSG_SET_DISCOVERABLE_ACK));
}


bt_state_t nx_bt_get_state(void) {
  return bt_state.state;
}

void btBeginInquiry(uint8_t max_devices,
       uint8_t timeout,
       uint8_t bt_remote_class[4])
{
  int i;
  uint8_t packet[11];

  packet[0] = 10;      /* length */
  packet[1] = BT_MSG_BEGIN_INQUIRY; /* begin inquiry */
  packet[2] = max_devices;
  packet[3] = 0;       /* timeout (hi) */
  packet[4] = timeout; /* timeout (lo) */

  for (i = 0 ; i < 4 ; i++)
    packet[5+i] = bt_remote_class[i];

  bt_set_checksum(packet+1, 10);

  do {
    _uartWrite(packet, 11);
  } while(!bt_wait_msg(BT_MSG_INQUIRY_RUNNING));

  bt_state.last_checked_id = 0;
  bt_state.state = BT_STATE_INQUIRING;
}

bool btHasFoundDevice(void)
{
  if (bt_state.state == BT_STATE_INQUIRING)
    return (bt_state.last_checked_id != bt_state.remote_id);
  return FALSE;
}

bool btGetDiscoveredDevice(bt_device_t *dev)
{
  if (btHasFoundDevice()) {
    memcpy(dev, (bt_device_t *)&(bt_state.remote_device), sizeof(bt_device_t));
    bt_state.last_checked_id = bt_state.remote_id;
    return TRUE;
  }

  return FALSE;
}

void nx_bt_cancel_inquiry(void)
{
  _uartWrite(bt_msg_cancel_inquiry, sizeof(bt_msg_cancel_inquiry));
  bt_wait_msg(BT_MSG_INQUIRY_STOPPED);
}


void btBeginKnownDevicesDumping(void)
{
  _uartWrite(bt_msg_dump_list, sizeof(bt_msg_dump_list));
  bt_state.state = BT_STATE_KNOWN_DEVICES_DUMPING;
}

bool btHasKnownDevice(void)
{
  if (bt_state.state == BT_STATE_KNOWN_DEVICES_DUMPING)
    return (bt_state.last_checked_id != bt_state.remote_id);
  return FALSE;
}

bool btGetKnownDevice(bt_device_t *dev)
{
  if (btHasKnownDevice()) {

    memcpy(dev, (bt_device_t *)&(bt_state.remote_device), sizeof(bt_device_t));
    bt_state.last_checked_id = bt_state.remote_id;

    return TRUE;
  }

  return FALSE;
}

bt_return_value_t nx_bt_add_known_device(bt_device_t *dev)
{
  int i;
  uint8_t packet[31];

  packet[0] = 30; /* length */
  packet[1] = BT_MSG_ADD_DEVICE;

  for (i = 0 ; i < BT_ADDR_SIZE ; i++)
    packet[2+i] = dev->addr[i];

  for (i = 0 ; i < BT_NAME_MAX_LNG && dev->name != '\0' ; i++)
    packet[2+BT_ADDR_SIZE+i] = dev->name[i];

  for ( ; i < BT_NAME_MAX_LNG ; i++)
    packet[2+BT_ADDR_SIZE+i] = '\0';

  for (i = 0 ; i < BT_CLASS_SIZE ; i++)
    packet[2+BT_ADDR_SIZE+BT_NAME_MAX_LNG+i] = dev->class[i];

  bt_set_checksum(packet+1, 30);

  _uartWrite(packet, 31);

  bt_wait_msg(BT_MSG_LIST_RESULT);

  if (bt_state.last_msg == BT_MSG_LIST_RESULT)
    return bt_state.args[0];

  return 0;
}

bt_return_value_t nx_bt_remove_known_device(uint8_t dev_addr[BT_ADDR_SIZE])
{
  int i;
  uint8_t packet[11];

  packet[0] = 10; /* length */
  packet[1] = BT_MSG_REMOVE_DEVICE;

  for (i = 0; i < BT_ADDR_SIZE; i++) {
    packet[2+i] = dev_addr[i];
  }

  bt_set_checksum(packet+1, 10);

  _uartWrite(packet, 11);

  bt_wait_msg(BT_MSG_LIST_RESULT);

  if (bt_state.last_msg == BT_MSG_LIST_RESULT)
    return bt_state.args[0];

  return 0;
}

bt_version_t nx_bt_get_version(void)
{
  bt_version_t ver = { 0, 0 };

  _uartWrite(bt_msg_get_version, sizeof(bt_msg_get_version));

  if (bt_wait_msg(BT_MSG_GET_VERSION_RESULT)) {
    ver.major = bt_state.args[0];
    ver.minor = bt_state.args[1];
  }

  return ver;
}

static uint32_t btGetFriendlyName(char *name)
{
  int i;

  _uartWrite(bt_msg_get_friendly_name, sizeof(bt_msg_get_friendly_name));

  if (bt_wait_msg(BT_MSG_GET_FRIENDLY_NAME_RESULT)) {

    for (i = 0 ;
         i < BT_ARGS_BUFSIZE && i < BT_NAME_MAX_LNG && bt_state.args[i] != '\0' ;
         i++)
      name[i] = bt_state.args[i];

    name[i] = '\0';

    return i;

  } else {
    name[0] = '\0';
    return 0;
  }
}

int nx_bt_checksum_errors(void)
{
  return bt_state.nmb_checksum_errors;
}

int nx_bt_open_port(void)
{
  USB_SEND("open_port()");

  _uartWrite(bt_msg_open_port, sizeof(bt_msg_open_port));

  if (!bt_wait_msg(BT_MSG_PORT_OPEN_RESULT))
    return -1;

  if (bt_state.args[0] == 0) /* status = failed */
    return -1;

  return bt_state.args[1]; /* handle */
}

bool nx_bt_close_port(int handle)
{
  uint8_t packet[5];

  USB_SEND("close_port()");

  packet[0] = 4; /* length */
  packet[1] = BT_MSG_CLOSE_PORT;
  packet[2] = (uint8_t)handle;

  bt_set_checksum(packet+1, 4);

  _uartWrite(packet, 5);

  do {
    if (!bt_wait_msg(BT_MSG_CLOSE_PORT_RESULT))
      return FALSE;
  } while (bt_state.args[1] != handle); /* second byte is the handle */

  return (bt_state.args[0] >= 1); /* status byte */
}

bool nx_bt_has_dev_waiting_for_pin(void)
{
  int i;

  for (i = 0 ; i < BT_ADDR_SIZE ; i++)
    if (bt_state.dev_waiting_for_pin[i] != 0) {
      USB_SEND("has_dev_waiting_for_pin() => TRUE");
      return TRUE;
    }

  return FALSE;
}

void nx_bt_send_pin(char *code)
{
  int i;
  uint8_t packet[27];

  if (!nx_bt_has_dev_waiting_for_pin())
    return;

  USB_SEND("send_pin()");

  packet[0] = 26;
  packet[1] = BT_MSG_PIN_CODE;

  for (i = 0 ; i < BT_ADDR_SIZE ; i++)
    packet[2+i] = bt_state.dev_waiting_for_pin[i];

  for (i = 0 ; i < BT_PIN_MAX_LNG && code[i] != '\0' ; i++)
    packet[2+BT_ADDR_SIZE+i] = code[i];

  for (; i < BT_PIN_MAX_LNG ; i++)
    packet[2+BT_ADDR_SIZE+i] = '\0';

  bt_set_checksum(packet+1, 26);

  _uartWrite(packet, 27);

  bt_wait_msg(BT_MSG_PIN_CODE_ACK);

  for (i = 0 ; i < BT_ADDR_SIZE ; i++)
    bt_state.dev_waiting_for_pin[i] = 0;
}

static bool btIsConnectionPending(void)
{
  int i;

  for (i = 0 ; i < BT_ADDR_SIZE ; i++)
    if (bt_state.dev_waiting_for_connection[i] != 0) {
      USB_SEND("connection_pending => TRUE");
      return TRUE;
    }

  return FALSE;
}

void btAcceptConnection(bool accept)
{
  int i;

  if (!btIsConnectionPending())
    return;

  USB_SEND("accept_connection()");

  if (accept)
    _uartWrite(bt_msg_accept_connection, sizeof(bt_msg_accept_connection));
  else
    _uartWrite(bt_msg_refuse_connection, sizeof(bt_msg_refuse_connection));

  for (i = 0 ; i < BT_ADDR_SIZE ; i++)
    bt_state.dev_waiting_for_connection[i] = 0;

  while(_uartIsWriting());
}

int nx_bt_connection_established(void)
{
  int handle;

  handle = bt_state.new_handle;

  if (handle >= 0) {
    USB_SEND("connection_established() => TRUE");
    bt_state.new_handle = -1;
  }

  return handle;
}


uint8_t nx_bt_get_link_quality(int handle)
{
  uint8_t packet[5];

  USB_SEND("get_link_quality()");
  packet[0] = 4;
  packet[1] = BT_MSG_GET_LINK_QUALITY;
  packet[2] = (uint8_t)handle;
  bt_set_checksum(packet+1, 4);

  _uartWrite(packet+1, 5);

  if (!bt_wait_msg(BT_MSG_LINK_QUALITY_RESULT))
    return FALSE;

  return bt_state.args[0];
}


bt_disconnection_status_t nx_bt_close_connection(int handle)
{
  uint8_t packet[5];
  int ret_handle = 0;

  USB_SEND("close_connection()");

  packet[0] = 4;
  packet[1] = BT_MSG_CLOSE_CONNECTION;
  packet[2] = (uint8_t)handle;

  bt_set_checksum(packet+1, 4);

  _uartWrite(packet+1, 5);

  while (ret_handle != handle) {
    if (!bt_wait_msg(BT_MSG_CLOSE_CONNECTION_RESULT))
      return FALSE;
    ret_handle = bt_state.args[1];
  }

  return bt_state.args[0];
}

void btStreamOpen(uint32_t handle)
{
  uint8_t packet[5];

  USB_SEND("stream_open()");

  /* we make sure that the callback won't be called anymore */

  /* send open stream message */

  packet[0] = 4; /* length */
  packet[1] = BT_MSG_OPEN_STREAM;
  packet[2] = (uint8_t)handle;

  bt_set_checksum(packet+1, 4);

  _uartWrite(packet, 5);

  while(_uartIsWriting());

  _uartSetCallback(NULL);

  /* set ARM_CMD to high to go in stream mode */
  *AT91C_PIOA_SODR = BT_ARM_CMD_PIN;

  bt_state.state = BT_STATE_STREAMING;
}

static inline void btStreamWrite(uint8_t *data, uint32_t length)
{
  USB_SEND("stream_write()");
  _uartWrite(data, length);
}

static bool btStreamIsOpen(void)
{
  return bt_state.state == BT_STATE_STREAMING;
}


static inline bool btStreamIsWriting(void)
{
  return _uartIsWriting();
}

static inline void btStreamRead(uint8_t *buf, uint32_t length)
{
  USB_SEND("stream_read()");
  _uartRead(buf, length);
}

static inline uint32_t btStreamReadCount(void)
{
  return _uartReadCount();
}

static void btStreamClose(void)
{
  USB_SEND("stream_close()");

  /* we put back the callback in place: */
  _uartSetCallback(btUartCommandCallback);

  /* return to command mode by lowering the ARM_CMD pin */
  *AT91C_PIOA_CODR = BT_ARM_CMD_PIN;
  bt_state.state = BT_STATE_WAITING;
}

/* to remove: */
static void nx_bt_debug(void)
{
  //nx_display_uint(bt_state.last_heartbeat);
  //nx_display_end_line();
  USB_SEND((char *)bt_state.cmds);
}


#if 0  
  {
    bt_device_t dev;

    /* Listing known devices */

    displayClear();
    displayString("Known devices: \n");
    btBeginKnownDevicesDumping();
    while(nx_bt_get_state() == BT_STATE_KNOWN_DEVICES_DUMPING) {
      if (btHasKnownDevice()) {
        btGetKnownDevice(&dev);
        displayString(dev.name);
        displayString("\n");
      }
    }

    busywait(4*1000*1000);
  }
  
  {
    bt_device_t dev;

    displayClear();
    displayString("Scanning: \n");

    btBeginInquiry(/* max dev : */ 255,
                   /* timeout : */ 0x20,
                   /* class :   */ (uint8_t[]){ 0, 0, 0, 0 });

    while(nx_bt_get_state() == BT_STATE_INQUIRING) {
      if (btHasFoundDevice()) {
        btGetDiscoveredDevice(&dev);
        btGetKnownDevice(&dev);
        displayString(dev.name);
        displayString("\n");
        nx_bt_add_known_device(&dev);
      }
    }

    busywait(4*1000*1000);
  }
  

  sensorportsEnableAnalog(0);
  sensorportsDigiClear(0,0);
  
  //displayString("Hello! ");
  //displayString(version);
  //displayString("\n");
  
  { 
    uint8_t s[BT_NAME_MAX_LNG+1];
    btGetFriendlyName(s);
    displayString("bt: ");
    displayString(s);
    displayString("\n");
  }
#endif    
