/* Copyright (C) 2007 the NxOS developers
 *
 * See AUTHORS for a full list of the developers.
 *
 * Redistribution of this file is permitted under
 * the terms of the GNU Public License (GPL) version 2.
 * 
 * Modified by Sivan Toledo.
 */
 
/** Two Wire Interface
 *
 * The TWI (Two Wire Interface) is the hardware I2C controller that
 * controls the AVR coprocessor communication bus. There are no other
 * peripherals on that bus, and no others can be added, so
 * this driver is used exclusively by the AVR driver.
 *
 * Warning Messing with this driver is a bad idea! If you mess with it
 * and successfully screw up the AVR driver, the coprocessor link will
 * time out, which basically crashes half of the ASSERTT's functionality. Be
 * careful.
 */

typedef enum  {
    TWI_UNINITIALIZED = 0,
    TWI_FAILED,
    TWI_READY,
    TWI_TX_BUSY,
    TWI_RX_BUSY,
} i2c_state_t;

static volatile struct {
  i2c_state_t mode;

  /* Address and size of a send/receive buffer. */
  uint8_t  *ptr;
  uint32_t len;
} _i2c_state = {
  TWI_UNINITIALIZED, /* Not initialized yet. */
  NULL,              /* No send/recv buffer. */
  0,                 /* And zero length, obviously. */
};

static void __attribute__ ((interrupt("IRQ"))) _i2c_isr(void) {
  /* Read the status register once to acknowledge all TWI interrupts. */
  uint32_t status = *AT91C_TWI_SR;

  /* Read mode and the status indicates a byte was received. */
  if (_i2c_state.mode == TWI_RX_BUSY && (status & AT91C_TWI_RXRDY)) {
    *(_i2c_state.ptr) = *AT91C_TWI_RHR;
    _i2c_state.ptr++;
    _i2c_state.len--;

    /* If only one byte is left to read, instruct the TWI to send a STOP
     * condition after the next byte.
     */
    if (_i2c_state.len == 1) {
      *AT91C_TWI_CR = AT91C_TWI_STOP;
    }

    /* If the read is over, inhibit all TWI interrupts and return to the
     * ready state.
     */
    if (_i2c_state.len == 0) {
      //extern dsr_t i2c_receive_done_dsr;
      //dsrPostFromIsr( &i2c_receive_done_dsr );

      *AT91C_TWI_IDR = ~0;
      _i2c_state.mode = TWI_READY;
    }
  }

  /* Write mode and the status indicated a byte was transmitted. */
  if (_i2c_state.mode == TWI_TX_BUSY && (status & AT91C_TWI_TXRDY)) {
    /* If that was the last byte, inhibit TWI interrupts and return to
     * the ready state.
     */
    if (_i2c_state.len == 0) {
      //extern dsr_t i2c_send_done_dsr;
      //dsrPostFromIsr( &i2c_send_done_dsr );
      
      *AT91C_TWI_IDR = ~0;
      _i2c_state.mode = TWI_READY;
    } else {
      /* Instruct the TWI to send a STOP condition at the end of the
       * next byte if this is the last byte.
       */
      if (_i2c_state.len == 1)
        *AT91C_TWI_CR = AT91C_TWI_STOP;

      /* Write the next byte to the transmit register. */
      *AT91C_TWI_THR = *_i2c_state.ptr;
      _i2c_state.ptr++;
      _i2c_state.len--;
    }
  }

  /* Check for error conditions. There are pretty critical failures,
   * since they indicate something is very wrong with either this
   * driver, or the coprocessor, or the hardware link between them.
   */
  if (status & (AT91C_TWI_OVRE | AT91C_TWI_UNRE | AT91C_TWI_NACK)) {
    *AT91C_TWI_CR = AT91C_TWI_STOP;
    *AT91C_TWI_IDR = ~0;
    _i2c_state.mode = TWI_FAILED;
    /* TODO: This should be an assertion failed, ie. a hard crash. */
  }
  
  aicUpdatePriority(); /* tell AIC we are done */
}

/*
 *  Initialize the TWI driver. 
 */
static
void _i2cInit(void)
{
  uint32_t clocks = 9;

  /* Power up the TWI and PIO controllers. */
  *AT91C_PMC_PCER = (1 << AT91C_ID_TWI) | (1 << AT91C_ID_PIOA);

  /* Inhibit all TWI interrupt sources. */
  *AT91C_TWI_IDR = ~0;

  /* If the system is rebooting, the coprocessor might believe that it
   * is in the middle of an I2C transaction. Furthermore, it may be
   * pulling the data line low, in the case of a read transaction.
   *
   * The TWI hardware has a bug that will lock it up if it is
   * initialized when one of the clock or data lines is low.
   *
   * So, before initializing the TWI, we manually take control of the
   * pins using the PIO controller, and manually drive a few clock
   * cycles, until the data line goes high.
   */
  *AT91C_PIOA_MDER = AT91C_PA3_TWD | AT91C_PA4_TWCK;
  *AT91C_PIOA_PER = AT91C_PA3_TWD | AT91C_PA4_TWCK;
  *AT91C_PIOA_ODR = AT91C_PA3_TWD;
  *AT91C_PIOA_OER = AT91C_PA4_TWCK;

  while (clocks > 0 && !(*AT91C_PIOA_PDSR & AT91C_PA3_TWD)) {
    *AT91C_PIOA_CODR = AT91C_PA4_TWCK;
    busywait(10);
    *AT91C_PIOA_SODR = AT91C_PA4_TWCK;
    busywait(10);
    clocks--;
  }

  /* Now that the I2C lines are clean, hand them back to the TWI
   * controller.
   */
  *AT91C_PIOA_PDR = AT91C_PA3_TWD | AT91C_PA4_TWCK;
  *AT91C_PIOA_ASR = AT91C_PA3_TWD | AT91C_PA4_TWCK;

  /* Reset the controller and configure its clock. The clock setting
   * makes the TWI run at 380kHz.
   */
  *AT91C_TWI_CR = AT91C_TWI_SWRST | AT91C_TWI_MSDIS;
  *AT91C_TWI_CWGR = 0x020f0f;
  *AT91C_TWI_CR = AT91C_TWI_MSEN;
  _i2c_state.mode = TWI_READY;

  /* Install the TWI interrupt handler. */
  aicInstallIsr(AT91C_ID_TWI, AIC_PRIO_MEDIUMLOW, AIC_TRIG_LEVEL, _i2c_isr);
}

static
void _i2cReadAsync(uint32_t dev_addr, uint8_t* data, uint32_t len)
{
  uint32_t mode = ((dev_addr & 0x7f) << 16) | AT91C_TWI_IADRSZ_NO | AT91C_TWI_MREAD;

  ASSERT(dev_addr == 1);
  ASSERT(data != NULL);
  ASSERT(len > 0);
  ASSERT(nx__i2c_ready());

  /* TODO: assert(nx__i2c_ready()) */

  _i2c_state.mode = TWI_RX_BUSY;
  _i2c_state.ptr = data;
  _i2c_state.len = len;

  *AT91C_TWI_MMR = mode;
  *AT91C_TWI_CR = AT91C_TWI_START | AT91C_TWI_MSEN;
  *AT91C_TWI_IER = AT91C_TWI_RXRDY;
}

static
void _i2cWriteAsync(uint32_t dev_addr, uint8_t* data, uint32_t len)
{
  uint32_t mode = ((dev_addr & 0x7f) << 16) | AT91C_TWI_IADRSZ_NO;

  ASSERT(dev_addr == 1);
  ASSERT(data != NULL);
  ASSERT(len > 0);
  ASSERT(nx__twi_ready());

  _i2c_state.mode = TWI_TX_BUSY;
  _i2c_state.ptr = data;
  _i2c_state.len = len;

  *AT91C_TWI_MMR = mode;
  *AT91C_TWI_CR = AT91C_TWI_START | AT91C_TWI_MSEN;
  *AT91C_TWI_IER = AT91C_TWI_TXRDY;
}

static
bool _i2cIsReady(void) {
  return (_i2c_state.mode == TWI_READY) ? TRUE : FALSE;
}
 
 
/*
 * Now come the AVR interface.
 */ 

typedef enum {
  BUTTON_NONE = 0, /* no button pressed.             */
  BUTTON_OK,       /* central orange button pressed. */
  BUTTON_CANCEL,   /* dark gray button pressed.      */
  BUTTON_LEFT,     
  BUTTON_RIGHT,    
} nxt_button_t;

typedef enum {
  MOTOR_A = 0,
  MOTOR_B = 1,
  MOTOR_C = 2,
} nxt_motor_t;

#define AVR_ADDRESS 1
#define AVR_MAX_FAILED_CHECKSUMS 3

const char _avr_init_handshake[] =
  "\xCC" "Let's samba nxt arm in arm, (c)LEGO System A/S";

static volatile struct {
  /* The current mode of the AVR state machine. */
  enum {
    AVR_UNINITIALIZED = 0, /* Initialization not completed. */
    AVR_LINK_DOWN,         /* No handshake has been sent. */
    AVR_INIT,              /* Handshake send in progress. */
    AVR_WAIT_2MS,          /* Timed wait after the handshake. */
    AVR_WAIT_1MS,          /* More timed wait. */
    AVR_SEND,              /* Sending of to_avr in progress. */
    AVR_RECV,              /* Reception of from_avr in progress. */
  } mode;

  /* Used to detect link failures and restart the AVR link. */
  uint8_t failed_consecutive_checksums;
} _avr_state = {
  AVR_UNINITIALIZED, /* We start uninitialized. */
  0,                 /* No failed checksums yet. */
};

/* Contains all the commands that are periodically sent to the AVR. */
static volatile struct {
  /* Tells the AVR to perform power management: */
  enum {
    AVR_RUN = 0,    /* No power management (normal runtime mode). */
    AVR_POWER_OFF,  /* Power down the brick. */
    AVR_RESET_MODE, /* Go into SAM-BA reset mode. */
  } power_mode;

  /* The speed and braking configuration of the motor ports. */
  int8_t  motor_speed[NXT_N_MOTORS];
  uint8_t motor_brake;

} _avr_to = {
  AVR_RUN,     /* Start in normal power mode. */
  { 0, 0, 0 }, /* All motors are off... */
  0            /* And set to coast. */
};

/* Contains all the status data periodically received from the AVR. */
static volatile struct {
  /* The analog reading of the analog pin on all active sensors. */
  uint16_t adc_value[NXT_N_SENSORS];

  /* The state of the NXT's buttons. Given the way that the buttons
   * are handled in hardware, only one button is reported pressed at a
   * time. See the nx_avr_button_t enumeration for values to test for.
   */
  uint8_t buttons;

  /* Battery information. */
  struct {
    bool is_aa; /* True if the power supply is AA batteries (as
                 * opposed to a battery pack).
                 */
    uint16_t charge; /* The remaining battery charge in mV. */
  } battery;

  /* The version of the AVR firmware. The currently supported version
   * is 1.1.
   */
  struct {
    uint8_t major;
    uint8_t minor;
  } version;
} _avr_from;


/* The following two arrays hold the data structures above, converted
 * into the raw ARM-AVR communication format. Data to send is
 * serialized into this buffer prior to sending, and received data is
 * received into here before being deserialized into the status
 * struct.
 */
static uint8_t _avr_raw_from[(2 * NXT_N_SENSORS) + /* Sensor A/D value. */
                       2 + /* Buttons reading.  */
                       2 + /* Battery type, charge and AVR firmware
                            * version. */
                       1]; /* Checksum. */
static uint8_t _avr_raw_to[1 + /* Power mode    */
                     1 + /* PWM frequency */
                     4 + /* output % for the 4 (?!)  motors */
                     1 + /* Output modes (brakes), one bit per motor */
                     1 + /* Input modes (sensor power) */
                     1]; /* Checksum */

/* Serialize the _avr_to data structure into _avr_raw_to, ready for
 * sending to the AVR.
 */
static void _avrPackTo(void) {
  uint32_t i;
  uint8_t checksum = 0;

  memset(_avr_raw_to, 0, sizeof(_avr_raw_to));

  /* Marshal the power mode configuration. */
  switch (_avr_to.power_mode) {
  case AVR_RUN:
    /* Normal operating mode. First byte is 0, and the second (PWM
     * frequency is set to 8.
     */
    _avr_raw_to[1] = 8;
    break;
  case AVR_POWER_OFF:
    /* Tell the AVR to shut us down. */
    _avr_raw_to[0] = 0x5A;
    _avr_raw_to[1] = 0;
    break;
  case AVR_RESET_MODE:
    /* Tell the AVR to boot SAM-BA. */
    _avr_raw_to[0] = 0x5A;
    _avr_raw_to[1] = 0xA5;
  }

  /* Marshal the motor speed settings. */
  _avr_raw_to[2] = _avr_to.motor_speed[0];
  _avr_raw_to[3] = _avr_to.motor_speed[1];
  _avr_raw_to[4] = _avr_to.motor_speed[2];

  /* _avr_raw_to[5] is the value for the 4th motor, which doesn't
   * exist. This is probably a bug in the AVR firmware, but it is
   * required. So we just latch the value to zero.
   */

  /* Marshal the motor brake settings, bits in 1 byte. */
  _avr_raw_to[6] = _avr_to.motor_brake;

  /* Calculate the checksum. */
  for (i=0; i<(sizeof(_avr_raw_to)-1); i++)
    checksum += _avr_raw_to[i];
  _avr_raw_to[sizeof(_avr_raw_to)-1] = ~checksum;
}

/* Small helper to convert two bytes into an uint16_t. */
static inline uint16_t _avr_unpack_word(uint8_t *word) {
  return *((uint16_t*)word);
}

/* Deserialize the AVR data structure in _avr_raw_from into the
 * _avr_from status structure.
 */
static void _avrUnpackFrom(void) {
  uint8_t checksum = 0;
  uint16_t word;
  uint32_t voltage;
  uint32_t i;
  uint8_t *p = _avr_raw_from;

  /* Compute the checksum of the received data. This is done by doing
   * the unsigned sum of all the bytes in the received buffer. They
   * should add up to 0xFF.
   */
  for (i=0; i<sizeof(_avr_raw_from); i++)
    checksum += _avr_raw_from[i];

  if (checksum != 0xff) {
    _avr_state.failed_consecutive_checksums++;
    return;
  } else {
    _avr_state.failed_consecutive_checksums = 0;
  }

  /* Unpack and store the 4 sensor analog readings. */
  for (i = 0; i < NXT_N_SENSORS; i++) {
    _avr_from.adc_value[i] = _avr_unpack_word(p);
    p += 2;
  }

  /* Grab the buttons word (an analog reading), and compute the state
   * of buttons from that.
   */
  word = _avr_unpack_word(p);
  p += 2;

  if (word > 1023)
    _avr_from.buttons = BUTTON_OK;
  else if (word > 720)
    _avr_from.buttons = BUTTON_CANCEL;
  else if (word > 270)
    _avr_from.buttons = BUTTON_RIGHT;
  else if (word > 60)
    _avr_from.buttons = BUTTON_LEFT;
  else
    _avr_from.buttons = BUTTON_NONE;

  /* Process the last word, which is a mix and match of many
   * values.
   */
  word = _avr_unpack_word(p);

  /* Extract the AVR firmware version, as well as the type of power
   * supply connected.
   */
  _avr_from.version.major = (word >> 13) & 0x3;
  _avr_from.version.minor = (word >> 10) & 0x7;
  _avr_from.battery.is_aa = (word & 0x8000) ? TRUE : FALSE;

  /* The rest of the word is the voltage value, in units of
   * 13.848mV. As the NXT does not have a floating point unit, the
   * multiplication by 13.848 is approximated by a multiplication by
   * 3545 followed by a division by 256.
   */
  voltage = word & 0x3ff;
  voltage = (voltage * 3545) >> 8;
  _avr_from.battery.charge = voltage;
}

/* 
 * Initialize the NXT-AVR communication. 
 * 
 * Must be called after i2cInit.
 * 
 */
static
void avrInit(void) {
  _i2cInit();
  
  _avr_state.mode = AVR_LINK_DOWN;
}

/* The main AVR driver state machine. This routine gets called
 * periodically every millisecond by the system timer code.
 *
 * It is called directly in the main system timer interrupt, and so
 * must return as fast as possible.
 */
static
void avrPeriodicTask(void) {
  /* The action taken depends on the state of the AVR
   * communication.
   */
  switch (_avr_state.mode) {
  case AVR_UNINITIALIZED:
    /* Because the system timer can call this update routine before
     * the driver is initialized, we have this safe state. It does
     * nothing and immediately returns.
     *
     * When the AVR driver initialization code runs, it will set the
     * state to AVR_LINK_DOWN, which will kickstart the state machine.
     */
    return;

  case AVR_LINK_DOWN:
    /* ARM-AVR link is not initialized. We need to send the hello
     * string to tell the AVR that we are alive. This will (among
     * other things) stop the "clicking brick" sound, and avoid having
     * the brick powered down after a few minutes by an AVR that
     * doesn't see us coming up.
     */
    _i2cWriteAsync(AVR_ADDRESS, (uint8_t*) _avr_init_handshake,
             sizeof(_avr_init_handshake)-1);
    _avr_state.failed_consecutive_checksums = 0;
    _avr_state.mode = AVR_INIT;
    break;

  case AVR_INIT:
    /* Once the transmission of the handshake is complete, go into a 2
     * millisecond wait, which is accomplished by the use of two
     * intermediate state machine states.
     */
    if (_i2cIsReady())
      _avr_state.mode = AVR_WAIT_2MS;
    break;

  case AVR_WAIT_2MS:
    /* Wait another millisecond... */
    _avr_state.mode = AVR_WAIT_1MS;
    break;

  case AVR_WAIT_1MS:
    /* Now switch the state to send mode, but also set the receive
     * done flag. On the next refresh cycle, the communication will be
     * in "production" mode, and will start by reading data back from
     * the AVR.
     */
    _avr_state.mode = AVR_SEND;
    break;

  case AVR_SEND:
    /* If the transmission is complete, switch to receive mode and
     * read the status structure from the AVR.
     */
    if (_i2cIsReady()) {
      _avr_state.mode = AVR_RECV;
      memset(_avr_raw_from, 0, sizeof(_avr_raw_from));
      _i2cReadAsync(AVR_ADDRESS, _avr_raw_from, sizeof(_avr_raw_from));
    }

  case AVR_RECV:
    /* If the transmission is complete, unpack the read data into the
     * _avr_from struct, pack the data in the _avr_to struct into a raw
     * buffer, and shovel that over the i2c bus to the AVR.
     */
    if (_i2cIsReady()) {
      _avrUnpackFrom();
      /* If the number of failed consecutive checksums is over the
       * restart threshold, consider the link down and reboot the
       * link. */
      if (_avr_state.failed_consecutive_checksums >= AVR_MAX_FAILED_CHECKSUMS) {
        _avr_state.mode = AVR_LINK_DOWN;
      } else {
        _avr_state.mode = AVR_SEND;
        _avrPackTo();
        _i2cWriteAsync(AVR_ADDRESS, _avr_raw_to, sizeof(_avr_raw_to));
      }
    }
    break;
  }
}

/****************** user interface functions *******************/

static
uint32_t sensorportsGetRaw(uint32_t n) {
  ASSERT(n < NXT_N_SENSORS);

  return _avr_from.adc_value[n];
}

static
void motorSet(uint32_t motor, int power_percent, bool brake) {
  interrupt_state_t intstate;
  
  {
    interrupt_state_t intstate = interruptsSaveAndDisable();

    _avr_to.motor_speed[motor] = power_percent;
    if (brake)
      _avr_to.motor_brake |= (1 << motor);
    else
      _avr_to.motor_brake &= ~(1 << motor);

    interruptsRestore(intstate);
  }
}

static
void powerOff(void) {
  while (1)
    _avr_to.power_mode = AVR_POWER_OFF;
}

/*
 * Total reset of the NXT's processor.
 */
#define powerReset() (*AT91C_RSTC_RCR = 0xA5000005)

static
void powerFirmwareUpdateMode(void) {
  while (1)
    _avr_to.power_mode = AVR_RESET_MODE;
}

static
uint32_t powerGetBatteryVoltage(void) {
  return _avr_from.battery.charge;
}

static
bool powerIsBatteryAA(void) {
  return _avr_from.battery.is_aa;
}

static
nxt_button_t buttonGet(void) {
  return _avr_from.buttons;
}

static
void avrGetVersion(uint8_t *major, uint8_t *minor) {
  {
    interrupt_state_t intstate = interruptsSaveAndDisable();

    if (major) *major = _avr_from.version.major;
    if (minor) *minor = _avr_from.version.minor;

    interruptsRestore(intstate);
  }
}

