/* Copyright (C) 2007 the NxOS developers
 *
 * See AUTHORS for a full list of the developers.
 *
 * Redistribution of this file is permitted under
 * the terms of the GNU Public License (GPL) version 2.
 * 
 * Modified by Sivan Toledo.
 * Interrupt-state manipulation code from eCos and FreeRTOS.
 */

typedef uint32_t interrupt_state_t;

#ifndef __thumb__

static inline uint32_t interruptsSaveAndEnable(void) {
  uint32_t cpsr;
  asm volatile (                            
        "mrs %0,cpsr;"
        "mrs r4,cpsr;"
        "bic r4,r4,#0xC0;"
        "msr cpsr,r4"
        : "=r" (cpsr) 
        : /* no inputs */ 
        : "r4" );
  return cpsr;
}

static inline uint32_t interruptsSaveAndDisable(void) {
  uint32_t cpsr;
  asm volatile (                            
        "mrs %0,cpsr;"
        "mrs r4,cpsr;"
        "orr r4,r4,#0xC0;"
        "msr cpsr,r4"
        : "=r" (cpsr) 
        : /* no inputs */ 
        : "r4" );
  return cpsr;
}

static inline void interruptsRestore(uint32_t cpsr) {
  asm volatile (                              
      "mrs r3,cpsr;"                          
      "and r4,%0,#0xC0;"                      
      "bic r3,r3,#0xC0;"                      
      "orr r3,r3,r4;"                         
      "msr cpsr,r3"                           
      :                                       
      : "r"(cpsr)                            
      : "r3", "r4"                            
      );
}

static inline void interruptsEnable(void) {
    asm volatile (                              
        "mrs r3,cpsr;"                          
        "bic r3,r3,#0xC0;"                      
        "msr cpsr,r3"                           
        :                                       
        :                                       
        : "r3"                                  
        );
}
        
#else // __thumb__
#error "interrupt state manipulation not supported in thumb mode"
#endif

typedef enum {
  AIC_PRIO_LOWEST = 0,
  AIC_PRIO_VERYLOW = 1,     
  AIC_PRIO_LOW = 2,     
  AIC_PRIO_MEDIUMLOW = 3,  
  AIC_PRIO_MEDIUMHIGH = 4, 
  AIC_PRIO_HIGH = 5,   
  AIC_PRIO_VERYHIGH = 6,
  AIC_PRIO_HIGHEST = 7,
} aic_priority_t;

/** Interrupt trigger modes.
 * 
 * The trigger mode to use usually depends on the specific hardware
 * peripheral being controlled. Consult the board documentation if you
 * need to set interrupt handlers.
 */
typedef enum {
  AIC_TRIG_LEVEL = 0,   /**< Level-triggered interrupt. */
  AIC_TRIG_EDGE = 1,    /**< Edge-triggered interrupt. */
} aic_trigger_mode_t;

#define aicEnable(vector)   (*AT91C_AIC_IECR = (1 << vector))
#define aicDisable(vector)  (*AT91C_AIC_IDCR = (1 << vector))
#define aicSet(vector)      (*AT91C_AIC_ISCR = (1 << vector))
#define aicClear(vector)    (*AT91C_AIC_ICCR = (1 << vector))
#define aicUpdatePriority() (*AT91C_AIC_EOICR = 0)

static
void __attribute__ ((interrupt("IRQ"))) spurious_isr(void) {
  aicUpdatePriority();
}

/* in case we are not starting from a clean state */
static
void aicInitSlow(void) {
  int i;

  /* If we're coming from a warm boot, the AIC may be in a weird
   * state. Do some cleaning up to bring the AIC back into a known
   * state:
   *  - All interrupt lines disabled,
   *  - No interrupt lines handled by the FIQ handler,
   *  - No pending interrupts,
   *  - AIC idle, not handling an interrupt.
   */
  *AT91C_AIC_IDCR = 0xFFFFFFFF;
  *AT91C_AIC_FFDR = 0xFFFFFFFF;
  *AT91C_AIC_ICCR = 0xFFFFFFFF;
  *AT91C_AIC_EOICR = 1;

  /* Enable debug protection. This is necessary for JTAG debugging, so
   * that the hardware debugger can read AIC registers without
   * triggering side-effects.
   */
  //*AT91C_AIC_DCR = 1;

  /* Set default handlers for all interrupt lines. */
  for (i = 0; i < 32; i++) {
    AT91C_AIC_SMR[i] = 0;
    AT91C_AIC_SVR[i] = 0; /* will cause a reset */
  }
  AT91C_AIC_SVR[AT91C_ID_FIQ] = 0;
  *AT91C_AIC_SPU = (uint32_t) spurious_isr;
}

/* 
 * This init routine should be called 
 * from the reset configuration and with interrupts still
 * disabled.
 */
static
void aicInit(void) {
  *AT91C_AIC_SPU = (uint32_t) spurious_isr;
}

static
void aicInstallIsr(uint32_t vector, aic_priority_t prio,
                   aic_trigger_mode_t trig_mode, void (*isr)(void) ) {
  /* 
   * Disable the interrupt we're installing. Getting interrupted while
   * we are tweaking it could be bad.
   */
  aicDisable(vector);
  aicClear(vector);

  AT91C_AIC_SMR[vector] = (trig_mode << 5) | prio;
  AT91C_AIC_SVR[vector] = (uint32_t)isr;

  aicEnable(vector);
}
