function [model] = generate_simple_model(n, model_type, opt_vis)
% 
% function [model] = generate_simple_model(n, model_type, opt_vis)
%
% Generates a simple model of an arc shaped conductive material with random conductance.
%   The model has approximately 'n' variables.
%   If 'model_type' is 'poisson' the model conatins linear finite-element 3x3 matrices.
%   If 'model_type' is 'laplacian' the model contains random 3x3 sdd matrices.
%   If 'opt_vis' is 'true' a simple illustration of the mesh is drawn.
%
% Example:
%   [model] = generate_simple_model(1000,'poisson',true) 
%    generates a model with 1008 variables and 1860 linear finite-elements,
%    it also draws the underlying mesh.
%
% S. Toledo, H. Avron, G. Shklarski. March 2007
% All rights reserved.

%% Parse input
if (n < 4)
    display('Invalid "n", using default n=100');
    n = 100;
end
model_type = lower(model_type);
if (~isequal(model_type,'poisson') && ~isequal(model_type,'laplacian'))
    display('Invalid "model_type", using default model_type=Poisson');
    model_type = 'poisson';
end

%% Construct the geometry

h = floor(n^0.6);
v = ceil(n^0.4);

x = [];
y = [];
triangles = [];

for i=1:h
    for j=1:v
        x = [x; i + rand(1)*(n>30)*.5];
        y = [y; j + sin(i/h*pi)*v];
        if ((i<h) && (j<v))
            triangles = [triangles; 
                        (i-1)*v+j, i*v+j, (i-1)*v+j+1;
                        (i-1)*v+j+1, i*v+j+1, i*v+j];
        end
    end
end


n = length(x);
m = size(triangles,1);

%% Optional visualizatoin

if (opt_vis == true)
    axis([0 max(h,2*v)*1.1 0 max(h,2*v)*1.1])
    for i=1:m
        line([y(triangles(i,1)),y(triangles(i,2))],[x(triangles(i,1)),x(triangles(i,2))]);
        line([y(triangles(i,1)),y(triangles(i,3))],[x(triangles(i,1)),x(triangles(i,3))]);
        line([y(triangles(i,3)),y(triangles(i,2))],[x(triangles(i,3)),x(triangles(i,2))]);
    end
end

%% Construct the model

model.total_num_vars = n;
model.n_elements = m;

element.matrix = zeros(3);
element.local_to_global = zeros(3,1);
model.elements = repmat(element,m,1);

if (isequal(model_type,'poisson'))
    for i=1:m
        lx = x(triangles(i,:));
        ly = y(triangles(i,:));
        if (det([1 lx(1) ly(1); 1 lx(2) ly(2); 1 lx(3) ly(3)]) < 0)
            triangles(i,[1 2 3]) =  triangles(i,[1 3 2]);
        end
        element.local_to_global = triangles(i,:);
        element.matrix = linear_element_2d(x(triangles(i,:)),y(triangles(i,:)),0.5+rand(1)/2);
        model.elements(i) = element;
    end
elseif (isequal(model_type,'laplacian'))    
    V = [ 1  1  0;
         -1  0  1;
          0 -1 -1];
    for i=1:m
        lx = x(triangles(i,:));
        ly = y(triangles(i,:));
        d = diag(1./[sqrt((lx(1)-lx(2))^2+(ly(1)-ly(2))^2)...
                    sqrt((lx(1)-lx(3))^2+(ly(1)-ly(3))^2)... 
                    sqrt((lx(2)-lx(3))^2+(ly(2)-ly(3))^2)]);
        element.local_to_global = triangles(i,:);
        element.matrix = (0.5+rand(1)/2)*V*d*V';
        model.elements(i) = element;
    end

end
