// stock.cpp
// Implementation of a single stock in the market

#include <assert.h>
#include <string.h>
#include "random.h"
#include "stock.h"


CStock::CStock(const char *sName, int nMin, int nMax, int nInit) {	
	assert(nMin <nMax && nMin >=1);

	bInitialized=0; bTrendInitialized=0; bStabilityInitialized=0;	
	if (nInit <nMin || nInit >nMax) nInit=(nMin +nMax) /2;

	CStock::nMin=nMin;
	CStock::nMax=nMax;
	CStock::nInit=nInit;
	strncpy(CStock::sName,sName,MAX_NAME);

	nCurrent=nInit;
	trend.SetSize(nMax -nMin +1);
	stability.SetSize(nMax -nMin +1);

	INFO_MSG(("Stock::Created stock %s (%d,%d,%d)\n",sName,nMin,nMax,nInit));	
}

CStock::~CStock() {
	INFO_MSG(("Stock::Deleted stock %s\n",sName));	
}


void CStock::setTrend(tTrend &values) {
	for (int i=0;i <nMax -nMin +1;i++) {
		trend[i]=values[i];

		if (trend[i] <0.) trend[i]=0.;
		if (trend[i] >1.) trend[i]=1.;
	}
	trend[0]=1.;
	trend[nMax -nMin]=0.;  

	INFO_MSG(("Stock::Initialized trend of stock %s: (",sName));		
	for (i=0;i <nMax -nMin +1;i++)
		INFO_MSG(("%f ",trend[i]));	
	INFO_MSG((")\n"));	

	bTrendInitialized=1;
	bInitialized=bTrendInitialized && bStabilityInitialized;
}


void CStock::setStability(tStability &values) {
	for (int i=0;i <nMax -nMin +1;i++) {
		stability[i]=values[i];

		if (stability[i] <0) stability[i]=0;		
	}
	
	INFO_MSG(("Stock::Initialized stability of stock %s: (",sName));	
	for (i=0;i <nMax -nMin +1;i++)
		INFO_MSG(("%f ",stability[i]));	
	INFO_MSG((")\n"));	

	bStabilityInitialized=1;
	bInitialized=bTrendInitialized && bStabilityInitialized;
}


int CStock::nextDay() {
	if (!bInitialized) return -1;

	int id = nCurrent - nMin;

	// Decide if the stock is going up or down
	bool bIncrease = SYSTEM_GENERATOR->bernoulli(trend[id]);

	// Decide the amount of change
	int nChange = SYSTEM_GENERATOR->poisson(stability[id]);

	// Change the current value
	if (bIncrease) {
		nCurrent=(nCurrent +nChange <=nMax)?nCurrent +nChange:nMax;
		INFO_MSG(("Stock::Increased %s by %d, new value is %d\n",sName,nChange,nCurrent));	
	}
	else {
		nCurrent=(nCurrent -nChange >=nMin)?nCurrent -nChange:nMin;
		INFO_MSG(("Stock::Decreased %s by %d, new value is %d\n",sName,nChange,nCurrent));	
	}

	return nCurrent;
}

void CStock::restart() {
	nCurrent=nInit;

	INFO_MSG(("Stock::Restarted %s, new value is %d\n",sName,nCurrent));	
}