// model.cpp
// Implementation of stock market and asset portfolio model

#include <assert.h>
#include "model.h"


CModel::CModel(const char *sModelName, int nValueBinSize) {
	/*/////////////// From File ////////////////
	tMoney nInit = 2.0;								
	//////////////////////////////////////////

	pMarket = new CStockMarket();
	pPortfolio = new CPortfolio(nInit);

	nDay=1;	
	CModel::nValueBinSize=nValueBinSize;

	/////////////// From File ////////////////	
	tTrend trend;
	tStability stability;
	trend.SetSize(15);
	stability.SetSize(15);
	for (int i=0;i <15;i++) {
		trend[i]=1. -1./30 *i;	
	  stability[i]=(float)(i /3. +1./3.);
		//stability[i]=(float)1;
	}
	pMarket->addStock("STCK1",16,30,21,trend,stability);

	for (i=0;i <10;i++) {
    trend[i]=0.5;	
		stability[i]=(float)3;		
	}
	//pMarket->addStock("STCK2",16,25,16,trend,stability);
	//////////////////////////////////////////

	INFO_MSG(("Model::Model is up with %d stocks\n",pMarket->getStocksCount()));	
	bModelUp=true;*/

	pMarket=0;
	pPortfolio=0;

	nDay=1;	
	CModel::nValueBinSize=nValueBinSize;

	bModelUp=load(sModelName);	

	if (bModelUp)
	  INFO_MSG(("Model::Model is up with %d stocks\n",pMarket->getStocksCount()));	
	else INFO_MSG(("Model::Error while loading model\n"));
}

CModel::~CModel() {
	delete pMarket;
	delete pPortfolio;

	bModelUp=0;
	INFO_MSG(("Model::Model is down\n"));	
}


void CModel::restart() {
	nDay=1;

	pMarket->restart();
	pPortfolio->restart();

	INFO_MSG(("Model::Model is restarted\n"));	
}


// The ecpected structure of the model file is:
//		[Portfolio init value]	(float)
//		[Number of stocks]			(int)
//		for each stock:
//			[Name]
//			[Min, Max, Init]			(int,int,int)
//			[Stock trend]					(array of float)
//			[Stock stability]			(array of float)

bool CModel::load(const char *sFileName) {
	FILE *fIn;
	
  // Open the model file
	fIn=fopen(sFileName,"r");
	if (!fIn) return false;
  
	// Read portfolio init value and number of stocks
	tMoney nInitValue;
	int nStocks;

	if (fscanf(fIn,"%f %d",&nInitValue,&nStocks)!=2) return false;		
	if (nInitValue <1. || nStocks <1) return false;

	pMarket = new CStockMarket();
	pPortfolio = new CPortfolio(nInitValue);

	for (int i=0;i <nStocks;i++) {
		char sName[MAX_NAME];
		int nMin, nMax,nInit;
		tTrend trend;
	  tStability stability;

		// Name
		if (fscanf(fIn,"%s",sName)!=1) return false;			
		
		// Min, Max, Init
		if (fscanf(fIn,"%d %d %d",&nMin,&nMax,&nInit)!=3) return false;		
		if (nMin >=nMax || nMin < 1) return false;
		
		int nSize = nMax -nMin +1;
		trend.SetSize(nSize);
	  stability.SetSize(nSize);

		// Trend, Stability
		float f;
		for (int j=0;j <nSize;j++) {
			if (fscanf(fIn,"%f",&f)!=1) return false;
			if (f <0. || f >1.) return false;			
			
			trend[j]=f;
		}

		for (j=0;j <nSize;j++) {
			if (fscanf(fIn,"%f",&f)!=1) return false;		
			if (f <0) return false;

			stability[j]=f;
		}

		pMarket->addStock(sName,nMin,nMax,nInit,trend,stability);
	}	

	fclose(fIn);
	return true;
}

tReward CModel::performAction(CAction &action) {	
	assert(action.nPercentage >0. && action.nPercentage <=1.);
	assert(action.nStock >=0 && action.nStock <=pMarket->getStocksCount());

  INFO_MSG(("Model::Performing action <%d,%f>\n",action.nStock,action.nPercentage));	

	// Perform the action, pay commision
	///////////
	
	tMoney nOldValue = pPortfolio->getValue();

	if (action.nStock != pPortfolio->getStock()) {
		// Currently not invested
		if (pPortfolio->getStock()==0) {
			// Should invest in some stock
			tMoney nCommission = pMarket->commission(Buy,
																							 pPortfolio->getValue() *action.nPercentage);
			pPortfolio->pay(nCommission);
			pPortfolio->invest(action.nStock,action.nPercentage);
		}		
		// Currently invested in some stock
		else 			
			if (action.nStock==0) {
				// Should sell all investments
				tMoney nCommission = pMarket->commission(Sell,pPortfolio->getInvestedValue());
				pPortfolio->pay(nCommission);
			  pPortfolio->invest(0,1.);
			}
			else {
				// Should invest in some other stock, first sell and then buy
				tMoney nCommission = pMarket->commission(Sell,pPortfolio->getInvestedValue());
				pPortfolio->pay(nCommission);
			  pPortfolio->invest(0,1.);

				nCommission = pMarket->commission(Buy,
				 																  pPortfolio->getValue() *action.nPercentage);
			  pPortfolio->pay(nCommission);
			  pPortfolio->invest(action.nStock,action.nPercentage);
			}
	}
	
	
	// New day in the market
	///////////

	int nOldStockValue = pMarket->getStockValue(pPortfolio->getStock());

	pMarket->nextDay();
	nDay++;

	
	// Compute the new portfolio value
	///////////

	int nNewStockValue = pMarket->getStockValue(pPortfolio->getStock());
	if (pPortfolio->getStock() !=0)
		pPortfolio->change(float(nNewStockValue) /float(nOldStockValue));
  tMoney nNewValue = pPortfolio->getValue();

  INFO_MSG(("Model::At the end of the day portfolio value is %f\n",pPortfolio->getValue()));	


	// Compute the reward
	///////////

	return nNewValue -nOldValue;
}

// This method is used to publish the state of the model. The internal
// state is translated to a model state. 

CState CModel::getState() {
	// Represent the portfolio value using bins
	tMoney nPortfolioValue = (tMoney)(int((pPortfolio->getValue() +nValueBinSize /2.) 
																		    /(float)nValueBinSize) *nValueBinSize);

	CState S(pPortfolio->getStock(),pPortfolio->getPercentage(),
					 nPortfolioValue);

	for (int i=1;i <=pMarket->getStocksCount();i++)
		S.setStockValue(i,pMarket->getStockValue(i));

	return S;
}