// random.cpp

#include <math.h>
#include "common.h"
#include "random.h"


// Implementation of methods of CRandomGenerator
////////////////////////////////////////////////

CRandomGenerator::CRandomGenerator() {
	INFO_MSG(("RandomGenerator::Generator is up\n"));
	
	nIdumUniform=-1, nIdumPossion =-1; nIdumBernoulli =-1;
}

CRandomGenerator::~CRandomGenerator() {
	INFO_MSG(("RandomGenerator::Generator is down\n"));	
}


// Returns the value ln(Gamma(xx) for xx > 0.  Full accuracy is 
// obtained for xx > 1. For 0 < xx < 1., the reflection formula (6.1.4)
// can be used first.
// (Adapted from Numerical Recipies in C)

double CRandomGenerator::gammln(double xx) {
	double x,y,tmp,ser;
	static double cof[6]={76.18009172947146,
		-86.50532032941677,24.01409824083091,
		-1.231739572450155, 0.1208650973866179e-2,
		-0.5395239384953e-5};
	int j;

	y=x=xx;
	tmp=x+5.5;
	tmp -= (x+0.5)*log(tmp);
	ser=1.000000000190015;
	for (j=0;j<=5;j++) ser += cof[j]/+y;
	
	return -tmp+log(2.5066282746310005*ser/x);
}		


// "Minimal" random number generator of Park and Miller with Bays-Durham 
// shuffle and added safeguards. Returns a uniform random deviate 
// between 0.0 and 1.0 (exclusive of the endpoint values). Call with 
// idum a negative integer to initialize; thereafter, do not alter idum
// between successive deviates in a sequence. RNMX should approximate 
// the largest oating value that is less than 1.
// (Adapted from Numerical Recipies in C)

double CRandomGenerator::flat(int *idum)
	#define IA 16807
	#define IM 2147483647
	#define AM (1.0/IM)
	#define IQ 127773
	#define IR 2836
	#define NTAB 32
	#define NDIV (1+(IM-1)/NTAB)
	#define EPS 1.2e-7
	#define RNMX (1.0-EPS)

{
	int j;
	long k;
	static long iy=0;
	static long iv[NTAB];
	double temp;

	if (*idum <=0 || !iy) {						// Initialize
		if (-(*idum) <1) *idum=1;				// Be sure to prevent idum = 0
		else *idum = -(*idum);
		for (j=NTAB+7;j>=0;j--) {				// Load the shu.e table (after 8 warm-ups)
			k=(*idum)/IQ;
			*idum=IA*(*idum -k*IQ) -IR*k;
			if (*idum <0) *idum +=IM;
			if (j < NTAB) iv[j] =*idum;
		}
		iy=iv[0];
	}
	
	k=(*idum) /IQ;									// Start here when not initializing
	*idum=IA*(*idum -k*IQ) -IR*k;		// Compute idum=(IA*idum) % IM without
	if (*idum <0) *idum += IM;			//		overflows by Schrage's method
	j=iy /NDIV;											// Will be in the range 0..NTAB-1
	iy=iv[j];												// Output previously stored value and refill the
	iv[j]=*idum;										//		shuffe table
	if ((temp=AM*iy) >RNMX)		 			// Because users don't expect endpoint values
		return RNMX;
	else return temp;
}


// Returns as a floating-point number an integer value that is a random
// deviate drawn from a Poisson distribution of mean xm, using flat(idum)
// as a source of uniform random deviates.
// (Adapted from Numerical Recipies in C)

long CRandomGenerator::poisson(float xm, int *idum) {
  static double sq, alxm, g, oldm =(-1.0);	// oldm is a flag for whether xm has changed
	double em,t,y;														//	since last call

	if (xm <12.0) {					// Use direct method
		if (xm!=oldm) {
			oldm=xm;
			g=exp(-xm);					// If xm is new, compute the exponential
		}

		em=-1.0;
		t=1.0;
		do {								// Instead of adding exponential deviates it is equiv-
			++em;							//	alent to multiply uniform deviates. We never
			t*=flat(idum);		//	actually have to take the log, merely com-
		} while (t >g);			//	pare to the pre-computed exponential

	} else {							// Use rejection method
		if (xm != oldm) {		// If xm has changed since the last call, then pre-
			oldm=xm;					//	compute some functions that occur below
			sq=sqrt(2.0*xm);
			alxm=log(xm);
			g=xm*alxm - gammln(xm +1.0);
		}
		do {	
			do {												// y is a deviate from a Lorentzian comparison 
				y=tan(PI*flat(idum));			//	function
				em=sq*y+xm;								// em is y, shifted and scaled
			} while (em < 0.0);					// Reject if in regime of zero probability
			em=floor(em);								// The trick for integer-valued distributions

			t=0.9*(1.0+y*y)*exp(em*alxm-gammln(em+1.0)-g);
			// The ratio of the desired distribution to the comparison function; 
			// we accept or reject by comparing it to another uniform deviate.
			// The factor 0.9 is chosen so that t never exceeds 1.
		} while (flat(idum) >t);
	}

	return long(em);
}


// Returns a success value of a bernoulli trial with success probability
// of, using flat(idum) as a source of uniform random deviates.

bool CRandomGenerator::bernoulli(float s, int *idum) {
	if (s <0.) s=0.;
	if (s >1.) s=1.;

	return (flat(idum) <s);
}
  
