program RUBC_ge
use base_lib
use omp_lib
implicit none


!RUBC_ge.f90
!
!This program solves, simulates, and computes IRFs for the GE version of the RUBC model
!
!"Really Uncertain Business Cycles"
!Nick Bloom, Max Floetotto, Nir Jaimovich, Itay Saporta-Eksten, Stephen J. Terry
!
!This version contains lagged unc in the forecast rule, and this
!version also clears a convexified excess demand function.
!
!This Version: September, 24, 2016
!Lagged unc in the forecast rule
!Clears markets using convexified excess demand
!Computes IRFs using difference method of Koop, et al. (1996)
!!!!!!

!!!!DECLARATION BLOCK

integer :: znum,anum,snum,knum,lnum,numstates,numexog,numendog,numper,numdiscard,seedint,vfmaxit,&
    accelmaxit,numconstants,ct,zct,act,sct,kct,lmin1ct,zprimect,aprimect,sprimect,primect,endogct,&
    exogct,polct,vfct,accelct,polstar,exogprimect,t,numsimIRF,lengthIRF,shockperIRF,numperIRF,&
    numdiscIRF,ainit,sinit,kbarnum,numfcst,fcstct,lct,kprimect,ind,piter,maxpit,doVFI,distinit,&
    GEct,maxGEit,perct,doIRF,pcutoff,checkbounds,shocklengthIRF,simct,GEerrorswitch,exitflag,&
    singleshock,smin1ct,seeddim,pnum,pind,IRFrestart,IRFsimst,IRFctst,writect,writect2
        
integer, allocatable :: exog_pos(:,:),endog_pos(:,:),loopind(:,:),polmat(:,:,:),polmatold(:,:,:),&
    asimpos(:),ssimpos(:),asimposIRF(:,:,:),ssimposIRF(:,:,:),kbarfcstinds(:,:,:,:),polmatp(:,:),&
    kprime_posp(:,:),lpol_posp(:,:),piterstoresim(:),piterstoresimIRF(:,:,:),seedarray(:),&
    polmatp_interp(:,:,:)

double precision :: start,finish,alpha,nu,theta,deltak,deltan,beta,capirrev,capfix,hirelin,firelin,&
    labfix,pval,wval,ajump,zjump,uncpers,uncfreq,rhoz,sigmaz,rhoa,sigmaa,zmin,zmax,amin,amax,kmin,kmax,&
    lmin,lmax,vferrortol,zval,aval,sval,kval,lmin1val,kprimeval,lval,ival,yval,ackval,aclval,vferror,&
    polerror,hval,changetol,kbarmin,kbarmax,kbarval,kbarfcstval,pfcstval,wfcstval,Vnextval,weight,&
    perror,perrortol,Yvalp,Ivalp,ACkvalp,AClvalp,Cvalp,pupdate,Kbarprimevalp,Hvalp,Lvalp,GEupdate,&
    xmean,ymean,x2mean,xymean,xval,kfcsterror,pfcsterror,fcsterrortol,pvala,pvalb,pvalc,fa,fb,fc,plb,&
    pub,pwidth,disttol,pwindow,KRMSEchange,pRMSEchange,KR2change,pR2change,RMSEchangetol,R2changetol,&
    maxDenHaanchangetol,avgDenHaanchangetol,pmaxDenHaanchange,KmaxDenHaanchange,pavgDenHaanchange,&
    KavgDenHaanchange,pwgt

double precision, allocatable :: V(:,:,:),asimshocks(:),ssimshocks(:),pr_mat_z(:,:,:),&
    pr_mat_a(:,:,:),pr_mat_s(:,:),pr_mat(:,:),z0(:),k0(:),l0(:),a0(:),sigmazgrid(:),sigmaagrid(:),&
    MATLABconstants(:),exog_key(:,:),endog_key(:,:),EVmat(:,:,:),Vold(:,:,:),RHSvec(:),kprime_key(:,:,:),&
    lpol_key(:,:,:),kprime_pos(:,:,:),lpol_pos(:,:,:),distzkl(:,:,:),Ysim(:),Ksim(:),Lsim(:),Isim(:),Hsim(:),&
    ACksim(:),AClsim(:),TOPksim(:),BOTksim(:),TOPlsim(:),BOTlsim(:),ADJksim(:),ADJlsim(:),asimshocksIRF(:,:),&
    ssimshocksIRF(:,:),YsimIRF(:,:,:),KsimIRF(:,:,:),LsimIRF(:,:,:),IsimIRF(:,:,:),HsimIRF(:,:,:),ACksimIRF(:,:,:),&
    AClsimIRF(:,:,:),ADJksimIRF(:,:,:),ADJlsimIRF(:,:,:),kbar0(:),kbarfcstmat(:,:,:,:),pfcstmat(:,:,:,:),kfcstcoeff(:,:,:,:),&
    pfcstcoeff(:,:,:,:),wfcstmat(:,:,:,:),Ymat(:,:,:,:),Imat(:,:),ACkmat(:,:,:,:,:),AClmat(:,:,:,:,:),&
    WLmat(:,:,:,:,:),kbarfcstweights(:,:,:,:),psim(:),EVmatp(:,:),Csim(:),Kfcstsim(:),pfcstsim(:),&
    kcoeffstore(:,:,:,:,:),pcoeffstore(:,:,:,:,:),kfcstcoeffnew(:,:,:,:),pfcstcoeffnew(:,:,:,:),&
    kfcsterrorstore(:),pfcsterrorstore(:),psimIRF(:,:,:),CsimIRF(:,:,:),KfcstsimIRF(:,:,:),pfcstsimIRF(:,:,:),&
    kfcsterrorstoreIRF(:,:,:),pfcsterrorstoreIRF(:,:,:),distzklbefore(:,:),distzklafter(:,:),pR2store(:,:,:,:),&
    KR2store(:,:,:,:),pRMSEstore(:,:,:,:),KRMSEstore(:,:,:,:),pSSE(:,:,:),KSSE(:,:,:),pSST(:,:,:),KSST(:,:,:),Kmean(:,:,:),&
    pmean(:,:,:),KRMSEchangestore(:),pRMSEchangestore(:),KR2changestore(:),pR2changestore(:),pDenHaanfcst(:),&
    KDenHaanfcst(:),pmaxDenHaanstat(:),KmaxDenHaanstat(:),pavgDenHaanstat(:),KavgDenHaanstat(:),&
    pmaxDenHaanchangestore(:),KmaxDenHaanchangestore(:),pavgDenHaanchangestore(:),KavgDenHaanchangestore(:),p0(:),&
    ep0(:),Cp0(:),Yp0(:),Ip0(:),ACkp0(:),AClp0(:),Kbarprimep0(:),Hp0(:),Lp0(:)

!!!!END DECLARATION BLOCK

!!!!PROGRAM SETUP BLOCK

doVFI = 0 !1 implies do VFI, 0 implies read from file
distinit = 0; !1 implies read initial distribution from file (only matters for GE solution, not IRF)
doIRF = 1; !1 implies do the IRF after GE iteration is complete
singleshock = 1; !1 implies a single shock period for the IRF, 0 implies a low unc period followed by high unc
checkbounds = 1; !1 implies that after the uncond sim you will check to see how often you the top of the grid
IRFrestart = 0; !0 implies that this is the first time you're doing the IRF, 1 implies that you need to only do from IRFsimst to numsimIRF, where IRFsimst is in IRFsimst.txt

!GE convergence criteria
!1 implies convergence of coefficients
!2 implies convergence of RMSE in changes
!3 implies convergence of R^2 in changes
!4 implies convergence of max Den Haan stat in changes
!5 implies convergence of avg Den Haan stat in changes
GEerrorswitch = 4; 

!call the clock to get the start time and start writing to the log file
start = omp_get_wtime()
open(13,file="RUBC_ge.txt")

!these lines simply space the log file and shell view
write(*,*) "OpenMP parallelization setup"
write(13,*) "OpenMP parallelization setup"

!NOTE: all writing commands are duplicated - once to the default * channel, which 
!is the shell you view in runtime, and once to channel 13, which is the log file

!test the parallelization to see how many threads this system can use, noting 
!that the OpenMP defaults are a good number of threads to select
!$omp parallel
write(*,*) "Parallel hello to you!"
write(13,*) "Parallel hello to you!"
!$omp end parallel

write(*,*) " "
write(13,*) " "

!!!!END PROGRAM SETUP BLOCK

!!!!!PARAMETER ENTRY BLOCK

!technology and adjustment costs
alpha = 0.25
nu = 0.5
theta = 2.0
deltak = 0.026
deltan = 0.088
beta = 0.95 ** 0.25
capirrev = 0.339; 
capfix = 0.0
hirelin = 0.018*4.0; 
firelin = hirelin
labfix = 0.024*4;

!grid sizes for exog processes
znum = 5; !idio prod
anum = 5; !agg prod
snum = 2; !volatility

!grid size for interpolation of excess demand function
pnum = 25
plb = 1.1;
pub = 1.5

!grid sizes for endog processes
knum = 91
lnum = 37

!grid size for the aggregate moments
kbarnum = 10

!set up the unc process
ajump = 1.60569110682638; !multiple of vol in agg case
zjump = 4.11699578856773; !multiple of vol in idio case
uncpers = 0.940556523390567; !cond. unc shock persistence
uncfreq = 0.0257892725462263; !cond. prob of unc shock

!set up the idio prod process
rhoz = 0.95; 
sigmaz = 0.0507515557155377
zmin = exp( - 2.0 * ( ( sigmaz ** 2.0 ) / ( 1.0 - rhoz ** 2.0 ) ) ** 0.5 );
zmax = exp( 2.0 * ( ( sigmaz ** 2.0 ) / ( 1.0 - rhoz ** 2.0 ) ) ** 0.5);

!set up the agg prod process
rhoa = 0.95;
sigmaa = 0.00668420914017636
amin = exp( - 2.0 * ( ( sigmaa ** 2.0 ) / ( 1.0 - rhoa ** 2.0 ) ) ** 0.5);
amax = exp( 2.0 * ( ( sigmaa ** 2.0 ) / ( 1.0 - rhoa ** 2.0 ) ) ** 0.5);

!set up the idio capital grid, respecting depreciation
kmin = 1.125
kmax = exp(log(kmin) - dble(knum-1) * log(1.0-deltak))
write(*,"(A,F10.3,A,F10.3,A)") "capital grid = (",kmin,",",kmax,")"
write(13,"(A,F10.3,A,F10.3,A)") "capital grid = (",kmin,",",kmax,")"

!set up the idio labor grid, respecting depreciation
lmin = 0.035
lmax = exp(log(lmin) - dble(lnum-1) * log(1.0-deltan))
write(*,"(A,F10.3,A,F10.3,A)") "labor grid = (",lmin,",",lmax,")"
write(13,"(A,F10.3,A,F10.3,A)") "labor grid = (",lmin,",",lmax,")"

!set up the aggregate capital grid
kbarmin = 3.0
kbarmax = 10.0

!figure out total number of states
numexog = znum*anum*snum*snum
numendog = knum*lnum
numfcst = kbarnum
numstates = numexog * numendog * numfcst

!control the uncondtional simulation
numper = 100
numdiscard = 50
seedint = 2501; call random_seed(size=seeddim)
ainit = 3; !initial agg prod val (uncond sim and IRF)
sinit = 1; !initial unc val (uncond sim only, 1 = low, 2 = high)

!control the IRF
numsimIRF = 2500; !number of shocked economies to simulate
lengthIRF = 100; !length of each economy
shockperIRF = 45; !period in which to shock the economy
shocklengthIRF = 5; !number of periods to have low unc 
numdiscIRF = 45; !number of periods to discard in the IRF

numperIRF = numsimIRF * lengthIRF; !total number of periods in IRF simulation

!control the vf process
vfmaxit=50; !number of VF iterations
vferrortol = 1e-4; !VF error tolerance
accelmaxit = 200; !number of Howard acceleration iterations

!control the price-clearing process
maxpit = 50
perrortol = 1.0e-4
disttol = 1e-4; !tolerance for ignoring this point in the dist
pwindow = 0.025
pcutoff = 15

!control the GE fcst rule update process
maxGEit = 1
GEupdate = 0.33; !step speed in fcst rule fixed-point iteration process
fcsterrortol = 0.0125; !the convergence tolerance for the forecast rules 
RMSEchangetol = 0.001; !the convergence tolerance for RMSE changes
R2changetol = 0.01; !the convergence tolerance for R2 changes
maxDenHaanchangetol = 0.01; !the convergence tolerance for Den Haan max statistic
avgDenHaanchangetol = 0.01; !the convergence tolerance for Den Haan avg statistic

!control what is considered a change
changetol = 1e-10; !what is a change for the purposes of AC functions?

!how many parameters to pass to MATLAB
numconstants = 20; !this is relevant for input/output only in pass to MATLAB

!!!!END PARAMETER ENTRY BLOCK

!!!!SOME INITIALIZATIONS/GRID SETUPS

!do allocations for all variable-dimension arrays
allocate(k0(knum),l0(lnum),z0(znum),a0(anum),pr_mat_z(znum,znum,snum),pr_mat_a(anum,anum,snum),pr_mat_s(snum,snum),&
    sigmaagrid(snum),sigmazgrid(snum),MATLABconstants(numconstants),exog_key(numexog,4),exog_pos(numexog,4),endog_key(numendog,2),&
    endog_pos(numendog,2),V(numendog,numexog,numfcst),pr_mat(numexog,numexog),EVmat(numexog,numendog,numfcst),&
    loopind(numstates,3),polmat(numendog,numexog,numfcst),Vold(numendog,numexog,numfcst),polmatold(numendog,numexog,numfcst),&
    RHSvec(numendog),kprime_key(numendog,numexog,numfcst),lpol_key(numendog,numexog,numfcst),&
    kprime_pos(numendog,numexog,numfcst),lpol_pos(numendog,numexog,numfcst),asimpos(numper),ssimpos(numper),&
    distzkl(znum,numendog,numper),&
    asimshocks(numper),ssimshocks(numper),Ysim(numper),Ksim(numper),Lsim(numper),Isim(numper),Hsim(numper),ACksim(numper),&
    AClsim(numper),TOPksim(numper),BOTksim(numper),TOPlsim(numper),BOTlsim(numper),ADJksim(numper),ADJlsim(numper),psim(numper),&
    asimposIRF(lengthIRF,numsimIRF,2),ssimposIRF(lengthIRF,numsimIRF,2),asimshocksIRF(lengthIRF,numsimIRF),&
    ssimshocksIRF(lengthIRF,numsimIRF),&
    YsimIRF(lengthIRF,numsimIRF,2),KsimIRF(lengthIRF,numsimIRF,2),LsimIRF(lengthIRF,numsimIRF,2),IsimIRF(lengthIRF,numsimIRF,2),&
    HsimIRF(lengthIRF,numsimIRF,2),ACksimIRF(lengthIRF,numsimIRF,2),AClsimIRF(lengthIRF,numsimIRF,2),&
    ADJksimIRF(lengthIRF,numsimIRF,2),ADJlsimIRF(lengthIRF,numsimIRF,2),kbar0(kbarnum),&
    kbarfcstmat(anum,snum,snum,numfcst),pfcstmat(anum,snum,snum,numfcst),kfcstcoeff(anum,snum,snum,2),&
    pfcstcoeff(anum,snum,snum,2),&
    wfcstmat(anum,snum,snum,numfcst),Ymat(znum,anum,knum,lnum),Imat(knum,knum),ACkmat(znum,anum,knum,lnum,knum),&
    AClmat(numexog,knum,lnum,lnum,numfcst),WLmat(anum,snum,snum,numfcst,lnum),kbarfcstweights(anum,snum,snum,numfcst),&
    kbarfcstinds(anum,snum,snum,numfcst),EVmatp(znum,numendog),polmatp(znum,numendog),kprime_posp(znum,numendog),&
    lpol_posp(znum,numendog),Csim(numper),Kfcstsim(numper),pfcstsim(numper),kcoeffstore(anum,snum,snum,2,maxGEit),&
    pcoeffstore(anum,snum,snum,2,maxGEit),kfcstcoeffnew(anum,snum,snum,2),pfcstcoeffnew(anum,snum,snum,2),piterstoresim(numper),&
    kfcsterrorstore(maxGEit),pfcsterrorstore(maxGEit),psimIRF(lengthIRF,numsimIRF,2),CsimIRF(lengthIRF,numsimIRF,2),&
    KfcstsimIRF(lengthIRF,numsimIRF,2),&
    pfcstsimIRF(lengthIRF,numsimIRF,2),kfcsterrorstoreIRF(lengthIRF,numsimIRF,2),pfcsterrorstoreIRF(lengthIRF,numsimIRF,2),&
    piterstoresimIRF(lengthIRF,numsimIRF,2),&
    distzklbefore(znum,numendog),distzklafter(znum,numendog),pR2store(anum,snum,snum,maxGEit),KR2store(anum,snum,snum,maxGEit),&
    pRMSEstore(anum,snum,snum,maxGEit),KRMSEstore(anum,snum,snum,maxGEit),&
    pSSE(anum,snum,snum),KSSE(anum,snum,snum),pSST(anum,snum,snum),&
    KSST(anum,snum,snum),Kmean(anum,snum,snum),pmean(anum,snum,snum),KRMSEchangestore(maxGEit),pRMSEchangestore(maxGEit),&
    KR2changestore(maxGEit),pR2changestore(maxGEit),pDenHaanfcst(numper),KDenHaanfcst(numper),pmaxDenHaanstat(maxGEit),&
    KmaxDenHaanstat(maxGEit),pavgDenHaanstat(maxGEit),KavgDenHaanstat(maxGEit),pmaxDenHaanchangestore(maxGEit),&
    KmaxDenHaanchangestore(maxGEit),pavgDenHaanchangestore(maxGEit),KavgDenHaanchangestore(maxGEit),&
    seedarray(seeddim),p0(pnum),ep0(pnum),Cp0(pnum),Yp0(pnum),Ip0(pnum),ACkp0(pnum),AClp0(pnum),&
    Kbarprimep0(pnum),Hp0(pnum),Lp0(pnum),polmatp_interp(znum,numendog,pnum))

!set up the idiosyncratic capital and labor grids, using the subroutine linspace which replicates MATLAB function
call linspace(k0,log(kmin),log(kmax),knum); k0=exp(k0);
call linspace(l0,log(lmin),log(lmax),lnum); l0=exp(l0);

!set up the price interpolation grid
call linspace(p0,plb,pub,pnum); 

!set up the aggregate capital grid
call linspace(kbar0,log(kbarmin),log(kbarmax),kbarnum); kbar0 = exp(kbar0);

!set up the sigma or volatility grids and transition matrix
sigmazgrid = (/ sigmaz , zjump*sigmaz /)
sigmaagrid = (/ sigmaa , ajump*sigmaa /)
pr_mat_s(1,:) = (/1.0 - uncfreq , uncfreq /); pr_mat_s(1,:) = pr_mat_s(1,:)/sum(pr_mat_s(1,:))
pr_mat_s(2,:) = (/1.0 - uncpers , uncpers /); pr_mat_s(2,:) = pr_mat_s(2,:)/sum(pr_mat_s(2,:))

!initialize the forecast rules for capital
kfcstcoeff(1,1,1,:) = (/     3.4883701731259348E-002 , 0.97816323437145902   /)
kfcstcoeff(1,1,2,:) = (/    0.11341450543770845      , 0.93446845613105323   /)
kfcstcoeff(1,2,1,:) = (/     8.2659417468540872E-002 , 0.94738790975180087    /)
kfcstcoeff(1,2,2,:) = (/     7.3992650524926817E-002 , 0.95314542859155149   /)
kfcstcoeff(2,1,1,:) = (/     6.9801632965144189E-002 , 0.95924790092171608   /)
kfcstcoeff(2,1,2,:) = (/     9.6869854892919188E-002 , 0.94674737813173393   /)
kfcstcoeff(2,2,1,:) = (/    -1.5198200525120494E-002 ,  1.0053506204501794   /)
kfcstcoeff(2,2,2,:) = (/     8.5644505131487247E-002 , 0.94825073875865629   /)
kfcstcoeff(3,1,1,:) = (/     7.1059009742438894E-002 , 0.95929594015477915   /)
kfcstcoeff(3,1,2,:) = (/    0.10361003156135673      , 0.94309069158317294   /)
kfcstcoeff(3,2,1,:) = (/    -1.2114029083137908E-002 ,  1.0050701906398618   /)
kfcstcoeff(3,2,2,:) = (/    0.10498704814269222      , 0.93830231329181357   /)
kfcstcoeff(4,1,1,:) = (/     8.6154896028691474E-002 , 0.95174989403992605   /)
kfcstcoeff(4,1,2,:) = (/    0.35499918205199588      , 0.79834051525315464   /)
kfcstcoeff(4,2,1,:) = (/     4.6871268484734885E-002 , 0.97151876356118050   /)
kfcstcoeff(4,2,2,:) = (/    0.10668363503282495      , 0.93903726967049195   /)
kfcstcoeff(5,1,1,:) = (/     6.3721630815253547E-002 , 0.96599026561709067   /)
kfcstcoeff(5,1,2,:) = (/     7.8256162907873308E-002 , 0.96062100758935975   /)
kfcstcoeff(5,2,1,:) = (/    -8.2561448035710049E-003 ,  1.0037793345238009   /)
kfcstcoeff(5,2,2,:) = (/     8.1714645298625446E-002 , 0.95517304563342342   /)

!initialize the forecast rules for price
pfcstcoeff(1,1,1,:) = (/    1.1938054825462210   ,   -0.50103229978883246   /)
pfcstcoeff(1,1,2,:) = (/    1.2539181794799696   ,   -0.52458170928850767   /)
pfcstcoeff(1,2,1,:) = (/    1.0958154816600427   ,   -0.45185488867534263   /)
pfcstcoeff(1,2,2,:) = (/    1.0807687617215449   ,   -0.43451008966245575   /)
pfcstcoeff(2,1,1,:) = (/    1.2977278863560828   ,   -0.56909103073513168   /)
pfcstcoeff(2,1,2,:) = (/    1.3136993044767480   ,   -0.56705747235436854   /)
pfcstcoeff(2,2,1,:) = (/    1.1214975318855860   ,   -0.47263906994322891   /)
pfcstcoeff(2,2,2,:) = (/    1.1091163957775103   ,   -0.45734756585867525   /)
pfcstcoeff(3,1,1,:) = (/    1.2410049894958490   ,   -0.54408431222977482   /)
pfcstcoeff(3,1,2,:) = (/    1.2968225669181099   ,   -0.56494312729351348   /)
pfcstcoeff(3,2,1,:) = (/    1.1168451203505385   ,   -0.47623955463264794   /)
pfcstcoeff(3,2,2,:) = (/    1.0815198433981648   ,   -0.44827765862410585   /)
pfcstcoeff(4,1,1,:) = (/    1.2915939490831028   ,   -0.58073215986956239   /)
pfcstcoeff(4,1,2,:) = (/    1.2165452715207383   ,   -0.52524615952452980   /)
pfcstcoeff(4,2,1,:) = (/    1.0669218705112986   ,   -0.45458372986307682   /)
pfcstcoeff(4,2,2,:) = (/    1.0738210768799916   ,   -0.45157430828261091   /)
pfcstcoeff(5,1,1,:) = (/    1.1421631956150495   ,   -0.50194291828954540   /)
pfcstcoeff(5,1,2,:) = (/    1.2100632167950525   ,   -0.52923055893166149   /)
pfcstcoeff(5,2,1,:) = (/    1.0319133174822022   ,   -0.44052091599534227   /)
pfcstcoeff(5,2,2,:) = (/    1.0048177592223289   ,   -0.41745907631785711   /)
        
!initialize the forecast grids, noting that this can be made more precise later
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    
    !insert the forecasts into the fcst array
    do fcstct=1,numfcst
        kbarval = kbar0(fcstct); !what is the value of aggregate capital?
        kbarfcstval = exp( kfcstcoeff(act,sct,smin1ct,1) + kfcstcoeff(act,sct,smin1ct,2) * log( kbarval ) ); !implied agg cap fcst
        pfcstval = exp( pfcstcoeff(act,sct,smin1ct,1) + pfcstcoeff(act,sct,smin1ct,2) * log( kbarval ) ); !implied price fcst
        
        !adjust the agg capital forecast to the boundaries of the grid
        kbarfcstval = minval( (/ maxval( (/ kbarmin , kbarfcstval /) ) , kbarmax /) )
        
        !find the interval of kbar0 in which kbarfcst lands, as well as the lin. interp. weight
        ind = fcstct; call hunt(kbar0,numfcst,kbarfcstval,ind)
        weight = ( kbarfcstval - kbar0(ind) ) / ( kbar0(ind+1) - kbar0(ind) )
        
        !got to make sure that out of bounds entries are handled correctly
        if (ind>=1.and.ind<kbarnum) then
            kbarfcstinds(act,sct,smin1ct,fcstct) = ind; kbarfcstweights(act,sct,smin1ct,fcstct) = weight
        else if (ind==0) then
            kbarfcstinds(act,sct,smin1ct,fcstct) = 1; kbarfcstweights(act,sct,smin1ct,fcstct) = 0.0
        else if (ind==kbarnum) then
            kbarfcstinds(act,sct,smin1ct,fcstct) = kbarnum-1; kbarfcstweights(act,sct,smin1ct,fcstct) = 1.0
        end if
        
        !insertion into arrays, including insertion of the implied wage
        kbarfcstmat(act,sct,smin1ct,fcstct) = kbarfcstval
        pfcstmat(act,sct,smin1ct,fcstct) = pfcstval
        wfcstmat(act,sct,smin1ct,fcstct) = theta / pfcstval
        
    end do !fcstct
end do !smin1ct
end do !sct
end do !act

!compute the idio & aggr transition matrices, via Tauchen (1986) adapted to stochastical vol case
call unctauchen(pr_mat_z,z0,znum,zmin,zmax,snum,sigmazgrid,rhoz)
call unctauchen(pr_mat_a,a0,anum,amin,amax,snum,sigmaagrid,rhoa)
 
!index the exogenous variables (z,a,s) & create the unified transition matrix over exogenous variables
ct=0
do zct=1,znum; do act=1,anum; do sct=1,snum; do smin1ct=1,snum;
    ct=ct+1;
    
    !indexing the variables
    exog_pos(ct,1) = zct; exog_key(ct,1) = z0(zct); !insert z
    exog_pos(ct,2) = act; exog_key(ct,2) = a0(act); !insert a
    exog_pos(ct,3) = sct; exog_key(ct,3) = dble(sct); !insert s
    exog_pos(ct,4) = smin1ct; exog_key(ct,4) = dble(smin1ct); !insert s_{-1}
    
    !creating the transition matrix
    do zprimect=1,znum; do aprimect=1,anum; do sprimect=1,snum
        !note that this only indexes numexog/snum of the next period's values, since sct today is smin1ct tomorrow and is FIXED!
        primect=(zprimect-1)*anum*snum*snum + (aprimect-1)*snum*snum + (sprimect-1)*snum + sct
        pr_mat(ct,primect) = pr_mat_z(zct,zprimect,sct) * pr_mat_a(act,aprimect,sct) * pr_mat_s(sct,sprimect)
    end do; end do; end do;
    
    !correction for roundoff error
    pr_mat(ct,:) = pr_mat(ct,:) / sum( pr_mat(ct,:) )
    
end do; end do; end do; end do;

!call the random numbers for unconditional simulation - occurs outside of fcst loop
!this werid block seeds the draws
do ct=1,seeddim
    seedarray(ct) = seedint+ct
end do !ct
call random_seed(put=seedarray)
call random_number(asimshocks); !U(0,1) in each cell of asimshocks
call random_number(ssimshocks); !U(0,1) in each cell of ssimshocks


!write the random numbers
open(8,file="asimshocks.txt")
do ct=1,numper
write(8,*) asimshocks(ct)
end do 
close(8)

open(8,file="ssimshocks.txt")
do ct=1,numper
write(8,*) ssimshocks(ct)
end do 
close(8)

!based on the random shocks drawn, simulate the aggregate exogenous processes, occurs outside of fcst loop
call uncexogsim(ssimshocks,asimshocks,ssimpos,asimpos,pr_mat_s,pr_mat_a,ainit,sinit,snum,anum,numper)

!if flag is set, call the shocks for the IRF simulation and perform the simulation

if (doIRF==1) then
    !first, call the random draws for unc and agg prod simulation in the IRF
    call random_number(asimshocksIRF); !U(0,1) in each cell of asimshocksIRF
    call random_number(ssimshocksIRF); !U(0,1) in each cell of ssimshocksIRF
        
    !then, based on these shocks, simulate the IRF unc and agg prod processes
    call uncexogsimIRF(ssimshocksIRF,asimshocksIRF,ssimposIRF,asimposIRF,pr_mat_a,pr_mat_s,ainit,&
        lengthIRF,numsimIRF,shockperIRF,shocklengthIRF,anum,snum,sinit,singleshock)
end if !doIRF

!index the endog variables (k,l_{-1}), noting that the index is used for states and policies
ct=0
do kct=1,knum; do lmin1ct=1,lnum
    ct=ct+1
    endog_pos(ct,1) = kct; endog_key(ct,1) = k0(kct); !insert k
    endog_pos(ct,2) = lmin1ct; endog_key(ct,2) = l0(lmin1ct); !insert l_{-1}
end do; end do;

!indexing for parallelization over V: parallel loops will go over "ct," with
!each thread pulling out endogct, exogct, and fcstct
ct=0
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
    ct=ct+1
    loopind(ct,1) = endogct; loopind(ct,2) = exogct; loopind(ct,3) = fcstct
end do !numfcst
end do !exogct
end do !endogct

!determine and write constants
MATLABconstants = (/ dble(znum),dble(knum),dble(lnum),dble(anum),dble(snum),&
    dble(kbarnum),dble(numper),dble(numdiscard),dble(numendog),dble(numexog),dble(numfcst),&
    dble(numstates),deltak,deltan,dble(numsimIRF),dble(lengthIRF),dble(shockperIRF),dble(numdiscIRF),&
    dble(shocklengthIRF),dble(singleshock)/)

open(8,file="MATLABconstants.txt")
do ct=1,numconstants
write(8,*) MATLABconstants(ct)
end do !ct
close(8)

!write labor grid
open(8,file="l0.txt")
do ct=1,lnum
write(8,*) l0(ct)
end do
close(8)

!write idio capital grid
open(8,file="k0.txt")
do ct=1,knum
write(8,*) k0(ct)
end do
close(8)

!write aggregate capital grid
open(8,file="kbar0.txt")
do ct=1,kbarnum
write(8,*) kbar0(ct)
end do
close(8)

!write idio prod grid
open(8,file="z0.txt")
do ct=1,znum
write(8,*) z0(ct)
end do
close(8)

!write agg prod grid
open(8,file="a0.txt")
do ct=1,anum
write(8,*) a0(ct)
end do
close(8)

!write endogenous variables key
open(8,file="endog_key.txt")
do ct=1,numendog
write(8,*) endog_key(ct,:)
end do !ct
close(8)

!write exogenous variables key
open(8,file="exog_key.txt")
do ct=1,numexog
write(8,*) exog_key(ct,:)
end do !ct
close(8)

!write endogenous variables pos key
open(8,file="endog_pos.txt")
do ct=1,numendog
write(8,*) endog_pos(ct,:)
end do !ct
close(8)

!write exogenous variables pos key
open(8,file="exog_pos.txt")
do ct=1,numexog
write(8,*) exog_pos(ct,:)
end do !ct
close(8)

!write agg prod series
open(8,file="asimpos.txt")
do t=1,numper
write(8,*) asimpos(t)
end do !t
close(8)

!write unc series
open(8,file="ssimpos.txt")
do t=1,numper
write(8,*) ssimpos(t)
end do !t
close(8)

if (doIRF==1) then
    !write agg prod series
    open(8,file="asimposIRF.txt")
    do ct=1,2
    do simct=1,numsimIRF
    do t=1,lengthIRF
    write(8,*) asimposIRF(t,simct,ct)
    end do !t
    end do !simct
    end do !ct
    close(8)

    !write unc series
    open(8,file="ssimposIRF.txt")
    do ct=1,2
    do simct=1,numsimIRF
    do t=1,lengthIRF
    write(8,*) ssimposIRF(t,simct,ct)
    end do !t
    end do !simct
    end do !ct
    close(8)
end if !doIRF

!!!!END INITIALIZATIONS/GRID SETUPS

do GEct=1,maxGEit; !GEct - THIS IS THE INDEX OVER THE NUMBER OF FCST RULE ITERATIONS

!!!!!VF ITERATION BLOCK

write(*,*) " "
write(13,*) " "


!now, create all of the matrices useful later for setting up RHS returns

!output Ymat(z,a,k,l)
!$omp parallel private(zct,act,kct,lct,zval,aval,kval,lval)
!$omp do collapse(4)
do zct=1,znum
do act=1,anum
do kct=1,knum
do lct=1,lnum
    zval = z0(zct); aval=a0(act); kval = k0(kct); lval=l0(lct)
    Ymat(zct,act,kct,lct) = y(zval,aval,kval,lval,alpha,nu)
end do; !lct
end do; !kct
end do; !act
end do; !zct
!$omp end do nowait
!$omp end parallel

!investment Imat(k,k')
do kct=1,knum
do kprimect=1,knum
    kval = k0(kct); kprimeval = k0(kprimect)
    Imat(kct,kprimect) = kprimeval - (1.0-deltak) * kval
end do !kprimect
end do !kct
    
!capital AC ACkmat(z,a,k,l,k')
!$omp parallel private(zct,act,kct,lct,kprimect,zval,aval,kval,lval,kprimeval)
!$omp do collapse(5)
do zct=1,znum
do act=1,anum
do kct=1,knum
do lct=1,lnum
do kprimect=1,knum
    zval = z0(zct); aval=a0(act); kval = k0(kct); lval = l0(lct); kprimeval = k0(kprimect)
    ACkmat(zct,act,kct,lct,kprimect) = &
        ACk(zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak)
end do !kprimect
end do !lct
end do !kct
end do !act
end do !zct
!$omp end do nowait
!$omp end parallel

!labor AC AClmat(exogct,k,l,l_{-1},K)
!$omp parallel private(zct,act,kct,lct,lmin1ct,sct,fcstct,zval,aval,kval,lval,&
!$omp& lmin1val,wfcstval,smin1ct,exogct)
!$omp do collapse(5)
do exogct=1,numexog
do kct=1,knum
do lct=1,lnum
do lmin1ct=1,lnum
do fcstct=1,numfcst
    zct = exog_pos(exogct,1);
    act = exog_pos(exogct,2);
    sct = exog_pos(exogct,3);
    smin1ct = exog_pos(exogct,4);
    
    zval = z0(zct); aval=a0(act); kval = k0(kct); lval = l0(lct); lmin1val = l0(lmin1ct);
    wfcstval = wfcstmat(act,sct,smin1ct,fcstct);
    AClmat(exogct,kct,lct,lmin1ct,fcstct) = &
        ACl(zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wfcstval)
        
end do !fcstct
end do !lmin1ct
end do !lct
end do !kct
end do !exogct
!$omp end do nowait
!$omp end parallel

!wage bill WLmat(a,s,s_{-1},K,l)
!$omp parallel private(act,sct,smin1ct,fcstct,lct)
!$omp do collapse(5)
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
do fcstct=1,numfcst
do lct=1,lnum
    WLmat(act,sct,smin1ct,fcstct,lct) = wfcstmat(act,sct,smin1ct,fcstct) * l0(lct)
end do !lct
end do !fcstct
end do !smin1ct
end do !sct
end do !act
!$omp end do
!$omp end parallel

write(*,"(A,F15.1,A)") "Finished setup of return matrices at ",omp_get_wtime()-start," seconds"
write(13,"(A,F15.1,A)") "Finished setup of return matrices at ",omp_get_wtime()-start," seconds"

!if you want to do VFI, just read the policies/VF in from files
if (doVFI==1) then
    
write(*,*) "Doing VFI."
write(13,*) "Doing VFI."

!initialize the VF and policies
if (GEct==1) then 
    !in this case, initialize using a stupid guess
    Vold(:,:,:) = 0.0;
    V(:,:,:) = 0.0
    polmatold(:,:,:) = numendog/2
    polmat(:,:,:) = 0
    EVmat(:,:,:) = 0.0
else if (GEct>1) then
    !in this case, initialize using last GE iteration
    Vold = V
    polmatold = polmat
    polmat(:,:,:) = 0
    EVmat(:,:,:) = 0.0
end if

!this loop is the "VFI loop" itself, although technically this is policy iteration
do vfct=1,vfmaxit
    
    !!!Howard acceleration step
    do accelct=1,accelmaxit
        
        !set up the parallelization for the Howard acceleration construction
        !$omp parallel private(ct,endogct,exogct,fcstct,zct,act,sct,kct,lmin1ct,polstar,&
        !$omp& kprimect,lct,ind,weight,Vnextval,smin1ct)
        !$omp do
        do ct=1,numstates
            
            !extract states from loop index matrix
            endogct=loopind(ct,1); exogct=loopind(ct,2); fcstct=loopind(ct,3);
            
            !extract positions associated with these states
            zct = exog_pos(exogct,1); act = exog_pos(exogct,2); sct = exog_pos(exogct,3);
            smin1ct=exog_pos(exogct,4)
            kct = endog_pos(endogct,1); lmin1ct= endog_pos(endogct,2); 
            
            !extract policy from polmatold
            polstar = polmatold(endogct,exogct,fcstct)
            
            !extract positions associated with the policies
            kprimect = endog_pos(polstar,1); lct = endog_pos(polstar,2)

            !initialize the RHS by forming the current period return
            V(endogct,exogct,fcstct) = pfcstmat(act,sct,smin1ct,fcstct) * ( Ymat(zct,act,kct,lct) &
                - ACkmat(zct,act,kct,lct,kprimect) - AClmat(exogct,kct,lct,lmin1ct,fcstct) &
                - Imat(kct,kprimect) - WLmat(act,sct,smin1ct,fcstct,lct) )
            !write(*,*) "V(endogct,exogct,fcstct) = ",V(endogct,exogct,fcstct)             
            !find interpolation interval and weights for fcst agg capital
            ind = kbarfcstinds(act,sct,smin1ct,fcstct)
            weight = kbarfcstweights(act,sct,smin1ct,fcstct)
            !write(*,*) "ind = ",ind
            !write(*,*) "weight = ",weight
            !add discounted expected continuation value
            do exogprimect=1,numexog
              
                !what is the linearly interpolated continuation value?
                Vnextval = weight * Vold(polstar,exogprimect,ind+1) + &
                    (1.0 - weight)*Vold(polstar,exogprimect,ind)
              
                !now, add the discounted expected value associated with this realization
                V(endogct,exogct,fcstct) = V(endogct,exogct,fcstct) + &
                    beta * pr_mat(exogct,exogprimect) * Vnextval
                
            end do !exogprimect
        !write(*,*) "made it here"
        !write(*,*) "V(endogct,exogct,fcstct) = ",V(endogct,exogct,fcstct)
        !write(*,*) "Vnextval = ",Vnextval
        
        end do !ct
        !$omp end do nowait
        !$omp end parallel
        
        !initialize for the next round of the Howard VFI
        Vold = V

    end do !accelct
    
    
    !!!create EVmat for RHS optimization
    
    !set up the parallelization for the creation of the continuation value
    !$omp parallel private(ct,polct,exogct,fcstct,act,sct,ind,weight,exogprimect,&
    !$omp& Vnextval,smin1ct)
    !$omp do
    do ct=1,numstates
        
        !extract exogenous and policy states
        polct=loopind(ct,1); exogct=loopind(ct,2); fcstct = loopind(ct,3)
        
        !what are act and sct?
        act = exog_pos(exogct,2); sct = exog_pos(exogct,3); smin1ct = exog_pos(exogct,4)
        
        !find interpolation interval and weights for fcst agg capital
        ind = kbarfcstinds(act,sct,smin1ct,fcstct)
        weight = kbarfcstweights(act,sct,smin1ct,fcstct)
              
        !create expected value for the next period based on policy, exogenous transitions, and forecast kbar
        EVmat(exogct,polct,fcstct) = 0.0
        do exogprimect=1,numexog; !loop over exogenous realizations next period
            
            !what is the linearly interpolated continuation value?
            Vnextval = weight * Vold(polct,exogprimect,ind+1) + &
                    (1.0 - weight)*Vold(polct,exogprimect,ind)

            !add the weighted continuation value to the next period
            EVmat(exogct,polct,fcstct) = EVmat(exogct,polct,fcstct) + pr_mat(exogct,exogprimect) * Vnextval
        end do !exogprimect
        
    end do !ct
    !$omp end do nowait
    !$omp end parallel

    !!!RHS optimization step
    
    !set up the parallelization for the optimization of the Bellman RHS
    !$omp parallel private(ct,endogct,exogct,fcstct,zct,act,sct,kct,lmin1ct,&
    !$omp& polct,kprimect,lct,RHSvec,polstar,smin1ct) 
    !$omp do
    do ct=1,numstates
        
       !extract states from loop index matrix
        endogct=loopind(ct,1); exogct=loopind(ct,2); fcstct=loopind(ct,3);
        
        !extract positions associated with these states
        zct = exog_pos(exogct,1); act = exog_pos(exogct,2); sct = exog_pos(exogct,3);
        smin1ct=exog_pos(exogct,4)
        kct = endog_pos(endogct,1); lmin1ct= endog_pos(endogct,2); 
        
        !create RHS vector by looping over policies
        do polct=1,numendog
            
            !extract positions associated with the policies
            kprimect = endog_pos(polct,1); lct = endog_pos(polct,2)

            !initialize the RHS by forming the current period return
            RHSvec(polct) = pfcstmat(act,sct,smin1ct,fcstct) * ( Ymat(zct,act,kct,lct) &
                - ACkmat(zct,act,kct,lct,kprimect) - AClmat(exogct,kct,lct,lmin1ct,fcstct) &
                - Imat(kct,kprimect) - WLmat(act,sct,smin1ct,fcstct,lct) )
              
            !actually form the RHS by adding continuation value
            RHSvec(polct) = RHSvec(polct) + beta * EVmat(exogct,polct,fcstct)
            
        end do !polct
        
        !extract policies via selection of max payoff
        polstar = maxloc(RHSvec,1)
        polmat(endogct,exogct,fcstct) = polstar
        V(endogct,exogct,fcstct) = RHSvec(polstar)
        
    end do !ct
    !$omp end do nowait
    !$omp end parallel
    
    !compute errors and output the info
    vferror = maxval(abs(V-Vold))
    polerror = maxval(abs(polmat-polmatold))
    
    write(13,"(A,I3,A,F15.5,A)") "VF iter = ",vfct," in ",omp_get_wtime()-start," seconds"
    write(13,"(A,I3,A,F15.5)") "VF iter = ",vfct,", VF error = ",vferror
    write(13,"(A,I3,A,F15.5)") "VF iter = ",vfct,", policy error = ",polerror
    write(13,*) " "
    
    write(*,"(A,I3,A,F15.5,A)") "VF iter = ",vfct," in ",omp_get_wtime()-start," seconds"
    write(*,"(A,I3,A,F15.5)") "VF iter = ",vfct,", VF error = ",vferror
    write(*,"(A,I3,A,F15.5)") "VF iter = ",vfct,", policy error = ",polerror
    write(*,*) " "
    
    !exit criterion, then initialize for the next round of VFI
    !if ((vferror<vferrortol .and. polerror<vferrortol).or.(vferror<vferrortol)) exit
    if (polerror<vferrortol) exit
    Vold = V
    polmatold = polmat
    
end do !vfmaxit

!convert polmat to policies for capital and labor
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
    !convert policies to actual capital and labor values
    kprime_key(endogct,exogct,fcstct) = endog_key(polmat(endogct,exogct,fcstct),1)
    kprime_pos(endogct,exogct,fcstct) = endog_pos(polmat(endogct,exogct,fcstct),1)
    
    !convert policies to capital and labor integer grid points
    lpol_key(endogct,exogct,fcstct) = endog_key(polmat(endogct,exogct,fcstct),2)
    lpol_pos(endogct,exogct,fcstct) = endog_pos(polmat(endogct,exogct,fcstct),2)
end do !fcstct
end do !exogct
end do !endogct

!display warning if hit the top of the grid or the bottom of the grid
!for either policy variable

write(*,*) " "
write(13,*) " "

write(*,*) "Hit the top or bottom of grid in theory?"
write(13,*) "Hit the top or bottom of grid in theory?"

if (maxval(kprime_pos)==knum) then 
    write(*,*) "Hit top of capital grid."    
    write(13,*) "Hit top of capital grid."
end if

if (maxval(lpol_pos)==lnum) then 
    write(*,*) "Hit top of labor grid."    
    write(13,*) "Hit top of labor grid."
end if

if (minval(kprime_pos)==1) then 
    write(*,*) "Hit bottom of capital grid."    
    write(13,*) "Hit bottom of capital grid."
end if

if (minval(lpol_pos)==1) then 
    write(*,*) "Hit bottom of labor grid."    
    write(13,*) "Hit bottom of labor grid."
end if

!if you don't want to do VFI, just read the policies/VF in from files
else if (doVFI==0) then

write(*,*) "Not doing VFI: reading value function and policies."
write(13,*) "Not doing VFI: reading value function and policies."

!read VF from file
open(8,file="V.txt")
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
read(8,*) V(endogct,exogct,fcstct)
end do !fcstct
end do !exogct
end do !endogct
close(8)

!read policy matrix from file
open(8,file="polmat.txt")
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
read(8,*) polmat(endogct,exogct,fcstct)
end do !fcstct
end do !exogct
end do !endogct
close(8)

!convert polmat to policies for capital and labor
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
    !convert policies to actual capital and labor values
    kprime_key(endogct,exogct,fcstct) = endog_key(polmat(endogct,exogct,fcstct),1)
    kprime_pos(endogct,exogct,fcstct) = endog_pos(polmat(endogct,exogct,fcstct),1)
    
    !convert policies to capital and labor integer grid points
    lpol_key(endogct,exogct,fcstct) = endog_key(polmat(endogct,exogct,fcstct),2)
    lpol_pos(endogct,exogct,fcstct) = endog_pos(polmat(endogct,exogct,fcstct),2)
end do !fcstct
end do !exogct
end do !endogct

end if !doVFI

write(*,*) " "
write(13,*) " "

!!!!!END VF ITERATION BLOCK

!!!!!SIMULATION BLOCK

write(*,*) "Doing unconditional simulation."
write(13,*) "Doing unconditional simulation."

!first, initialize the distribution over endogenous variables
distzkl(:,:,:) = 0.0; !set everything to zero

if (distinit==1) then
    
    !if you read the initial distribution in
    open(8,file="distzklinit.txt")
    do zct=1,znum
    do endogct = 1,numendog
        read(8,*) distzkl(zct,endogct,1)    
    end do !endogct
    end do !zct
    close(8)
    
else 

  
    !chosen point to center the mass
    kct = 50
    lmin1ct=20
    endogct = (kct-1)*lnum + lmin1ct
    
    distzkl(:,(endogct-5):(endogct+5),1) = 1.0

end if !distinit

!now, round the dist
distzkl(:,:,1) = distzkl(:,:,1) / sum(distzkl(:,:,1))

!then, initialize the aggregate series to 0
Ysim(:) = 0.0; Ksim = Ysim; Lsim = Ysim; Isim = Ysim; Hsim = Ysim; ACksim = Ysim; AClsim = Ysim;
TOPksim = Ysim; BOTksim = Ysim; TOPlsim = Ysim; BOTlsim = Ysim; ADJksim = Ysim; ADJlsim = Ysim;
Csim = Ysim; psim =Ysim;

!start with aggregate capital guessed at some reasonable value
Ksim(1) = (kbarmin + kbarmax)/dble(2.0)

write(*,"(A,F15.1,A)") "Starting actual sim at ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Starting actual sim at ",omp_get_wtime()-start," seconds."

!now, loop over periods in the simulation, tracking the endogenous
!movement of weight around the grid
do t=1,numper-1
    
    !aggregate states
    if (t>1) then
        !in this case you don't get a segmentation fault when looking for s_{-1} index
        act = asimpos(t); sct=ssimpos(t); smin1ct = ssimpos(t-1)
    else if (t==1) then
        !initialize the very first one to low unc previously
        act = asimpos(t); sct=ssimpos(t); smin1ct = 1;
    end if
    aval = a0(act)
    
    !what is the next period forecast capital, in grid boundaries? index and weight?
    kbarfcstval = exp( kfcstcoeff(act,sct,smin1ct,1) + kfcstcoeff(act,sct,smin1ct,2) * log( Ksim(t) ) );
    kbarfcstval = minval( (/ maxval( (/ kbarmin , kbarfcstval /) ) , kbarmax /) )   
    ind = kbarnum/2; call hunt(kbar0,numfcst,kbarfcstval,ind)
    weight = ( kbarfcstval - kbar0(ind) ) / ( kbar0(ind+1) - kbar0(ind) )
    
    !guard against off grid point values
    if (ind==kbarnum) then 
        ind = kbarnum-1;
        weight = 1.0;
    else if (ind==0) then
        ind = 1;
        weight = 0.0;
    end if
    
    Kfcstsim(t+1) = kbarfcstval
    
    !what is the current period forecast price?
    pfcstsim(t) = exp( pfcstcoeff(act,sct,smin1ct,1) + pfcstcoeff(act,sct,smin1ct,2) * log( Ksim(t) ) );
    
    
    !note that EVmat doesn't depend on p, so it can be pre-computed now, as EVmatp
    
    !set up the parallelization for the creation of the continuation value
    !$omp parallel private(zct,polct,exogct,exogprimect,Vnextval)
    !$omp do collapse(2)
    do zct=1,znum
    do polct=1,numendog
        
        !what is the implied value of exogct, given simulated aggregate states?
        exogct = (zct-1)*anum*snum*snum + (act-1)*snum*snum + (sct-1)*snum  + smin1ct
              
        !create expected value for the next period based on policy, exogenous transitions,
        !and forecast capital
        EVmatp(zct,polct) = 0.0
        do exogprimect=1,numexog; !loop over exogenous realizations next period
            
            !what is the linearly interpolated continuation value?
            Vnextval = weight * V(polct,exogprimect,ind+1) + &
                    (1.0 - weight)*V(polct,exogprimect,ind)

            !add the weighted continuation value to the next period
            EVmatp(zct,polct) = EVmatp(zct,polct) + pr_mat(exogct,exogprimect) * Vnextval
            
        end do !exogprimect
    
    end do !zct
    end do !endogct
    !$omp end do nowait
    !$omp end parallel

	!set up the interpolation of the excess demand function e(p) = 1/p - C(p)
	ep0(:) = 0.0
	Cp0(:) = 0.0
	Yp0(:) = 0.0
	Ip0(:) = 0.0
	ACkp0(:) = 0.0
	AClp0(:) = 0.0
	Kbarprimep0(:) = 0.0
	Hp0(:) = 0.0
	Lp0(:) = 0.0
	polmatp_interp(:,:,:) = 0
	
	do piter = 1,pnum
        !initialize the price-dependent policies to 0
        polmatp(:,:) = 0; kprime_posp(:,:) = 0; lpol_posp(:,:) = 0
		
		pval = p0(piter) !loop over the vector of pre-stored p values
	
		wval = theta / pval; !what is the wage implied by this value of p?
        
        !this block computes the policies given the current pval & wval at each
        !point (z,kl_{-1})
        
        Yvalp = 0.0; 
        Ivalp = 0.0; 
        ACkvalp = 0.0; 
        AClvalp = 0.0; 
        Kbarprimevalp = 0.0; 
        Hvalp = 0.0;
        Lvalp = 0.0
        
        !$omp parallel private(zct,endogct,exogct,kct,lmin1ct,RHSvec,polct,kprimect,&
        !$omp& lct,polstar) reduction(+:Yvalp,Ivalp,ACkvalp,AClvalp,Kbarprimevalp,Hvalp,&
        !$omp& Lvalp)
        !$omp do collapse(2)
        do zct=1,znum
        do endogct=1,numendog
            !restriction on which policies to compute - important for time
            if (distzkl(zct,endogct,t)>disttol) then                        
            !extract states from loop indexes
            exogct = (zct-1)*anum*snum*snum + (act-1)*snum*snum + (sct-1)*snum  + smin1ct
            
            !extract positions associated with these states
            kct = endog_pos(endogct,1);
            lmin1ct= endog_pos(endogct,2); 
            
            !create RHS vector by looping over policies
            RHSvec(:) = 0.0
            do polct=1,numendog
                
                !extract positions associated with the policies
                kprimect = endog_pos(polct,1); lct = endog_pos(polct,2)

                !initialize the RHS by forming the current period return
                RHSvec(polct) = pval * ( Ymat(zct,act,kct,lct) &
                    - ACkmat(zct,act,kct,lct,kprimect) &
                    - Imat(kct,kprimect) - wval * l0(lct) &
                    - ACl(z0(zct),a0(act),k0(kct),l0(lct),alpha,nu,l0(lmin1ct),hirelin,firelin,&
                    labfix,deltan,wval))
                  
                !actually form the RHS by adding continuation value
                RHSvec(polct) = RHSvec(polct) + beta * EVmatp(zct,polct)
               
            end do !polct
                
            !what is the new optimum value in index form and in capital and labor indexes?
            polmatp(zct,endogct) =  maxloc(RHSvec,1)
            kprime_posp(zct,endogct) = endog_pos(polmatp(zct,endogct),1)
            lpol_posp(zct,endogct) = endog_pos(polmatp(zct,endogct),2)
            
            !what are the investment and labor for this position?
            lct = lpol_posp(zct,endogct); !current labor, policy
            kprimect = kprime_posp(zct,endogct); !next period capital, policy
            
            !what are the aggregates?
            Yvalp = Yvalp + distzkl(zct,endogct,t) * Ymat(zct,act,kct,lct)
            Ivalp = Ivalp + distzkl(zct,endogct,t) * Imat(kct,kprimect)
            ACkvalp = ACkvalp + distzkl(zct,endogct,t) * ACkmat(zct,act,kct,lct,kprimect)
            AClvalp = AClvalp + distzkl(zct,endogct,t) * ACl(z0(zct),a0(act),k0(kct),&
                l0(lct),alpha,nu,l0(lmin1ct),hirelin,firelin,labfix,deltan,wval)
            Kbarprimevalp = Kbarprimevalp + distzkl(zct,endogct,t) * k0(kprimect)
            Hvalp = Hvalp + distzkl(zct,endogct,t) * ( l0(lct) - (1.0-deltan) * l0(lmin1ct))
            Lvalp = Lvalp + distzkl(zct,endogct,t) * l0(lct)
        
            end if !disttol
        
        end do !endogct
        end do !zct
        !$omp end do nowait
        !$omp end parallel
        
        !what is implied consumption?
        Cvalp = Yvalp - Ivalp - ACkvalp - AClvalp
        
        !insert implied value of excess demand, and all other values into storage vectors
        ep0(piter) = 1.0/pval - Cvalp
        Cp0(piter) = Cvalp
       	Yp0(piter) = Yvalp
		Ip0(piter) = Ivalp
		ACkp0(piter) = ACkvalp
		AClp0(piter) = AClvalp
		Kbarprimep0(piter) = Kbarprimevalp
		Hp0(piter) = Hvalp
		Lp0(piter) = Lvalp
		polmatp_interp(:,:,piter) = polmatp
		
		!write(*,*) "piter = ",piter," pval = ",pval," e(pval) = ",ep0(piter), "C(pval) = ",Cp0(piter)
	
	end do !piter


    !set up the boundaries of the bisection
    pvala = plb
    pvalb = pub
    pvalc = pvala + dble(0.67) * (pvalb-pvala) 

    !iterate over the market-clearing value of p, for each value
    !redoing the optimization of the RHS of the Bellman equation
    !and recomputing policies
    do piter=1,maxpit
        !initialize the price-dependent policies to 0
        polmatp(:,:) = 0; kprime_posp(:,:) = 0; lpol_posp(:,:) = 0
        
        !brent setup
        if (piter==1) pval = pvala
        if (piter==2) pval = pvalb
        if (piter==3) pval = pvalc
        
        !brent restart
        if (piter==pcutoff) then 
            pvala = pfcstsim(t)-4.0*pwindow;
            pval = pvala;
        else if (piter==pcutoff+1) then 
            pvalb =  pfcstsim(t)+4.0*pwindow;
            pval = pvalb
        else if (piter==pcutoff+2) then
            pvalc = pvala + dble(0.67) * (pvalb-pvala) 
            pval = pvalc
        end if
        
        !if not restarting or initializing
        if ((piter>3.and.piter<pcutoff).or.(piter>pcutoff+2)) then 
            
        
            !first, try inverse quadratic interpolation of the excess demand function
            pval = ( pvala * fb * fc ) / ( (fa - fb) * (fa - fc) ) &
                + ( pvalb * fa * fc ) / ( (fb - fa) * (fb - fc ) ) &
                + ( pvalc * fa * fb ) / ( (fc - fa) * (fc - fb ) )
       
            !if it lies within bounds, and isn't too close to the bounds, then done
            
            !o/w, take bisection step
            if ((minval( (/ abs(pvala - pval), abs(pvalb-pval) /) )<&
                    abs( (pvalb-pvala)/dble(9.0) ) ).or.(pval<pvala).or.(pval>pvalb))   then
                pval = (pvala + pvalb) / dble(2.0)
         
            end if
            
        end if 
        
        
		!!!actually evaluate consumption approximation function
		pval = minval( (/ maxval( (/ p0(1) , pval /) ) , p0(pnum) /) )   
	    pind = pnum/2; call hunt(p0,pnum,pval,pind)
	    pwgt = (pval - p0(pind)) / (p0(pind + 1) - p0(pind))
	    
	    !guard against off grid point values
    	if (pind==pnum) then 
        	pind = pnum-1;
	        pwgt = 1.0;
    	else if (pind==0) then
        	pind = 1;
	        pwgt = 0.0;
    	end if
		
		!actually perform linear interpolation of the consumption function
		Cvalp = Cp0(pind)*(1.0-pwgt) + Cp0(pind+1)*pwgt
		        
        !are you initializing the brent?
        if (piter==1) fa = (1/pval) - Cvalp
        if (piter==2) fb = (1/pval) - Cvalp
        if (piter==3) fc = (1/pval) - Cvalp
        
        !are you restarting the brent?
        if (piter==pcutoff) fa = (1/pval) - Cvalp
        if (piter==pcutoff+1) fb = (1/pval) - Cvalp
        if (piter==pcutoff+2) fc = (1/pval) - Cvalp
        
        !what is the error given by this implied consumption?
        perror = (1/pval) - Cvalp
        
        !if not restarting or initializing
        if ((piter>3.and.piter<pcutoff).or.(piter>pcutoff+2)) then 
            if (perror<0) then
                pvalc = pvalb; fc = fb;
                pvalb = pval; fb = perror;
                !pval a doesn't change
            else if (perror>=0) then
                pvalc = pvala; fc = fa;
                pvala = pval; fa = perror;
                !pval b doesn't change
            end if
        end if
      
        !exit criterion for market-clearing
        perror = log(pval*Cvalp)
        if (abs(perror)<perrortol.and.piter>2) exit
      
        end do !piter
       
    !record number of price iterations
    piterstoresim(t) = piter-1
    
    !output price stats on certain periods
!    if (mod(t,50)==1) then 
        write(*,"(A,I5,A,I5,A,F10.4,A,F10.4)") "t = ",t,", piter = ",piter-1,&
        ", pval = ",pval,", perror = ",perror
        
        write(13,"(A,I5,A,I5,A,F10.4,A,F10.4)") "t = ",t,", piter = ",piter-1,&
        ", pval = ",pval,", perror = ",perror
        
 !   end if
    
    !insert market-clearing price and other linearly interpolated stuff into sim series
    psim(t) = pval; !this is the most recently run p from the clearing algorithm
    Csim(t) = Cvalp; !this is already the linearly interpolated consumption C
    Ysim(t) = Yp0(pind)*(1.0-pwgt) + Yp0(pind+1)*pwgt; !linearly interpolated output Y
    Isim(t) = Ip0(pind)*(1.0-pwgt) + Ip0(pind+1)*pwgt; !linearly interpolated investment I
    ACksim(t) = ACkp0(pind)*(1.0-pwgt) + ACkp0(pind+1)*pwgt; !linearly interpolated ACk 
    AClsim(t) = AClp0(pind)*(1.0-pwgt) + AClp0(pind+1)*pwgt; !linearly interpolated ACl
    Ksim(t+1) = Kbarprimep0(pind)*(1.0-pwgt) + Kbarprimep0(pind+1)*pwgt; !linearly interpolated K' 
    Hsim(t) = Hp0(pind)*(1.0-pwgt) + Hp0(pind+1)*pwgt; !linearly interpolated hiring H
    Lsim(t) = Lp0(pind)*(1.0-pwgt) + Lp0(pind+1)*pwgt; !linearly interpolated labor input L
    
    !now that the market-clearing price is determined, move on to insert weight into the next period, according to the 
    !linearly interpolated rule
    
    do zct=1,znum
    do endogct=1,numendog
        if (distzkl(zct,endogct,t)>disttol) then       

        !based on the latest price, what is the policy here at pind?
        polstar = polmatp_interp(zct,endogct,pind)
        
        !insert distributional weight in appropriate slots next period
        distzkl(:,polstar,t+1) = distzkl(:,polstar,t+1) + &
        	pr_mat_z(zct,:,sct)*distzkl(zct,endogct,t) * (1.0-pwgt)

        !based on the latest price, what is the policy here at pind+1?
        polstar = polmatp_interp(zct,endogct,pind+1)
        
        !insert distributional weight in appropriate slots next period
        distzkl(:,polstar,t+1) = distzkl(:,polstar,t+1) + &
        	pr_mat_z(zct,:,sct)*distzkl(zct,endogct,t) * pwgt
        	
        
        end if 
    end do !endogct
    end do !zct
    
    !now, round to make sure that you're ending up with a distribution which makes sense each period
    distzkl(:,:,t+1) = distzkl(:,:,t+1)/sum(distzkl(:,:,t+1))
        
end do !t

write(*,*) " "
write(*,"(A,F15.1,A)") "Finished dist manipulation in ",omp_get_wtime()-start," seconds."    

write(13,*) " "
write(13,"(A,F15.1,A)") "Finished dist manipulation in ",omp_get_wtime()-start," seconds."    

if (checkbounds==1) then
    
    write(*,*) " "
    write(*,*) "Checking for bounds of state space in simulation."
    
    write(13,*) " "
    write(13,*) "Checking for bounds of state space in simulation."
    
    TOPksim(:) = 0.0
    BOTksim(:) = 0.0
    TOPlsim(:) = 0.0 
    BOTlsim(:) = 0.0
    
    do t = 1,numper-1
        do zct=1,znum
        do endogct=1,numendog
            kct = endog_pos(endogct,1); lmin1ct = endog_pos(endogct,2);
            if (kct==1) BOTksim(t) = BOTksim(t) + distzkl(zct,endogct,t)
            if (lmin1ct==1) BOTlsim(t) = BOTlsim(t) + distzkl(zct,endogct,t)
            if (kct==knum) TOPksim(t) = TOPksim(t) + distzkl(zct,endogct,t)
            if (lmin1ct==lnum) TOPlsim(t) = TOPlsim(t) + distzkl(zct,endogct,t)
        end do !endogct
        end do !zct
        
    end do !t




end if !checkbounds
write(*,*) " "
write(*,"(A,F15.1,A)") "Finished simulation in ",omp_get_wtime()-start," seconds."    

write(13,*) " "
write(13,"(A,F15.1,A)") "Finished simulation in ",omp_get_wtime()-start," seconds."

!!!!!END SIMULATION BLOCK

!!!!START THE FCST RULE UPDATE BLOCK

!first, keep a record of the fcst rule coefficients for capital and price
kcoeffstore(:,:,:,:,GEct) = kfcstcoeff(:,:,:,:)
pcoeffstore(:,:,:,:,GEct) = pfcstcoeff(:,:,:,:)

!do the OLS estimation to update the fcst rules

do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    
    !do kprime first
    xmean = 0.0
    ymean = 0.0
    x2mean = 0.0
    xymean = 0.0
    perct=0
    do t = (numdiscard+1),(numper-1)
        if ((asimpos(t)==act.and.ssimpos(t)==sct).and.ssimpos(t-1)==smin1ct) then
            perct = perct+1
            xval = log(Ksim(t))
            yval = log(Ksim(t+1))
            
            xmean = xmean + xval
            ymean = ymean + yval
            x2mean = x2mean + (xval**2.0)
            xymean = xymean + (xval * yval)
            
        end if
    end do !t
    
    xmean = xmean / dble(perct)
    ymean = ymean / dble(perct)
    x2mean = x2mean / dble(perct)
    xymean = xymean / dble(perct)
        
    !insert the implied OLS coefficients into the coefficient storage
    kfcstcoeffnew(act,sct,smin1ct,2) = ( xymean - xmean * ymean ) / ( x2mean - ( xmean ** 2.0 ) )
    kfcstcoeffnew(act,sct,smin1ct,1) = ymean - kfcstcoeffnew(act,sct,smin1ct,2) * xmean
    
    !then do price
    xmean = 0.0
    ymean = 0.0
    x2mean = 0.0
    xymean = 0.0
    perct=0
    do t = (numdiscard+1),(numper-1)
        if ((asimpos(t)==act.and.ssimpos(t)==sct).and.ssimpos(t-1)==smin1ct) then
            perct = perct+1
            xval = log(Ksim(t))
            yval = log(psim(t))
            
            xmean = xmean + xval
            ymean = ymean + yval
            x2mean = x2mean + (xval**2.0)
            xymean = xymean + (xval * yval)
            
        end if
    end do !t
    
    xmean = xmean / dble(perct)
    ymean = ymean / dble(perct)
    x2mean = x2mean / dble(perct)
    xymean = xymean / dble(perct)
    
    !insert the implied OLS coefficients into the coefficient storage
    pfcstcoeffnew(act,sct,smin1ct,2) = ( xymean - xmean * ymean ) / ( x2mean - ( xmean ** 2.0 ) )
    pfcstcoeffnew(act,sct,smin1ct,1) = ymean - pfcstcoeffnew(act,sct,smin1ct,2) * xmean
    
end do !smin1ct
end do !sct
end do !act

!compute the error - how far apart are assumed and eqbm OLS estimates?
kfcsterror = maxval(abs(kfcstcoeff-kfcstcoeffnew))
pfcsterror = maxval(abs(pfcstcoeff-pfcstcoeffnew))

!insert errors into storage
kfcsterrorstore(GEct) = kfcsterror
pfcsterrorstore(GEct) = pfcsterror

!!!!compute RMSE and Rsquared for the forecast rules
pSSE(:,:,:) = 0.0
KSSE(:,:,:) = 0.0
pSST(:,:,:) = 0.0
KSST(:,:,:) = 0.0
Kmean(:,:,:) = 0.0
pmean(:,:,:) = 0.0

!loop over aggregate states
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    !find mean and fcst errors first for price
    perct = 0
    do t=(numdiscard+1),(numper-1)
        if ((asimpos(t)==act.and.ssimpos(t)==sct).and.ssimpos(t-1)==smin1ct) then
            perct = perct+1
            pmean(act,sct,smin1ct) = pmean(act,sct,smin1ct) + log(psim(t))
            pSSE(act,sct,smin1ct) = pSSE(act,sct,smin1ct) + ( log(psim(t)) - log(pfcstsim(t)) )**dble(2.0)
        end if !act and sct flag
    end do !t
    
    !normalize mean, compute RMSE for price
    pmean(act,sct,smin1ct) = pmean(act,sct,smin1ct)/dble(perct)
    pRMSEstore(act,sct,smin1ct,GEct) = sqrt( pSSE(act,sct,smin1ct) / dble(perct) )
    
    !then for capital
    perct = 0
    do t=(numdiscard+1),(numper-1)
        if ((asimpos(t)==act.and.ssimpos(t)==sct).and.ssimpos(t-1)==smin1ct) then
            perct = perct+1
            Kmean(act,sct,smin1ct) = Kmean(act,sct,smin1ct) + log(Ksim(t))
            KSSE(act,sct,smin1ct) = KSSE(act,sct,smin1ct) + ( log(Ksim(t)) - log(Kfcstsim(t)) )**dble(2.0)
        end if !act and sct flag
    end do !t
    
    !normalize mean, compute RMSE for capital
    Kmean(act,sct,smin1ct) = Kmean(act,sct,smin1ct)/dble(perct)
    KRMSEstore(act,sct,smin1ct,GEct) = sqrt( KSSE(act,sct,smin1ct) / dble(perct) )
    
    !now compute the total sum squares
    do t=(numdiscard+1),(numper-1)
        if ((asimpos(t)==act.and.ssimpos(t)==sct).and.ssimpos(t-1)==smin1ct) then
            pSST(act,sct,smin1ct) = pSST(act,sct,smin1ct) + (log(psim(t)) - pmean(act,sct,smin1ct))**dble(2.0)
            KSST(act,sct,smin1ct) = KSST(act,sct,smin1ct) + (log(Ksim(t)) - Kmean(act,sct,smin1ct))**dble(2.0)
        end if !act and sct flag
    end do !t
    
    !now, compute the R^2 values
    pR2store(act,sct,smin1ct,GEct) = 1.0 - ( pSSE(act,sct,smin1ct) / pSST(act,sct,smin1ct) )
    KR2store(act,sct,smin1ct,GEct) = 1.0 - ( KSSE(act,sct,smin1ct) / KSST(act,sct,smin1ct) )
end do !smin1ct
end do !sct
end do !act



!!!record the change of the RMSE's and the R^2's
if (GEct==1) then
    
    !in this case, it isn't change, just absolute RMSE and distance of R^2 from 1
    KRMSEchange = maxval(abs(KRMSEstore(:,:,:,GEct)))
    pRMSEchange = maxval(abs(pRMSEstore(:,:,:,GEct)))
    KR2change = maxval(abs(1 - KR2store(:,:,:,GEct)))
    pR2change = maxval(abs(1 - pR2store(:,:,:,GEct)))

else 

    !now, it's the change in the metrics
    KRMSEchange = maxval(abs(KRMSEstore(:,:,:,GEct)-KRMSEstore(:,:,:,GEct-1)))
    pRMSEchange = maxval(abs(pRMSEstore(:,:,:,GEct)-pRMSEstore(:,:,:,GEct-1)))
    KR2change = maxval(abs(KR2store(:,:,:,GEct)-KR2store(:,:,:,GEct-1)))
    pR2change = maxval(abs(pR2store(:,:,:,GEct)-pR2store(:,:,:,GEct-1)))

end if

!store the RMSE and R2 errors
KRMSEchangestore(GEct) = KRMSEchange
pRMSEchangestore(GEct) = pRMSEchange
KR2changestore(GEct) = KR2change
pR2changestore(GEct) = pR2change


!!!Compute the Den Haan fcst series

!initialize the entire fcst series
pDenHaanfcst(:) = 0.0
KDenHaanfcst(:) = 0.0

!initialize the period before the sample
pDenHaanfcst(numdiscard) = log(psim(numdiscard))
KDenHaanfcst(numdiscard) = log(Ksim(numdiscard))

!initialize the statistics
pmaxDenHaanstat(GEct) = 0.0
KmaxDenHaanstat(GEct) = 0.0
pavgDenHaanstat(GEct) = 0.0
KavgDenHaanstat(GEct) = 0.0

!loop over the periods
do t=(numdiscard+1),(numper-1)
    
    !get states from last period, when the capital forecast was being made
    act = asimpos(t-1)
    sct = ssimpos(t-1)
    smin1ct = ssimpos(t-2)
    
    !insert the capital fcst
    KDenHaanfcst(t) = kfcstcoeff(act,sct,smin1ct,1) + kfcstcoeff(act,sct,smin1ct,2)*KDenHaanfcst(t-1)
    
    !now, get states from this period, when the price forecast is being made
    act = asimpos(t)
    sct = ssimpos(t)
    smin1ct = ssimpos(t-1)
    
    !insert the price fcst
    pDenHaanfcst(t) = pfcstcoeff(act,sct,smin1ct,1) + pfcstcoeff(act,sct,smin1ct,2)*KDenHaanfcst(t)
    
    !iterate the max statistics
    pmaxDenHaanstat(GEct) = maxval((/pmaxDenHaanstat(GEct), abs(pDenHaanfcst(t) - log(psim(t))) /))
    KmaxDenHaanstat(GEct) = maxval((/KmaxDenHaanstat(GEct), abs(KDenHaanfcst(t) - log(Ksim(t))) /))
    
    !iterate the average statistics
    pavgDenHaanstat(GEct) = pavgDenHaanstat(GEct) + abs(pDenHaanfcst(t) - log(psim(t)))
    KavgDenHaanstat(GEct) = KavgDenHaanstat(GEct) + abs(KDenHaanfcst(t) - log(Ksim(t)))
    
end do !t

!normalize the average statistics
pavgDenHaanstat(GEct) = pavgDenHaanstat(GEct) / (dble(numper-numdiscard-1))
KavgDenHaanstat(GEct) = KavgDenHaanstat(GEct) / (dble(numper-numdiscard-1))


!record the change of the avg and max statistics
if (GEct==1) then

    pmaxDenHaanchange = pmaxDenHaanstat(GEct)
    KmaxDenHaanchange = KmaxDenHaanstat(GEct)
    pavgDenHaanchange = pavgDenHaanstat(GEct)
    KavgDenHaanchange = KavgDenHaanstat(GEct)

else if (GEct>1) then

    pmaxDenHaanchange = abs(pmaxDenHaanstat(GEct)-pmaxDenHaanstat(GEct-1))
    KmaxDenHaanchange = abs(KmaxDenHaanstat(GEct)-KmaxDenHaanstat(GEct-1))
    pavgDenHaanchange = abs(pavgDenHaanstat(GEct)-pavgDenHaanstat(GEct-1))
    KavgDenHaanchange = abs(KavgDenHaanstat(GEct)-KavgDenHaanstat(GEct-1))

end if
    
!store the changes    
pmaxDenHaanchangestore(GEct) = pmaxDenHaanchange
KmaxDenHaanchangestore(GEct) = KmaxDenHaanchange
pavgDenHaanchangestore(GEct) = pavgDenHaanchange
KavgDenHaanchangestore(GEct) = KavgDenHaanchange

!display the old and the new coefficients
write(*,*) " "
write(*,"(A,I5)") "GE Iteration ",GEct
write(*,"(A,F10.5)") "Capital Fcst Coeff Error = ",kfcsterror
write(*,"(A,F10.5)") "Price Fcst Coeff Error = ",pfcsterror
write(*,"(A,F10.5)") "Capital Fcst RMSE Change = ",KRMSEchange
write(*,"(A,F10.5)") "Price Fcst RMSE Change = ",pRMSEchange
write(*,"(A,F10.5)") "Capital Fcst R2 Change = ",KR2change
write(*,"(A,F10.5)") "Price Fcst R2 Change = ",pR2change
write(*,"(A,F10.5,F10.5)") "Capital Den Haan % Error (avg,max) = ",KavgDenHaanstat(GEct),KmaxDenHaanstat(GEct)
write(*,"(A,F10.5,F10.5)") "Price Den Haan % Error (avg,max) = ",pavgDenHaanstat(GEct),pmaxDenHaanstat(GEct)
write(*,"(A,F10.5,F10.5)") "Chg. in Capital Den Haan % Error (avg,max) = ",KavgDenHaanchange,KmaxDenHaanchange
write(*,"(A,F10.5,F10.5)") "Chg. in Price Den Haan % Error (avg,max) = ",pavgDenHaanchange,pmaxDenHaanchange
write(*,*) " "
write(*,*) "Capital Forecast Coefficients"

write(13,*) " "
write(13,"(A,I5)") "GE Iteration ",GEct
write(13,"(A,F10.5)") "Capital Fcst Coeff Error = ",kfcsterror
write(13,"(A,F10.5)") "Price Fcst Coeff Error = ",pfcsterror
write(13,"(A,F10.5)") "Capital Fcst RMSE Change = ",KRMSEchange
write(13,"(A,F10.5)") "Price Fcst RMSE Change = ",pRMSEchange
write(13,"(A,F10.5)") "Capital Fcst R2 Change = ",KR2change
write(13,"(A,F10.5)") "Price Fcst R2 Change = ",pR2change
write(13,"(A,F10.5,F10.5)") "Capital Den Haan % Error (avg,max) = ",KavgDenHaanstat(GEct),KmaxDenHaanstat(GEct)
write(13,"(A,F10.5,F10.5)") "Price Den Haan % Error (avg,max) = ",pavgDenHaanstat(GEct),pmaxDenHaanstat(GEct)
write(13,"(A,F10.5,F10.5)") "Chg. in Capital Den Haan % Error (avg,max) = ",KavgDenHaanchange,KmaxDenHaanchange
write(13,"(A,F10.5,F10.5)") "Chg. in Price Den Haan % Error (avg,max) = ",pavgDenHaanchange,pmaxDenHaanchange
write(13,*) " "
write(13,*) "Capital Forecast Coefficients"

do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    write(*,"(F10.5,F10.5)") kfcstcoeff(act,sct,smin1ct,:)
    write(13,"(F10.5,F10.5)") kfcstcoeff(act,sct,smin1ct,:)
end do !smin1ct
end do !sct
end do !act

write(*,*) " "
write(*,*) "Estimated Capital Forecast Coefficients"

write(13,*) " "
write(13,*) "Estimated Capital Forecast Coefficients"

do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    write(*,"(F10.5,F10.5)") kfcstcoeffnew(act,sct,smin1ct,:)
    write(13,"(F10.5,F10.5)") kfcstcoeffnew(act,sct,smin1ct,:)
end do !smin1ct
end do !sct
end do !act

write(*,*) " "
write(*,*) "Capital RMSE, R^2"

write(13,*) " "
write(13,*) "Capital RMSE, R^2"

do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    write(*,"(F20.10,A,F20.10)") KRMSEstore(act,sct,smin1ct,GEct),', ',KR2store(act,sct,smin1ct,GEct)
    write(13,"(F20.10,A,F20.10)") KRMSEstore(act,sct,smin1ct,GEct),', ',KR2store(act,sct,smin1ct,GEct)
end do !smin1ct
end do !sct
end do !act


write(*,*) " "
write(*,*) "Price Forecast Coefficients"

write(13,*) " "
write(13,*) "Price Forecast Coefficients"

do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    write(*,"(F10.5,F10.5)") pfcstcoeff(act,sct,smin1ct,:)
    write(13,"(F10.5,F10.5)") pfcstcoeff(act,sct,smin1ct,:)
end do !smin1ct
end do !sct
end do !act

write(*,*) " "
write(*,*) "Estimated Price Forecast Coefficients"

write(13,*) " "
write(13,*) "Estimated Price Forecast Coefficients"

do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    write(*,"(F10.5,F10.5)") pfcstcoeffnew(act,sct,smin1ct,:)
    write(13,"(F10.5,F10.5)") pfcstcoeffnew(act,sct,smin1ct,:)
end do !smin1ct
end do !sct
end do !act
write(*,*) " "

write(*,*) " "
write(*,*) "Price RMSE, R^2"

write(13,*) " "
write(13,*) "Price RMSE, R^2"

do act=1,anum
do sct=1,snum
do smin1ct=1,snum
    write(*,"(F20.10,A,F20.10)") pRMSEstore(act,sct,smin1ct,GEct),', ',pR2store(act,sct,smin1ct,GEct)
    write(13,"(F20.10,A,F20.10)") pRMSEstore(act,sct,smin1ct,GEct),', ',pR2store(act,sct,smin1ct,GEct)
end do !smin1ct
end do !sct
end do !act

!!!!WRITE THE PRELIM RESULTS BEFORE DETERMINING EXIT STATUS

write(*,*) "Writing preliminary output to .txt files."
write(13,*) "Writing preliminary output to .txt files."

write(*,*) " "
write(13,*) " "


if (doVFI==1) then
!VF
open(8,file="V.txt")
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
write(8,*) V(endogct,exogct,fcstct)
end do !fcstct
end do !exogct
end do !endogct
close(8)

!policy matrix
open(8,file="polmat.txt")
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
write(8,*) polmat(endogct,exogct,fcstct)
end do !fcstct
end do !exogct
end do !endogct
close(8)

!kprime policy key
open(8,file="kprime_key.txt")
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
write(8,*) kprime_key(endogct,exogct,fcstct)        
end do !fcstct
end do !exogct
end do !endogct
close(8)

!labor policy key
open(8,file="lpol_key.txt")
do endogct=1,numendog
do exogct=1,numexog
do fcstct=1,numfcst
write(8,*) lpol_key(endogct,exogct,fcstct)        
end do !fcstct
end do !exogct
end do !endogct
close(8)
end if; !doVFI==1

!output series
open(8,file="Ysim.txt")
do t=1,numper
write(8,*) Ysim(t)
end do !t
close(8)

!capital series
open(8,file="Ksim.txt")
do t=1,numper
write(8,*) Ksim(t)
end do !t
close(8)

!investment series
open(8,file="Isim.txt")
do t=1,numper
write(8,*) Isim(t)
end do !t
close(8)

!labor series
open(8,file="Lsim.txt")
do t=1,numper
write(8,*) Lsim(t)
end do !t
close(8)

!hiring series
open(8,file="Hsim.txt")
do t=1,numper
write(8,*) Hsim(t)
end do !t
close(8)

!consumption series
open(8,file="Csim.txt")
do t=1,numper
write(8,*) Csim(t)
end do !t
close(8)

!eqbm price series
open(8,file="psim.txt")
do t=1,numper
write(8,*) psim(t)
end do !t
close(8)


!capital AC series
open(8,file="ACksim.txt")
do t=1,numper
write(8,*) ACksim(t)
end do !t
close(8)

!labor AC series
open(8,file="AClsim.txt")
do t=1,numper
write(8,*) AClsim(t)
end do !t
close(8)

!top capital grid series
open(8,file="TOPksim.txt")
do t=1,numper
write(8,*) TOPksim(t)
end do !t
close(8)

!bottom capital grid series
open(8,file="BOTksim.txt")
do t=1,numper
write(8,*) BOTksim(t)
end do !t
close(8)

!top labor grid series
open(8,file="TOPlsim.txt")
do t=1,numper
write(8,*) TOPlsim(t)
end do !t
close(8)

!bottom labor grid series
open(8,file="BOTlsim.txt")
do t=1,numper
write(8,*) BOTlsim(t)
end do !t
close(8)


!forecast price series
open(8,file="pfcstsim.txt")
do t=1,numper
write(8,*) pfcstsim(t)
end do !t
close(8)

!forecast capital series
open(8,file="Kfcstsim.txt")
do t=1,numper
write(8,*) Kfcstsim(t)
end do !t
close(8)

!price iterations series
open(8,file="piterstoresim.txt")
do t=1,numper
write(8,*) piterstoresim(t)
end do !t
close(8)

!capital fcst coeffs
open(8,file="kcoeffstore.txt")
do ct=1,maxGEit
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
write(8,*) kcoeffstore(act,sct,smin1ct,:,ct)
end do !smin1ct
end do !sct
end do !act
end do !ct
close(8)

!price fcst coeffs
open(8,file="pcoeffstore.txt")
do ct=1,maxGEit
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
write(8,*) pcoeffstore(act,sct,smin1ct,:,ct)
end do !smin1ct
end do !sct
end do !act
end do !ct
close(8)

!capital fcst errors
open(8,file="kfcsterrorstore.txt")
do ct=1,maxGEit
write(8,*) kfcsterrorstore(ct)
end do !ct
close(8)

!price fcst errors
open(8,file="pfcsterrorstore.txt")
do ct=1,maxGEit
write(8,*) pfcsterrorstore(ct)
end do !ct
close(8)

!capital RMSE
open(8,file="KRMSEstore.txt")
do ct=1,maxGEit
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
write(8,*) KRMSEstore(act,sct,smin1ct,ct)
end do !smin1ct
end do !sct
end do !act
end do !ct
close(8)

!capital R2
open(8,file="KR2store.txt")
do ct=1,maxGEit
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
write(8,*) KR2store(act,sct,smin1ct,ct)
end do !smin1ct
end do !sct
end do !act
end do !ct
close(8)

!price RMSE
open(8,file="pRMSEstore.txt")
do ct=1,maxGEit
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
write(8,*) pRMSEstore(act,sct,smin1ct,ct)
end do !smin1ct
end do !sct
end do !act
end do !ct
close(8)

!price R2
open(8,file="pR2store.txt")
do ct=1,maxGEit
do act=1,anum
do sct=1,snum
do smin1ct=1,snum
write(8,*) pR2store(act,sct,smin1ct,ct)
end do !smin1ct
end do !sct
end do !act
end do !ct
close(8)

!capital RMSE change storage
open(8,file="KRMSEchangestore.txt")
do ct=1,maxGEit
write(8,*) KRMSEchangestore(ct)
end do !ct
close(8)

!price RMSE change storage
open(8,file="pRMSEchangestore.txt")
do ct=1,maxGEit
write(8,*) pRMSEchangestore(ct)
end do !ct
close(8)

!capital R2 change storage
open(8,file="KR2changestore.txt")
do ct=1,maxGEit
write(8,*) KR2changestore(ct)
end do !ct
close(8)

!price R2 change storage
open(8,file="pR2changestore.txt")
do ct=1,maxGEit
write(8,*) pR2changestore(ct)
end do !ct
close(8)

!capital max DenHaan stat storage
open(8,file="KmaxDenHaanstat.txt")
do ct=1,maxGEit
write(8,*) KmaxDenHaanstat(ct)
end do !ct
close(8)

!price max DenHaan stat storage
open(8,file="pmaxDenHaanstat.txt")
do ct=1,maxGEit
write(8,*) pmaxDenHaanstat(ct)
end do !ct
close(8)

!capital avg DenHaan stat storage
open(8,file="KavgDenHaanstat.txt")
do ct=1,maxGEit
write(8,*) KavgDenHaanstat(ct)
end do !ct
close(8)

!price avg DenHaan stat storage
open(8,file="pavgDenHaanstat.txt")
do ct=1,maxGEit
write(8,*) pavgDenHaanstat(ct)
end do !ct
close(8)


!capital max DenHaan stat change storage
open(8,file="KmaxDenHaanchangestore.txt")
do ct=1,maxGEit
write(8,*) KmaxDenHaanchangestore(ct)
end do !ct
close(8)

!price max DenHaan stat change storage
open(8,file="pmaxDenHaanchangestore.txt")
do ct=1,maxGEit
write(8,*) pmaxDenHaanchangestore(ct)
end do !ct
close(8)

!capital avg DenHaan stat change storage
open(8,file="KavgDenHaanchangestore.txt")
do ct=1,maxGEit
write(8,*) KavgDenHaanchangestore(ct)
end do !ct
close(8)

!price avg DenHaan stat change storage
open(8,file="pavgDenHaanchangestore.txt")
do ct=1,maxGEit
write(8,*) pavgDenHaanchangestore(ct)
end do !ct
close(8)

!Den Haan capital fcst series
open(8,file="KDenHaanfcst.txt")
do t=1,numper
write(8,*) KDenHaanfcst(t)
end do !t
close(8)

!Den Haan price fcst series
open(8,file="pDenHaanfcst.txt")
do t=1,numper
write(8,*) pDenHaanfcst(t)
end do !t
close(8)

!the entire unconditional simulation distribution
open(8,file="distzkl.txt")
do zct=1,znum
do endogct=1,numendog
do t=1,numper
write(8,*) distzkl(zct,endogct,t)
end do !t
end do !endogct
end do !zct
close(8)

!!!!END PRELIM FILE OUTPUT BLOCK

!exit criterion for the GEct looop
exitflag = 0
!coefficient convergence
if (GEerrorswitch==1) then
    if (pfcsterror<fcsterrortol.and.kfcsterror<fcsterrortol) exitflag = 1
!RMSE convergence    
else if (GEerrorswitch==2) then
    if (KRMSEchange<RMSEchangetol.and.pRMSEchange<RMSEchangetol) exitflag = 1
!R2 convergence    
else if (GEerrorswitch==3) then
    if (KR2change<R2changetol.and.pR2change<R2changetol) exitflag = 1
!max Den Haan stat convergence    
else if (GEerrorswitch==4) then
    if (KmaxDenHaanchange<maxDenHaanchangetol.and.pmaxDenHaanchange<maxDenHaanchangetol) exitflag = 1    
!avg Den Haan stat convergence    
else if (GEerrorswitch==5) then
    if (KavgDenHaanchange<avgDenHaanchangetol.and.pavgDenHaanchange<avgDenHaanchangetol) exitflag = 1    
end if

!this actually exits the GE loop if you've converged on the desired metric
if ((exitflag==1).or.(maxGEit==1)) exit

!update the forecast rule coefficients with gain - i.e. dampened fixed point iteration
kfcstcoeff = kfcstcoeff + GEupdate * (kfcstcoeffnew - kfcstcoeff)
pfcstcoeff = pfcstcoeff + GEupdate * (pfcstcoeffnew - pfcstcoeff)

!if the forecast rule has not converged, need to update forecast matrices used
!in the VFI for the next GE iteration
do act=1,anum
do sct=1,snum
do smin1ct=1,snum

    !insert the forecasts into the fcst array
    do fcstct=1,numfcst
        kbarval = kbar0(fcstct); !what is the value of aggregate capital?
        kbarfcstval = exp( kfcstcoeff(act,sct,smin1ct,1) + kfcstcoeff(act,sct,smin1ct,2) * log( kbarval ) ); !implied agg cap fcst
        pfcstval = exp( pfcstcoeff(act,sct,smin1ct,1) + pfcstcoeff(act,sct,smin1ct,2) * log( kbarval ) ); !implied price fcst
        
        !adjust the agg capital forecast to the boundaries of the grid
        kbarfcstval = minval( (/ maxval( (/ kbarmin , kbarfcstval /) ) , kbarmax /) )
        
        !find the interval of kbar0 in which kbarfcst lands, as well as the lin. interp. weight
        ind = fcstct; call hunt(kbar0,numfcst,kbarfcstval,ind)
        weight = ( kbarfcstval - kbar0(ind) ) / ( kbar0(ind+1) - kbar0(ind) )
        
            !got to make sure that out of bounds entries are handled correctly
        if (ind>=1.and.ind<kbarnum) then
            kbarfcstinds(act,sct,smin1ct,fcstct) = ind; kbarfcstweights(act,sct,smin1ct,fcstct) = weight
        else if (ind==0) then
            kbarfcstinds(act,sct,smin1ct,fcstct) = 1; kbarfcstweights(act,sct,smin1ct,fcstct) = 0.0
        else if (ind==kbarnum) then
            kbarfcstinds(act,sct,smin1ct,fcstct) = kbarnum-1; kbarfcstweights(act,sct,smin1ct,fcstct) = 1.0
        end if        
        
        !insertion into arrays, including insertion of the implied wage
        kbarfcstmat(act,sct,smin1ct,fcstct) = kbarfcstval
        pfcstmat(act,sct,smin1ct,fcstct) = pfcstval
        wfcstmat(act,sct,smin1ct,fcstct) = theta / pfcstval
        
    end do !fcstct
end do !smin1ct
end do !sct
end do !act

!!!!END THE FCST RULE UPDATE BLOCK

end do; !GEct - THIS ENDS THE GE FCST RULE ITERATING

!!!!!IRF BLOCK
if (doIRF==1) then

!then, initialize the aggregate series to 0
YsimIRF(:,:,:) = 0.0; KsimIRF = YsimIRF; LsimIRF = YsimIRF; IsimIRF = YsimIRF; HsimIRF = YsimIRF; 
ACksimIRF = YsimIRF; AClsimIRF = YsimIRF; ADJksimIRF = YsimIRF; ADJlsimIRF = YsimIRF; 
CsimIRF = YsimIRF; psimIRF =YsimIRF;

!start with aggregate capital guessed at some reasonable value from uncond simulation
KsimIRF(1,:,:) = Ksim(numper-5)

write(*,"(A,F15.1,A)") "Starting actual IRF sim at ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Starting actual IRF sim at ",omp_get_wtime()-start," seconds."


if (IRFrestart==0) then
	
	IRFsimst = 1
	IRFctst=1
	
else if (IRFrestart==1) then

	!read in the IRF start index and existing data
	open(8,file="IRFsimst.txt")
	read(8,*) IRFsimst
	read(8,*) IRFctst
	close(8)
	
	!read all of the output from the IRF simulation .txt files
!output series
open(8,file="YsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) YsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!capital series
open(8,file="KsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) KsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!investment series
open(8,file="IsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) IsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!labor series
open(8,file="LsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) LsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!hiring series
open(8,file="HsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) HsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!consumption series
open(8,file="CsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) CsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!eqbm price series
open(8,file="psimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) psimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!capital AC series
open(8,file="ACksimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) ACksimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!labor AC series
open(8,file="AClsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) AClsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!forecast price series
open(8,file="pfcstsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) pfcstsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!forecast capital series
open(8,file="KfcstsimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) KfcstsimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!price iterations series
open(8,file="piterstoresimIRF.txt")
do ct=1,2
do simct=1,numsimIRF
do t=1,lengthIRF
read(8,*) piterstoresimIRF(t,simct,ct)
end do !t
end do !simct
end do !ct
close(8)

!the entire unconditional simulation distribution
open(8,file="distzkl.txt")
do zct=1,znum
do endogct=1,numendog
do t=1,numper
read(8,*) distzkl(zct,endogct,t)
end do !t
end do !endogct
end do !zct
close(8)

	
end if !IRFrestart==1


!now, loop over periods in the simulation, tracking the endogenous
!movement of weight around the grid


do ct=IRFctst,2

do simct=IRFsimst,numsimIRF
    
    !first, initialize the distribution over endogenous variables
    distzklbefore(:,:) = 0.0; !set everything to zero

    !take the dist from 10 periods before
    !the end of the last unconditional simulation
    distzklbefore(:,:) = distzkl(:,:,numper-5); !uniformly distributed

    !now, round the dist
    distzklbefore(:,:) = distzklbefore(:,:) / sum(distzklbefore(:,:))

    
do t=1,lengthIRF-1
    
    
    !aggregate states
    if (t>1) then
        act = asimposIRF(t,simct,ct); sct=ssimposIRF(t,simct,ct); smin1ct=ssimposIRF(t-1,simct,ct)
    else if (t==1) then
        act = asimposIRF(t,simct,ct); sct=ssimposIRF(t,simct,ct); smin1ct=1
    end if
    aval = a0(act)
    
    !what is the next period forecast capital, in grid boundaries? index and weight?
    kbarfcstval = exp( kfcstcoeff(act,sct,smin1ct,1) + kfcstcoeff(act,sct,smin1ct,2) * log( KsimIRF(t,simct,ct) ) );
    kbarfcstval = minval( (/ maxval( (/ kbarmin , kbarfcstval /) ) , kbarmax /) )   
    ind = kbarnum/2; call hunt(kbar0,numfcst,kbarfcstval,ind)
    weight = ( kbarfcstval - kbar0(ind) ) / ( kbar0(ind+1) - kbar0(ind) )
    
    !guard against off grid point values
    if (ind==kbarnum) then 
        ind = kbarnum-1;
        weight = 1.0;
    else if (ind==0) then
        ind = 1;
        weight = 0.0;
    end if
    
    KfcstsimIRF(t+1,simct,ct) = kbarfcstval
    
    !what is the current period forecast price?
    pfcstsimIRF(t,simct,ct) = exp( pfcstcoeff(act,sct,smin1ct,1) + pfcstcoeff(act,sct,smin1ct,2) * log( KsimIRF(t,simct,ct) ) );

    !note that EVmat doesn't depend on p, so it can be pre-computed now, as EVmatp
    
    !set up the parallelization for the creation of the continuation value
    !$omp parallel private(zct,polct,exogct,exogprimect,Vnextval)
    !$omp do collapse(2)
    do zct=1,znum
    do polct=1,numendog
        
        !what is the implied value of exogct, given simulated aggregate states?
        exogct = (zct-1)*anum*snum*snum + (act-1)*snum*snum + (sct-1)*snum + smin1ct
              
        !create expected value for the next period based on policy, exogenous transitions,
        !and forecast capital
        EVmatp(zct,polct) = 0.0
        do exogprimect=1,numexog; !loop over exogenous realizations next period
            
            !what is the linearly interpolated continuation value?
            Vnextval = weight * V(polct,exogprimect,ind+1) + &
                    (1.0 - weight)*V(polct,exogprimect,ind)

            !add the weighted continuation value to the next period
            EVmatp(zct,polct) = EVmatp(zct,polct) + pr_mat(exogct,exogprimect) * Vnextval
            
        end do !exogprimect
    
    end do !zct
    end do !endogct
    !$omp end do nowait
    !$omp end parallel


	!set up the interpolation of the excess demand function e(p) = 1/p - C(p)
	ep0(:) = 0.0
	Cp0(:) = 0.0
	Yp0(:) = 0.0
	Ip0(:) = 0.0
	ACkp0(:) = 0.0
	AClp0(:) = 0.0
	Kbarprimep0(:) = 0.0
	Hp0(:) = 0.0
	Lp0(:) = 0.0
	polmatp_interp(:,:,:) = 0
	
	do piter = 1,pnum
        !initialize the price-dependent policies to 0
        polmatp(:,:) = 0; kprime_posp(:,:) = 0; lpol_posp(:,:) = 0
		
		pval = p0(piter) !loop over the vector of pre-stored p values
	
		wval = theta / pval; !what is the wage implied by this value of p?
        
        !this block computes the policies given the current pval & wval at each
        !point (z,kl_{-1})
        
        Yvalp = 0.0; 
        Ivalp = 0.0; 
        ACkvalp = 0.0; 
        AClvalp = 0.0; 
        Kbarprimevalp = 0.0; 
        Hvalp = 0.0;
        Lvalp = 0.0
        
        !$omp parallel private(zct,endogct,exogct,kct,lmin1ct,RHSvec,polct,kprimect,&
        !$omp& lct,polstar) reduction(+:Yvalp,Ivalp,ACkvalp,AClvalp,Kbarprimevalp,Hvalp,&
        !$omp& Lvalp)
        !$omp do collapse(2)
        do zct=1,znum
        do endogct=1,numendog
            !restriction on which policies to compute - important for time
            if (distzklbefore(zct,endogct)>disttol) then                        
            !extract states from loop indexes
            exogct = (zct-1)*anum*snum*snum + (act-1)*snum*snum + (sct-1)*snum  + smin1ct
            
            !extract positions associated with these states
            kct = endog_pos(endogct,1);
            lmin1ct= endog_pos(endogct,2); 
            
            !create RHS vector by looping over policies
            RHSvec(:) = 0.0
            do polct=1,numendog
                
                !extract positions associated with the policies
                kprimect = endog_pos(polct,1); lct = endog_pos(polct,2)

                !initialize the RHS by forming the current period return
                RHSvec(polct) = pval * ( Ymat(zct,act,kct,lct) &
                    - ACkmat(zct,act,kct,lct,kprimect) &
                    - Imat(kct,kprimect) - wval * l0(lct) &
                    - ACl(z0(zct),a0(act),k0(kct),l0(lct),alpha,nu,l0(lmin1ct),hirelin,firelin,&
                    labfix,deltan,wval))
                  
                !actually form the RHS by adding continuation value
                RHSvec(polct) = RHSvec(polct) + beta * EVmatp(zct,polct)
               
            end do !polct
                
            !what is the new optimum value in index form and in capital and labor indexes?
            polmatp(zct,endogct) =  maxloc(RHSvec,1)
            kprime_posp(zct,endogct) = endog_pos(polmatp(zct,endogct),1)
            lpol_posp(zct,endogct) = endog_pos(polmatp(zct,endogct),2)
            
            !what are the investment and labor for this position?
            lct = lpol_posp(zct,endogct); !current labor, policy
            kprimect = kprime_posp(zct,endogct); !next period capital, policy
            
            !what are the aggregates?
            Yvalp = Yvalp + distzklbefore(zct,endogct) * Ymat(zct,act,kct,lct)
            Ivalp = Ivalp + distzklbefore(zct,endogct) * Imat(kct,kprimect)
            ACkvalp = ACkvalp + distzklbefore(zct,endogct) * ACkmat(zct,act,kct,lct,kprimect)
            AClvalp = AClvalp + distzklbefore(zct,endogct) * ACl(z0(zct),a0(act),k0(kct),&
                l0(lct),alpha,nu,l0(lmin1ct),hirelin,firelin,labfix,deltan,wval)
            Kbarprimevalp = Kbarprimevalp + distzklbefore(zct,endogct) * k0(kprimect)
            Hvalp = Hvalp + distzklbefore(zct,endogct) * ( l0(lct) - (1.0-deltan) * l0(lmin1ct))
            Lvalp = Lvalp + distzklbefore(zct,endogct) * l0(lct)
        
            end if !disttol
        
        end do !endogct
        end do !zct
        !$omp end do nowait
        !$omp end parallel
        
        !what is implied consumption?
        Cvalp = Yvalp - Ivalp - ACkvalp - AClvalp
        
        !insert implied value of excess demand, and all other values into storage vectors
        ep0(piter) = 1.0/pval - Cvalp
        Cp0(piter) = Cvalp
       	Yp0(piter) = Yvalp
		Ip0(piter) = Ivalp
		ACkp0(piter) = ACkvalp
		AClp0(piter) = AClvalp
		Kbarprimep0(piter) = Kbarprimevalp
		Hp0(piter) = Hvalp
		Lp0(piter) = Lvalp
		polmatp_interp(:,:,piter) = polmatp
	
	end do !piter


    !set up the boundaries of the bisection
        pvala = plb
        pvalb = pub
        pvalc = pvala + dble(0.67) * (pvalb-pvala) 

    !iterate over the market-clearing value of p, for each value
    !redoing the optimization of the RHS of the Bellman equation
    !and recomputing policies
    do piter=1,maxpit
        !initialize the price-dependent policies to 0
        polmatp(:,:) = 0; kprime_posp(:,:) = 0; lpol_posp(:,:) = 0
        
        !brent setup
        if (piter==1) pval = pvala
        if (piter==2) pval = pvalb
        if (piter==3) pval = pvalc
        
        !brent restart
        if (piter==pcutoff) then 
            pvala = pfcstsim(t)-4.0*pwindow;
            pval = pvala;
        else if (piter==pcutoff+1) then 
            pvalb =  pfcstsim(t)+4.0*pwindow;
            pval = pvalb
        else if (piter==pcutoff+2) then
            pvalc = pvala + dble(0.67) * (pvalb-pvala) 
            pval = pvalc
        end if
        
        !if not restarting or initializing
        if ((piter>3.and.piter<pcutoff).or.(piter>pcutoff+2)) then 
            
        
            !first, try inverse quadratic interpolation of the excess demand function
            pval = ( pvala * fb * fc ) / ( (fa - fb) * (fa - fc) ) &
                + ( pvalb * fa * fc ) / ( (fb - fa) * (fb - fc ) ) &
                + ( pvalc * fa * fb ) / ( (fc - fa) * (fc - fb ) )
       
            !if it lies within bounds, and isn't too close to the bounds, then done
            
            !o/w, take bisection step
            if ((minval( (/ abs(pvala - pval), abs(pvalb-pval) /) )<&
                    abs( (pvalb-pvala)/dble(9.0) ) ).or.(pval<pvala).or.(pval>pvalb))   then
                pval = (pvala + pvalb) / dble(2.0)
         
            end if
            
        end if 
        
        
		!!!actually evaluate consumption approximation function
		pval = minval( (/ maxval( (/ p0(1) , pval /) ) , p0(pnum) /) )   
	    pind = pnum/2; call hunt(p0,pnum,pval,pind)
	    pwgt = (pval - p0(pind)) / (p0(pind + 1) - p0(pind))
	    
	    !guard against off grid point values
    	if (pind==pnum) then 
        	pind = pnum-1;
	        pwgt = 1.0;
    	else if (pind==0) then
        	pind = 1;
	        pwgt = 0.0;
    	end if
		
		!actually perform linear interpolation of the consumption function
		Cvalp = Cp0(pind)*(1.0-pwgt) + Cp0(pind+1)*pwgt
		        
        !are you initializing the brent?
        if (piter==1) fa = (1/pval) - Cvalp
        if (piter==2) fb = (1/pval) - Cvalp
        if (piter==3) fc = (1/pval) - Cvalp
        
        !are you restarting the brent?
        if (piter==pcutoff) fa = (1/pval) - Cvalp
        if (piter==pcutoff+1) fb = (1/pval) - Cvalp
        if (piter==pcutoff+2) fc = (1/pval) - Cvalp
        
        !what is the error given by this implied consumption?
        perror = (1/pval) - Cvalp
        
        !if not restarting or initializing
        if ((piter>3.and.piter<pcutoff).or.(piter>pcutoff+2)) then 
            if (perror<0) then
                pvalc = pvalb; fc = fb;
                pvalb = pval; fb = perror;
                !pval a doesn't change
            else if (perror>=0) then
                pvalc = pvala; fc = fa;
                pvala = pval; fa = perror;
                !pval b doesn't change
            end if
        end if
      
        !exit criterion for market-clearing
        perror = log(pval*Cvalp)
        if (abs(perror)<perrortol.and.piter>2) exit
      
        end do !piter
       
	!record number of price iterations
    piterstoresimIRF(t,simct,ct) = piter-1
    
    !output price stats on certain periods
    if (t==25) then 
        write(*,"(A,I5,A,I5,A,F10.4,A,F10.4)") "simct = ",simct,", piter = ",piter-1,&
        ", pval = ",pval,", perror = ",perror
        
        write(13,"(A,I5,A,I5,A,F10.4,A,F10.4)") "simct = ",simct,", piter = ",piter-1,&
        ", pval = ",pval,", perror = ",perror
        
    end if
    
    !insert market-clearing price and other linearly interpolated stuff into sim series
    psimIRF(t,simct,ct) = pval; !this is the most recently run p from the clearing algorithm
    CsimIRF(t,simct,ct) = Cvalp; !this is already the linearly interpolated consumption C
    YsimIRF(t,simct,ct)  = Yp0(pind)*(1.0-pwgt) + Yp0(pind+1)*pwgt; !linearly interpolated output Y
    IsimIRF(t,simct,ct) = Ip0(pind)*(1.0-pwgt) + Ip0(pind+1)*pwgt; !linearly interpolated investment I
    ACksimIRF(t,simct,ct) = ACkp0(pind)*(1.0-pwgt) + ACkp0(pind+1)*pwgt; !linearly interpolated ACk 
    AClsimIRF(t,simct,ct) = AClp0(pind)*(1.0-pwgt) + AClp0(pind+1)*pwgt; !linearly interpolated ACl
    KsimIRF(t+1,simct,ct) = Kbarprimep0(pind)*(1.0-pwgt) + Kbarprimep0(pind+1)*pwgt; !linearly interpolated K' 
    HsimIRF(t,simct,ct) = Hp0(pind)*(1.0-pwgt) + Hp0(pind+1)*pwgt; !linearly interpolated hiring H
    LsimIRF(t,simct,ct) = Lp0(pind)*(1.0-pwgt) + Lp0(pind+1)*pwgt; !linearly interpolated labor input L
    
    !now that the market-clearing price is determined, move on to insert weight into the next period, according to the 
    !linearly interpolated rule
    
    do zct=1,znum
    do endogct=1,numendog
        if (distzklbefore(zct,endogct)>disttol) then       

        !based on the latest price, what is the policy here at pind?
        polstar = polmatp_interp(zct,endogct,pind)
        
        !insert distributional weight in appropriate slots next period
        distzklafter(:,polstar) = distzklafter(:,polstar) + &
        pr_mat_z(zct,:,sct)*distzklbefore(zct,endogct) * (1.0-pwgt)

        !based on the latest price, what is the policy here at pind+1?
        polstar = polmatp_interp(zct,endogct,pind+1)
        
        !insert distributional weight in appropriate slots next period
        distzklafter(:,polstar) = distzklafter(:,polstar) + &
	        pr_mat_z(zct,:,sct)*distzklbefore(zct,endogct) * pwgt
        	
        
        end if 
    end do !endogct
    end do !zct
    
    !now, round to make sure that you're ending up with a distribution which makes sense each period
    distzklafter(:,:) = distzklafter(:,:)/sum(distzklafter(:,:))
    
    !now, make the "before" distribution equal to the after distribution
    distzklbefore = distzklafter
    distzklafter(:,:) = 0.0


end do !t




!write all of the output from the IRF simulation to .txt files
!output series
open(8,file="YsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) YsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!capital series
open(8,file="KsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) KsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!investment series
open(8,file="IsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) IsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!labor series
open(8,file="LsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) LsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!hiring series
open(8,file="HsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) HsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!consumption series
open(8,file="CsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) CsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!eqbm price series
open(8,file="psimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) psimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!capital AC series
open(8,file="ACksimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) ACksimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!labor AC series
open(8,file="AClsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) AClsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!forecast price series
open(8,file="pfcstsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) pfcstsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!forecast capital series
open(8,file="KfcstsimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) KfcstsimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

!price iterations series
open(8,file="piterstoresimIRF.txt")
do writect2=1,2
do writect=1,numsimIRF
do t=1,lengthIRF
write(8,*) piterstoresimIRF(t,writect,writect2)
end do !t
end do !simct
end do !writect2
close(8)

open(8,file="IRFsimst.txt")
write(8,*) simct
write(8,*) ct
close(8)


end do !simct
end do !ct

write(*,"(A,F15.1,A)") "Finished IRF simulation in ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Finished IRF simulation in ",omp_get_wtime()-start," seconds."

end if !doIRF
!!!!!END IRF BLOCK

!record time before beginning file output
write(*,"(A,F15.1,A)") "Finished calculations at ",omp_get_wtime()-start," seconds"
write(13,"(A,F15.1,A)") "Finished calculations at ",omp_get_wtime()-start," seconds"

write(*,*) " "
write(13,*) " "

!!!! FINAL FILE OUTPUT BLOCK

write(*,*) "Writing final output to .txt files."
write(13,*) "Writing final output to .txt files."

write(*,*) " "
write(13,*) " "



!!!!END FINAL FILE OUTPUT BLOCK

!call the clock to get the end time and closes log file
finish = omp_get_wtime()
write(*,"(A,F15.1,A)") "Finished completely in ",finish-start," seconds."
write(13,"(A,F15.1,A)") "Finished completely in ",finish-start," seconds."
close(13)

!!!!INTERNAL FUNCTION AND SUBROUTINES BLOCK
contains 

subroutine unctauchen(transarray,grid,znum,zmin,zmax,snum,sigmagrid,rho)
implicit none 

!this routine discretizes the AR(1) productivity processes, subject to 
!unc shocks

!note that this was double-checked - it duplicates fn_tranvarm2.m results

!input/output declarations
integer :: znum,snum
double precision :: zmin,zmax,rho
double precision :: transarray(znum,znum,snum),grid(znum),sigmagrid(snum)

!other declarations
integer :: zct,sct,zprimect
double precision :: log_grid(znum),standdev,gridinc

!create grid
call linspace(log_grid,log(zmin),log(zmax),znum)
grid = exp(log_grid)
gridinc = log_grid(2)-log_grid(1)

!loop over unc states
do sct=1,snum
    standdev = sigmagrid(sct)
    do zct=1,znum
        !middle intervals
        do zprimect=2,(znum-1)
            transarray(zct,zprimect,sct) = &
                normcdf(log_grid(zprimect)+gridinc/2.0,rho*log_grid(zct),standdev) - &
                normcdf(log_grid(zprimect)-gridinc/2.0,rho*log_grid(zct),standdev)
        end do !zprimect
        !first interval & last interval take remainder of mass
        transarray(zct,1,sct) = normcdf(log_grid(1)+gridinc/2.0,rho*log_grid(zct),standdev)
        transarray(zct,znum,sct) = 1.0 - normcdf(log_grid(znum)-gridinc/2.0,rho*log_grid(zct),standdev)
    end do !zct
    
    !impose that everything sums to 1 with rounding
    do zct=1,znum
        transarray(zct,:,sct) = transarray(zct,:,sct) / sum(transarray(zct,:,sct))
    end do !zct
    
end do !sct

end subroutine unctauchen

subroutine uncexogsimIRF(ssimshocksIRF,asimshocksIRF,ssimposIRF,asimposIRF,pr_mat_a,pr_mat_s,ainit,&
        lengthIRF,numsimIRF,shockperIRF,shocklengthIRF,anum,snum,sinit,singleshock)
implicit none

!this subroutine simulates the unc and agg prod processes in the IRF experiment

!input/output declarations
integer :: ainit,lengthIRF,numsimIRF,shockperIRF,shocklengthIRF,asimposIRF(lengthIRF,numsimIRF,2),&
    ssimposIRF(lengthIRF,numsimIRF,2),anum,snum,sinit,singleshock
double precision :: ssimshocksIRF(lengthIRF,numsimIRF),asimshocksIRF(lengthIRF,numsimIRF),&
    pr_mat_a(anum,anum,snum),pr_mat_s(snum,snum)

!other declarations
integer :: t,simct,perct,sprimect,ct,aprimect,dropflag,verct
double precision :: ssimgrid(snum),asimgrid(anum)

!initialize the simulation for agg prod - unc is initialized below
asimposIRF(1,:,:) = ainit
ssimposIRF(1,:,:) = sinit



!!!START HERE IN THE SINGLE SHOCK CASE
if (singleshock==1) then 
do verct=1,2 !count over the 1 = no unc, 2 = yes unc versions
do simct=1,numsimIRF; !count IRF experiments
do t=2,lengthIRF; !count periods
    
    if ((t==shockperIRF).and.(verct==2)) then !this is the period of high uncertainty shock
        
        ssimposIRF(t,simct,verct) = 2
        
    else !this means either before or after the shock: unc should evolve normally
    
        sprimect=0; !this will store new value
    
        !create vector of cumulative probability thresholds from transition mat
        ssimgrid(1) = pr_mat_s(ssimposIRF(t-1,simct,verct),1)
        do ct=2,snum
            ssimgrid(ct) = ssimgrid(ct-1) + pr_mat_s(ssimposIRF(t-1,simct,verct),ct)
        end do !ct

        !compare the uniform shock to the thresholds, and decide which bin
        if (ssimshocksIRF(t,simct)<ssimgrid(1)) then
            sprimect = 1
        else 
            do ct=2,snum
                if (ssimgrid(ct-1)<=ssimshocksIRF(t,simct).and.ssimshocksIRF(t,simct)<ssimgrid(ct)) then
                    sprimect = ct
                end if
            end do !ct
        end if

        !fill in the simulated unc position
        ssimposIRF(t,simct,verct) = sprimect
    
    end if
end do !t
end do !simct
end do !verct
end if !singleshock == 1

!now, simulate the agg prod process, according to the simulated unc process from above
do verct=1,2
do simct=1,numsimIRF
do t=2,lengthIRF; !note that this counter is over all the periods, not differentiating across simulations
    aprimect=0; !this will store the new value of agg prod
    
    !create vector of cumulative probability thresholds from transition mat
    !note that the third entry takes uncertainty into account
    asimgrid(1) = pr_mat_a(asimposIRF(t-1,simct,verct),1,ssimposIRF(t-1,simct,verct))
    do ct=2,anum
        asimgrid(ct) = asimgrid(ct-1) + pr_mat_a(asimposIRF(t-1,simct,verct),ct,ssimposIRF(t-1,simct,verct))
    end do !ct
    
    !compare the uniform shock to the thresholds, and decide which bin
    if (asimshocksIRF(t,simct)<asimgrid(1)) then
        aprimect = 1
    else 
        do ct=2,anum
            if (asimgrid(ct-1)<=asimshocksIRF(t,simct).and.asimshocksIRF(t,simct)<asimgrid(ct)) then
                aprimect = ct
            end if
        end do !ct
    end if
    
    !fill in the simulated unc position
    asimposIRF(t,simct,verct) = aprimect
end do !t
end do !simct
end do !verct

end subroutine uncexogsimIRF

subroutine uncexogsim(ssimshocks,asimshocks,ssimpos,asimpos,pr_mat_s,pr_mat_a,ainit,sinit,snum,anum,numper)
implicit none

!this subroutine simulates the uncertainty and aggregate productivity processes

!input/output declarations
integer :: numper,ainit,sinit,snum,anum,ssimpos(numper),asimpos(numper)
double precision :: ssimshocks(numper),asimshocks(numper),pr_mat_s(snum,snum),&
    pr_mat_a(anum,anum,snum)
    
!other declarations
integer :: t,ct,sprimect,aprimect
double precision :: ssimgrid(snum),asimgrid(anum)

!initialize the simulation for uncertainty and for agg prod
ssimpos(1) = sinit
asimpos(1) = ainit

!now simulate the uncertainty process
do t=2,numper
    sprimect=0; !this will store new value
   
    !create vector of cumulative probability thresholds from transition mat
    ssimgrid(1) = pr_mat_s(ssimpos(t-1),1)
    do ct=2,snum
        ssimgrid(ct) = ssimgrid(ct-1) + pr_mat_s(ssimpos(t-1),ct)
    end do !ct

    !compare the uniform shock to the thresholds, and decide which bin
    if (ssimshocks(t)<ssimgrid(1)) then
        sprimect = 1
    else 
        do ct=2,snum
            if (ssimgrid(ct-1)<=ssimshocks(t).and.ssimshocks(t)<ssimgrid(ct)) then
                sprimect = ct
            end if
        end do !ct
    end if

    !fill in the simulated unc position
    ssimpos(t) = sprimect
end do !t

!given the uncertainty process, simulate the aggregate productivity process
do t=2,numper
    aprimect=0; !this will store the new value of agg prod
    
    !create vector of cumulative probability thresholds from transition mat
    !note that the third entry takes uncertainty into account
    asimgrid(1) = pr_mat_a(asimpos(t-1),1,ssimpos(t-1))
    do ct=2,anum
        asimgrid(ct) = asimgrid(ct-1) + pr_mat_a(asimpos(t-1),ct,ssimpos(t-1))
    end do !ct
    
    !compare the uniform shock to the thresholds, and decide which bin
    if (asimshocks(t)<asimgrid(1)) then
        aprimect = 1
    else 
        do ct=2,anum
            if (asimgrid(ct-1)<=asimshocks(t).and.asimshocks(t)<asimgrid(ct)) then
                aprimect = ct
            end if
        end do !ct
    end if
    
    !fill in the simulated unc position
    asimpos(t) = aprimect
end do !t
end subroutine uncexogsim

double precision function y(zval,aval,kval,lval,alpha,nu)
implicit none

!this function evaluates output given parameters and states

double precision :: zval,aval,kval,lval,alpha,nu

y = zval * aval * (kval ** alpha) * (lval ** nu)
    
end function

double precision function ACk(zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak)
implicit none

!this function evaluates capital AC given parameters and states

!input parameters
double precision :: zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak

!other parameters
double precision :: yval,changetol,ival

changetol = 1.0e-10
ival = kprimeval - (1.0-deltak) * kval

!start the AC at 0
ACk = 0.0

!take out the partial irreversibility costs
if (ival<-changetol) then
    ACk = ACk - ival * capirrev
end if

!take out the fixed disruption costs
if (abs(ival)>changetol) then
    yval = y(zval,aval,kval,lval,alpha,nu)
    ACk = ACk + yval * capfix
end if

end function ACk

double precision function ACl(zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wval)
implicit none

!this function evaluates labor AC given parameters and states

!input parameters
double precision :: zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wval

!other parameters
double precision :: hval,changetol,yval

changetol = 1.0e-10

hval = lval - (1.0-deltan)*lmin1val

!take out the fixed and linear costs if you changed costs
ACl = 0.0
if (abs(hval)>changetol) then
    yval = y(zval,aval,kval,lval,alpha,nu)
    ACl = ACl + labfix * yval; !the fixed costs
    if (hval<-changetol) ACl = ACl - hval * wval * firelin; !the linear firing costs
    if (hval>changetol) ACl = ACl + hval * wval * hirelin; !the linear hiring costs
end if
end function ACl

end program RUBC_ge
