!!!!!!
!RUBC_Brock_Mirman.f90
!
!This program solves, simulates, and computes IRFs for the 
!Brock-Mirman model.
!
!"Really Uncertain Business Cycles"
!Nick Bloom, Max Floetotto, Nir Jaimovich, Itay Saporta-Eksten, Stephen J. Terry
!
!This Version: October 12, 2014
!!!!!!
module fixedparam
implicit none

double precision, parameter :: alpha = 0.33,&
    deltak = 0.09,&
    beta = 0.95, &
    rhoa = 0.9,&
    sigmaa = 0.025,&
    sigma = 1,&
    nstdeva = 3.0,&
    kmin = 2.5,&
    kmax = 4.9,&
    errtol = 1e-4,&
    disttol = 1e-9,&
    shockprob = 1.0
    
integer, parameter :: checkbounds = 1,&
    GEerrorswitch = 4,&
    anum = 25,&
    knum = 350,&
    numper = 5000,&
    numdiscard = 500,&
    polmaxit=50,&
    accelmaxit = 200,&
    distmaxit=5000,&
    numsimIRF=75000,&
    lengthIRF = 100,&
    shockperIRF = 25,&
    shockval = 120,&
    IRFswitch=2

end module fixedparam

program RUBC_Brock_Mirman
use base_lib
use omp_lib
use fixedparam
implicit none

double precision :: aval,kval,kprimeval,cval,uval,yval,start,vferr,polerr,disterr

double precision, allocatable :: asimshocks(:),k0(:),pr_mat_a(:,:),a0(:),RETURNMAT(:,:,:),V(:,:),Vold(:,:),&
    RHSvec(:),EVmat(:,:),kprimepol(:,:),Ksim(:),asim(:),asimgrid(:),Ysim(:),Csim(:),Isim(:),dist(:,:),distold(:,:),&
    adist(:),kdist(:),asimIRF(:,:),asimshocksIRF(:,:),KsimIRF(:,:),IRFshocks(:),YsimIRF(:,:),CsimIRF(:,:),IsimIRF(:,:)

integer :: seeddim,seedint,ct,act,kct,aprimect,kprimect,polct,accelct,polstar,t,distct,simct

integer, allocatable :: seedarray(:),kprime(:,:),kprimeold(:,:),asimpos(:),Ksimpos(:),asimposIRF(:,:),KsimposIRF(:,:)

start = omp_get_wtime()
write(*,*) "######################################"
write(*,*) "Starting program for simple RBC model."
write(*,*) " "

seedint = 2501; !random number seed
call random_seed(size=seeddim)

allocate(seedarray(seeddim),asimshocks(numper),k0(knum),pr_mat_a(anum,anum),a0(anum),&
    RETURNMAT(anum,knum,knum),kprime(anum,knum),V(anum,knum),Vold(anum,knum),kprimeold(anum,knum),RHSvec(knum),&
    EVmat(anum,knum),kprimepol(anum,knum),asimpos(numper),Ksim(numper),asim(numper),asimgrid(anum),Ksimpos(numper),&
    Ysim(numper),Csim(numper),Isim(numper),dist(anum,knum),distold(anum,knum),adist(anum),kdist(knum),&
    asimIRF(numsimIRF,lengthIRF),asimshocksIRF(numsimIRF,lengthIRF),KsimIRF(numsimIRF,lengthIRF),&
    asimposIRF(numsimIRF,lengthIRF),KsimposIRF(numsimIRF,lengthIRF),IRFshocks(numsimIRF),YsimIRF(numsimIRF,lengthIRF),&
    CsimIRF(numsimIRF,lengthIRF),IsimIRF(numsimIRF,lengthIRF))

!seed the random numbers
do ct=1,seeddim
    seedarray(ct) = seedint+ct
end do !ct
call random_seed(put=seedarray)

!draw the random numbers
call random_number(asimshocks); !U(0,1) in each cell of asimshocks
call random_number(asimshocksIRF); !U(0,1) in each cell of asimshocksIRF
call random_number(IRFshocks); !U(0,1) in each cell of IRFshocks

!set up the capital grid
call linspace(k0,log(kmin),log(kmax),knum); k0=exp(k0);

!set up the agg prod grid
call idio_tauchen(anum,rhoa,sigmaa,pr_mat_a,a0,nstdeva)

!set up the return matrix
RETURNMAT(:,:,:) = 0.0
!RETURNMAT(a,k,k')
do act=1,anum
do kct=1,knum
do kprimect=1,knum
    
    kval = k0(kct)
    kprimeval = k0(kprimect)
    aval = a0(act)
    
    yval = yprod(aval,kval)
    
    cval = yval + (1.0-deltak)*kval - kprimeval
    if (cval<=0.0) then
        uval = -1000000000.0
    else if (cval>0.0) then
        uval = util(cval)
    end if
    
    RETURNMAT(act,kct,kprimect) = uval
    
    
end do !kprimect
end do !kct
end do !act

write(*,*) " "
write(*,*) "Finished basic setup for VFI at ",omp_get_wtime()-start," seconds."


!now, perform the VFI, using Howard acceleration

!initialize
do act=1,anum
    kprimeold(act,:) = knum/2
end do !act
Vold(:,:) = 0.0
V(:,:) = 0.0

do polct=1,polmaxit
    
    !do howard acceleration
    do accelct=1,accelmaxit
        
        do act=1,anum
        do kct=1,knum
            
            !what is policy?
            kprimect = kprimeold(act,kct)
            
            !what is current return?
            V(act,kct) = RETURNMAT(act,kct,kprimect)
            
            !add continuation value
            do aprimect=1,anum
                V(act,kct) = V(act,kct) + pr_mat_a(act,aprimect) * beta * Vold(aprimect,kprimect)
            end do !aprimect
            
        end do !kct
        end do !act
        
        !update
        Vold = V
        V(:,:) = 0.0
        
    end do !accelct
    
    !construct continuation value EVmat(a,k')
    EVmat(:,:) = 0.0
    do act=1,anum
    do kprimect=1,knum
        do aprimect=1,anum
            EVmat(act,kprimect) = EVmat(act,kprimect) + pr_mat_a(act,aprimect)*beta * Vold(aprimect,kprimect)
        end do !aprimect
    end do !kprimect
    end do !act
    
    
    !now, actually do the optimization
    do act=1,anum
    do kct=1,knum
        
        !construct RHSvec
        RHSvec(:) = 0.0
        do kprimect=1,knum
            RHSvec(kprimect) = RETURNMAT(act,kct,kprimect) + EVmat(act,kprimect)
        end do !kprimect
        
        polstar = maxloc(RHSvec,1)
        kprime(act,kct) =  polstar
        V(act,kct) = RHSvec(polstar)
        
    end do !act
    end do !act
    
    !compute errors and end if converged
    vferr = maxval(V-Vold)
    polerr = maxval(dble(kprime)-dble(kprimeold))
    
    write(*,*) "VF iteration ",polct
    write(*,*) "VF err ",vferr
    write(*,*) "Policy err ",polerr
    
    !exit if policies have converged
    if (polerr<errtol) exit
    
    !update if not converged
    Vold = V
    kprimeold = kprime
    
end do !polct

do act=1,anum
do kct=1,knum
    kprimepol(act,kct)=k0(kprime(act,kct))
end do 
end do 


!VF diagnostics
write(*,*) " "
write(*,*) "Done with VFI."
if (minval(kprime)==1) write(*,*) "Max capital policy hits bottom of grid in theory."
if (maxval(kprime)==knum) write(*,*) "Min capital policy hits top of grid in theory."


!compute unconditional distribution
write(*,*) " "
write(*,*) "Doing ergodic dist calculations."
distold(:,:) = 0.0
distold(anum/2,knum/2) = 1.0

do distct=1,distmaxit
    
    dist(:,:) = 0.0
    
    !loop over states
    do act=1,anum
    do kct=1,knum
        
        kprimect=kprime(act,kct)
        
        !pushforward of weight
        do aprimect=1,anum
            dist(aprimect,kprimect) =  dist(aprimect,kprimect) + distold(act,kct)*pr_mat_a(act,aprimect)
        end do !aprimect
        
    end do !kct
    end do !act
    
    disterr =maxval(abs(dist-distold))
    if (mod(distct,25)==1) then
        write(*,*) distct, sum(dist),disterr
    end if 
    if (disterr<disttol) exit
    
    dist = dist/sum(dist)
    distold = dist
    
    
end do !distct

!marginal dists
adist(:) = 0.0
do act=1,anum
    adist(act) = sum(dist(act,:))
end do 

kdist(:) = 0.0
do kct=1,knum
    kdist(kct) = sum(dist(:,kct))
end do 

write(*,*) "Done with ergodic dist calculations."
write(*,"(A,F7.4,A,F7.4)") "Weight on min, max of capital grid = ",kdist(1),", ",kdist(knum)

!now, perform unconditional simulation
write(*,*) " "
write(*,*) "Now, doing unconditional simulation."
asimpos(1) = anum/2
asim(1) = a0(anum/2)
Ksim(1) = k0(knum/2)
Ksimpos(1) = knum/2
do t=2,numper
    
    act=asimpos(t-1)
    
    aprimect=0; !this will store the new value
    
    !create vector of cumulative probability thresholds from the transition mat
    asimgrid(:) = 0.0
    asimgrid(1) = pr_mat_a(act,1)
    do ct=2,anum
        asimgrid(ct) = asimgrid(ct-1) + pr_mat_a(act,ct)
    end do !ct
    
    !compare U(0,1) draws to cumulative probability thresholds
    if (asimshocks(t)<asimgrid(1)) then
        aprimect=1
    else 
        do ct=2,anum
            if (asimgrid(ct-1)<=asimshocks(t).and.asimshocks(t)<asimgrid(ct)) then
                aprimect=ct
            end if
        end do !ct
    end if
    
    !fill in the simulated position for idio prod
    asimpos(t) = aprimect
        
end do !t

!endog. cap. sims
do t=1,numper-1
    
    
    act = asimpos(t)
    kct = Ksimpos(t)
    kprimect = kprime(act,kct)
    
    Ksimpos(t+1) = kprimect
    Ksim(t+1) = k0(kprimect)
end do !t

!process
do t=1,numper
    asim(t) = a0(asimpos(t))
    Ysim(t) = yprod(asim(t),Ksim(t))
    if (t<numper) Isim(t) = Ksim(t+1)-(1.0-deltak)*Ksim(t)
    if (t<numper) Csim(t) = Ysim(t) - Isim(t)
end do 


write(*,*) " "
write(*,*) "Finished unconditional simulation at ",omp_get_wtime()-start," seconds."


!now, perform conditional or IRF simulation
write(*,*) " "
write(*,*) "Now, doing IRF simulation."


do simct=1,numsimIRF
    asimposIRF(simct,1) = anum/2
    asimIRF(simct,1) = a0(anum/2)
    KsimIRF(simct,1) = k0(knum/2)
    KsimposIRF(simct,1) = knum/2
    do t=2,lengthIRF
        
        act=asimposIRF(simct,t-1)
        
        aprimect=0; !this will store the new value
        
        !create vector of cumulative probability thresholds from the transition mat
        asimgrid(:) = 0.0
        asimgrid(1) = pr_mat_a(act,1)
        do ct=2,anum
            asimgrid(ct) = asimgrid(ct-1) + pr_mat_a(act,ct)
        end do !ct
        
        !compare U(0,1) draws to cumulative probability thresholds
        if (asimshocksIRF(simct,t)<asimgrid(1)) then
            aprimect=1
        else 
            do ct=2,anum
                if (asimgrid(ct-1)<=asimshocksIRF(simct,t).and.asimshocksIRF(simct,t)<asimgrid(ct)) then
                    aprimect=ct
                end if
            end do !ct
        end if
        
        !fill in the simulated position for idio prod
        asimposIRF(simct,t) = aprimect
        
        !if it's a productivity shock, then shock productivity
        if (IRFswitch==1) then
        if (t==shockperIRF) then
            if (IRFshocks(simct)<shockprob) then
                asimposIRF(simct,t) = shockval
            end if
        end if
        end if
            
    end do !t

    !endog. cap. sims
    do t=1,lengthIRF-1
        
        
        act = asimposIRF(simct,t)
        kct = KsimposIRF(simct,t)
        
        !if it's a capital shock, then shock capital
        if (IRFswitch==2) then
            if (t==shockperIRF) then
            if (IRFshocks(simct)<shockprob) then
                kct = shockval
                KsimposIRF(simct,t) = shockval
                KsimIRF(simct,t) = k0(shockval)
            end if
            end if
        end if
        
        
        kprimect = kprime(act,kct)
        
        KsimposIRF(simct,t+1) = kprimect
        KsimIRF(simct,t+1) = k0(kprimect)
    end do !t

    !process
    do t=1,lengthIRF
        asimIRF(simct,t) = a0(asimposIRF(simct,t))
        YsimIRF(simct,t) = yprod(asimIRF(simct,t),KsimIRF(simct,t))
        
      
        if (t<lengthIRF-1) IsimIRF(simct,t) = KsimIRF(simct,t+1)-(1.0-deltak)*KsimIRF(simct,t)
        
        if (IRFswitch==2) then
        if (t==(shockperIRF-1)) then
            if (IRFshocks(simct)<shockprob) then
                kct = KsimposIRF(simct,t)
                act = asimposIRF(simct,t)
                kprimect = kprime(act,kct)
                
                IsimIRF(simct,t) = k0(kprimect)-(1.0-deltak)*k0(kct)
            end if
        end if
        end if
        
        
        if (t<lengthIRF-1) CsimIRF(simct,t) = YsimIRF(simct,t) - IsimIRF(simct,t) 
    end do 

end do !simct

write(*,*) " "
write(*,*) "Finished IRF simulation at ",omp_get_wtime()-start," seconds."


write(*,*) " "
write(*,*) "Writing output files."
!output block
open(8,file="k0.txt")
do ct=1,knum
write(8,*) k0(ct)
end do 
close(8)

open(8,file="a0.txt")
do ct=1,anum
write(8,*) a0(ct)
end do 
close(8)

open(8,file="pr_mat_a.txt")
do ct=1,anum
write(8,*) pr_mat_a(ct,:)
end do 
close(8)

open(8,file="kprime.txt")
do act=1,anum
do kct=1,knum
write(8,*) kprime(act,kct)
end do 
end do
close(8)

open(8,file="V.txt")
do act=1,anum
do kct=1,knum
write(8,*) V(act,kct)
end do 
end do
close(8)

open(8,file="constants.txt")
write(8,*) anum
write(8,*) knum
write(8,*) beta
write(8,*) deltak
write(8,*) alpha
write(8,*) numper
write(8,*) numsimIRF
write(8,*) lengthIRF
write(8,*) shockperIRF
close(8)

open(8,file="asimpos.txt")
do t=1,numper
write(8,*) asimpos(t)
end do !t
close(8)

open(8,file="asim.txt")
do t=1,numper
write(8,*) asim(t)
end do !t
close(8)

open(8,file="Ksimpos.txt")
do t=1,numper
write(8,*) Ksimpos(t)
end do !t
close(8)

open(8,file="Ksim.txt")
do t=1,numper
write(8,*) Ksim(t)
end do !t
close(8)

open(8,file="Csim.txt")
do t=1,numper
write(8,*) Csim(t)
end do !t
close(8)

open(8,file="Ysim.txt")
do t=1,numper
write(8,*) Ysim(t)
end do !t
close(8)


open(8,file="Isim.txt")
do t=1,numper
write(8,*) Isim(t)
end do !t
close(8)

open(8,file="dist.txt")
do act=1,anum
do kct=1,knum
write(8,*) dist(act,kct)
end do
end do
close(8)

open(8,file="adist.txt")
do act=1,anum
write(8,*) adist(act)
end do !act
close(8)

open(8,file="kdist.txt")
do kct=1,knum
write(8,*) kdist(kct)
end do !kct
close(8)

open(8,file="asimposIRF.txt")
do simct=1,numsimIRF
do t=1,lengthIRF
write(8,*) asimposIRF(simct,t)
end do !t
end do
close(8)

open(8,file="asimIRF.txt")
do simct=1,numsimIRF
do t=1,lengthIRF
write(8,*) asimIRF(simct,t)
end do !t
end do
close(8)

open(8,file="KsimposIRF.txt")
do simct=1,numsimIRF
do t=1,lengthIRF
write(8,*) KsimposIRF(simct,t)
end do !t
end do
close(8)

open(8,file="KsimIRF.txt")
do simct=1,numsimIRF
do t=1,lengthIRF
write(8,*) KsimIRF(simct,t)
end do !t
end do
close(8)

open(8,file="CsimIRF.txt")
do simct=1,numsimIRF
do t=1,lengthIRF
write(8,*) CsimIRF(simct,t)
end do !t
end do
close(8)

open(8,file="YsimIRF.txt")
do simct=1,numsimIRF
do t=1,lengthIRF
write(8,*) YsimIRF(simct,t)
end do !t
end do
close(8)

open(8,file="IsimIRF.txt")
do simct=1,numsimIRF
do t=1,lengthIRF
write(8,*) IsimIRF(simct,t)
end do !t
end do
close(8)


write(*,*) " "
write(*,*) "Finished program for the Brock-Mirman model at ",omp_get_wtime()-start," seconds."
write(*,*) "######################################"




contains

double precision function yprod(aval,kval)
implicit none

!this function evaluates output given parameters and states

double precision :: aval,kval

yprod =  aval * (kval ** alpha) 
    
end function

double precision function util(cval)
implicit none

!this function evaluates the utility function

double precision :: cval

!util =  (cval**(1.0-sigma))/(1.0-sigma)
util = log(cval)
    
end function


end program RUBC_Brock_Mirman