##################################
# boot.R
#
# Uses micro and macro uncertainty series created in 
# MATLAB program moments_data.m to compute the bootstrapped
# covariance matrix of the mean, standard deviation, skewness
# and serial correlation of the IQR of Census TFP shocks and 
# the GARCH(1,1)-implied conditional heteroskedasticity of the
# change in the quarterly aggregate Solow residual from John
# Fernald's website.  The asymptotic covariance matrix is saved
# in the text file SIGMA.txt and the moments and standard errors
# are saved in MOMVEC_SE.txt.
#
# "Really Uncertain Business Cycles"
# Nick Bloom, Max Floetotto, Nir Jaimovich, Itay Saporta-Eksten, Stephen J. Terry
#
# This Version: Dec 28, 2013
##################################

rm(list=ls())
library(boot)
library(moments)
library(matlab)

set.seed(2501)

numboots = 50000

#read in  micro data
MICRODATA = scan('MICRODATA.txt')

#read in macro data
MACRODATA = scan('MACRODATA.txt')
MACRODATA = MACRODATA[5:length(MACRODATA)]; 

#compute moments function
momvec = function(ONESERIES){
  
  momvec = c(0.0,0.0,0.0,0.0)
  
  n = length(ONESERIES)[1]
  
  momvec[1] = mean(ONESERIES,na.rm=TRUE)
  momvec[2] = var(ONESERIES,na.rm=TRUE)^0.5
  momvec[3] = skewness(ONESERIES,na.rm=TRUE)
  momvec[4] = corr(cbind(ONESERIES[1:(n-1)],ONESERIES[2:n]))
  return(momvec)
  
}

bootmomvec = function(DATA){
  
  Tyrs = dim(DATA)[1]
  Tqtr = 4*Tyrs

  MICRO = DATA[,1]

  MACRO = zeros(Tqtr,1)
  ct = 0
  for (yr in 1:Tyrs){
  for (qtr in (1:4)){
    ct = ct+1
    MACRO[ct] = DATA[yr,qtr+1]
  }}

  bootmomvec = zeros(8,1)
  bootmomvec[1:4] = momvec(MICRO)
  bootmomvec[5:8] = momvec(MACRO)
  return(bootmomvec)
}

#how many periods and block length suggested by Politis and Romano (1994)
Tyrs = length(MICRODATA)
meanblocklength = ceiling(Tyrs^(1/3))

#insert data into stacked form
DATA = zeros(Tyrs,5)
DATA[,1] = MICRODATA
ct = 0
for (yr in 1:Tyrs){
  for (qtr in (1:4)){
    ct = ct+1
    DATA[yr,qtr+1] = MACRODATA[ct]
  }}


#do bootstrapped statistics
BOOTSTATS = tsboot(DATA,bootmomvec,numboots,sim="geom",l=meanblocklength)$t

#compute covariance matrix function
covmat = function(DATA){
  
  n = dim(DATA)[2]
  T = dim(DATA)[1]
  
  covmat = array(0,dim=c(n,n))
  
  #compute 2nd-moment array
  for (t in (1:T)){
    
    for (i in 1:n){
      for (j in 1:n){
        covmat[i,j] = covmat[i,j] + DATA[t,i]*DATA[t,j]
        
      }
    }
    
  }
  
  covmat = (1/T)*covmat
  
  #compute means
  meanvec = array(0,dim=c(n,1))
  for (i in 1:n){meanvec[i] = mean(DATA[,i],na.rm=TRUE)}
  
  #adjust from 2nd-moments to var/cov
  for (i in 1:n){
    for (j in 1:n){
        covmat[i,j] = covmat[i,j] - meanvec[i]*meanvec[j]
       
    }
  }
  
  return(covmat)
  
}

#bootstrapped covariance matrices for micro and macro 
SIGMA = Tyrs*covmat(BOOTSTATS)

#data moment vec
MOMVEC = bootmomvec(DATA)
sink(file="MOMVEC_SE.txt")
print("MOMENTS (SE)")
print(cbind(MOMVEC,diag(SIGMA/Tyrs)^0.5))
sink(file=NULL)
sink(file=NULL)

write.table(SIGMA,file="SIGMA.txt",col.names=FALSE,row.names=FALSE)