function MICROMOM(nfirms,numyrs,numyrdiscard)
%%%%%%
%MICROMOM.m
%
%This program takes as direct inputs dimensions for the unconditional 
%simulation of the firm panel from the model (at annual frequency).  The 
%then reads the tfp of the panel from the Fortran output, and performs the 
%panel regression and computes the moments from the IQR of the micro
%regressions.
%
%"Really Uncertain Business Cycles"
%Nick Bloom, Max Floetotto, Nir Jaimovich, Itay Saporta-Eksten, Stephen J. Terry
%
%This Version: Dec 9, 2013
%%%%%%%

%seed random number generator
rng(2501);

%delete previous files if they exist
delete MICROMOM.txt; delete IQR.txt;

%read in the firm tfp panel from Fortran output file
tfpannualvec = importdata('tfpannual.txt');

%declare correctly dimensioned array and transform
tfpannual=zeros(nfirms,numyrs);
ct=0;
for firmct=1:nfirms;
    for t=1:numyrs;
        ct=ct+1;
        tfpannual(firmct,t) = tfpannualvec(ct);
    end;
end;

%now, discard the burn-in periods
tfpannual = tfpannual(:,(numyrdiscard+1):numyrs);
numyrs = numyrs-numyrdiscard;

%now, create the regression arrays, for 
%tfp_{jt+1} = alpha_j + beta_t + rho * tfp_{jt} + eps_{jt}
X = zeros(nfirms * (numyrs-1),nfirms + (numyrs-2) + 1);
Y = zeros(nfirms * (numyrs-1),1);

%create LHS array
Y = reshape(tfpannual(:,2:numyrs)',nfirms*(numyrs-1),1);

%insert firm dummies into the regressor array
X(:,1:nfirms) = kron(eye(nfirms),ones(numyrs-1,1));

%insert the time dummies into the regressor array
X(:,(nfirms+1):(nfirms+numyrs-2)+1)= kron(ones(nfirms,1),eye(numyrs-1));
X(:,nfirms + (numyrs-2) + 1) = 0.0;

%insert the lagged tfp into the regressor array
X(:,end) = reshape(tfpannual(:,1:(numyrs-1))',nfirms*(numyrs-1),1);

%OLS coefficients in thetahat
thetahat = (X'*X) \ (X'*Y);

%residuals
Yhat = X * thetahat;
Resids = Y - Yhat;

%compute measurement error standard deviation
mestd = std(Resids(:));

ResidsnoME = reshape(Resids,(numyrs-1),nfirms);

%what is the cross-sectional IQR of the resids for each time period?
IQRnoME=iqr(ResidsnoME,2); %numyrs-1 x 1
MICROMOMnoME=zeros(4,1);
MICROMOMnoME(1) = mean(IQRnoME);
MICROMOMnoME(2) = std(IQRnoME);
MICROMOMnoME(3) = skewness(IQRnoME);
MICROMOMnoME(4) = corr(IQRnoME(2:end),IQRnoME(1:(end-1)));

%now, redo the analysis after adding white noise to the tfp firm-level data
me = randn(size(tfpannual))*mestd;
tfpannual = tfpannual + me;

%clear previous regression arrays
clear X Y thetahat Yhat Resids;

%reform initial regression arrays
X = zeros(nfirms * (numyrs-1),nfirms + (numyrs-2) + 1);
Y = zeros(nfirms * (numyrs-1),1);

%create LHS array
Y = reshape(tfpannual(:,2:numyrs)',nfirms*(numyrs-1),1);

%insert firm dummies into the regressor array
X(:,1:nfirms) = kron(eye(nfirms),ones(numyrs-1,1));

%insert the time dummies into the regressor array
X(:,(nfirms+1):(nfirms+numyrs-2)+1)= kron(ones(nfirms,1),eye(numyrs-1));
X(:,nfirms + (numyrs-2) + 1) = 0.0;

%insert the lagged tfp into the regressor array
X(:,end) = reshape(tfpannual(:,1:(numyrs-1))',nfirms*(numyrs-1),1);

%OLS coefficients in thetahat
thetahat = (X'*X) \ (X'*Y);

%residuals
Yhat = X * thetahat;
Resids = Y - Yhat;

%Resids is nfirms * (numyrs-1) column of residuals, need to reshape to
%(numyrs-1) * nfirms  array for cross-sectional moment calculation
Resids = reshape(Resids,(numyrs-1),nfirms);

%what is the cross-sectional IQR of the resids for each time period?
IQR=iqr(Resids,2); %numyrs-1 x 1

%compute the micro moments
MICROMOM=zeros(4,1);
MICROMOM(1) = mean(IQR);
MICROMOM(2) = std(IQR);
MICROMOM(3) = skewness(IQR);
MICROMOM(4) = corr(IQR(2:end),IQR(1:(end-1)));

IQR_str = ['IQR.txt'];
MICROMOM_str = ['MICROMOM.txt'];

save(IQR_str,'IQR','-ascii');
save(MICROMOM_str,'MICROMOM','-ascii');


end