program RUBC_pe
use base_lib
use omp_lib
implicit none

!!!!!!
!RUBC_pe.f90
!
!This program solves, simulates, and computes IRFs for the PE version of the RUBC model.
!
!"Really Uncertain Business Cycles"
!Nick Bloom, Max Floetotto, Nir Jaimovich, Itay Saporta-Eksten, Stephen J. Terry
!
!This Version: Nov 18, 2013
!!!!!!

!!!!DECLARATION BLOCK

character(len=*), parameter :: basedir="."

integer :: znum,anum,snum,knum,lnum,numstates,numexog,numendog,numper,numdiscard,seedint,vfmaxit,&
    accelmaxit,numconstants,ct,zct,act,sct,kct,lmin1ct,zprimect,aprimect,sprimect,primect,endogct,&
    exogct,polct,vfct,accelct,polstar,exogprimect,t,numsimIRF,lengthIRF,shockperIRF,numperIRF,&
    numdiscIRF,ainit,sinit,seeddim
        
integer, allocatable :: exog_pos(:,:),endog_pos(:,:),loopind(:,:),polmat(:,:),polmatold(:,:),&
    loopRind(:,:),asimpos(:),ssimpos(:),asimposIRF(:),ssimposIRF(:),seedarray(:)

double precision :: start,finish,alpha,nu,theta,deltak,deltan,beta,capirrev,capfix,hirelin,firelin,&
    labfix,pval,wval,ajump,zjump,uncpers,uncfreq,rhoz,sigmaz,rhoa,sigmaa,zmin,zmax,amin,amax,kmin,kmax,&
    lmin,lmax,vferrortol,zval,aval,sval,kval,lmin1val,kprimeval,lval,ival,yval,ackval,aclval,vferror,&
    polerror,hval,changetol

double precision, allocatable :: V(:,:),Rmat(:,:,:),asimshocks(:),ssimshocks(:),pr_mat_z(:,:,:),&
    pr_mat_a(:,:,:),pr_mat_s(:,:),pr_mat(:,:),z0(:),k0(:),l0(:),a0(:),sigmazgrid(:),sigmaagrid(:),&
    MATLABconstants(:),exog_key(:,:),endog_key(:,:),EVmat(:,:),Vold(:,:),RHSvec(:),kprime_key(:,:),&
    lpol_key(:,:),kprime_pos(:,:),lpol_pos(:,:),distzkl(:,:,:),Ysim(:),Ksim(:),Lsim(:),Isim(:),Hsim(:),&
    ACksim(:),AClsim(:),TOPksim(:),BOTksim(:),TOPlsim(:),BOTlsim(:),ADJksim(:),ADJlsim(:),asimshocksIRF(:),&
    ssimshocksIRF(:),distzklIRF(:,:,:),YsimIRF(:),KsimIRF(:),LsimIRF(:),IsimIRF(:),HsimIRF(:),ACksimIRF(:),&
    AClsimIRF(:),ADJksimIRF(:),ADJlsimIRF(:)

!!!!END DECLARATION BLOCK

!!!!PROGRAM SETUP BLOCK

!call the clock to get the start time and start writing to the log file
start = omp_get_wtime()
open(13,file=basedir//"\RUBC_pe.txt")

!these lines simply space the log file and shell view
write(*,*) "OpenMP parallelization setup"
write(13,*) "OpenMP parallelization setup"

!figure out parallelization options with current cpu setup
write(*,*) "available threads = ",omp_get_max_threads();
write(13,*) "available threads = ",omp_get_max_threads();
call omp_set_num_threads(omp_get_max_threads())

!NOTE: all writing commands are duplicated - once to the default * channel, which 
!is the shell you view in runtime, and once to channel 13, which is the log file

!test the parallelization to see how many threads this system can use
!$omp parallel
write(*,*) "Parallel hello to you!"
write(13,*) "Parallel hello to you!"
!$omp end parallel

write(*,*) " "
write(13,*) " "

!!!!END PROGRAM SETUP BLOCK

!!!!!PARAMETER ENTRY BLOCK

!technology and adjustment costs
alpha = 0.25
nu = 0.5
theta = 2.0
deltak = 0.026
deltan = 0.088
beta = 0.95 ** 0.25
capirrev = 0.339; 
capfix = 0.0
hirelin = 0.018*4.0; 
firelin = hirelin
labfix = 0.024*4;

!grid sizes for exog processes
znum = 5; !idio prod
anum = 5; !agg prod
snum = 2; !volatility

!grid sizes for endog processes
!knum = 100
!lnum = 45
knum = 50
lnum = 25

!PE price and wage parameters
pval = 1.4; wval = theta/pval;

!set up the unc process
ajump = 1.6060678164419682; !multiple of vol in agg case
zjump = 4.1365844523734641; !multiple of vol in idio case
uncpers = 0.94296716663676217; !cond. unc shock persistence
uncfreq = 0.025857201385718384; !cond. prob of unc shock

!set up the idio prod process
rhoz = 0.95; 
sigmaz = 0.050752106584379765
zmin = exp( - 2.0 * ( ( sigmaz ** 2.0 ) / ( 1.0 - rhoz ** 2.0 ) ) ** 0.5 );
zmax = exp( 2.0 * ( ( sigmaz ** 2.0 ) / ( 1.0 - rhoz ** 2.0 ) ) ** 0.5);

!set up the agg prod process
rhoa = 0.95;
sigmaa = 0.0066771142154949305;
amin = exp( - 2.0 * ( ( sigmaa ** 2.0 ) / ( 1.0 - rhoa ** 2.0 ) ) ** 0.5);
amax = exp( 2.0 * ( ( sigmaa ** 2.0 ) / ( 1.0 - rhoa ** 2.0 ) ) ** 0.5);

!set up the idio capital grid, respecting depreciation
kmin = 1.125
kmax = exp(log(kmin) - dble(knum-1) * log(1.0-deltak))
write(*,"(A,F10.3,A,F10.3,A)") "capital grid = (",kmin,",",kmax,")"
write(13,"(A,F10.3,A,F10.3,A)") "capital grid = (",kmin,",",kmax,")"

!set up the idio labor grid, respecting depreciation
lmin = 0.035
lmax = exp(log(lmin) - dble(lnum-1) * log(1.0-deltan))
write(*,"(A,F10.3,A,F10.3,A)") "labor grid = (",lmin,",",lmax,")"
write(13,"(A,F10.3,A,F10.3,A)") "labor grid = (",lmin,",",lmax,")"

!figure out total number of states
numexog = znum*anum*snum
numendog = knum*lnum
numstates = numexog * numendog

!control the uncondtional simulation
numper = 5000   
numdiscard = 500
seedint = 2501; call random_seed(size=seeddim)
ainit = 3; !initial agg prod val (uncond sim and IRF)
sinit = 1; !initial unc val (uncond sim only, 1 = low, 2 = high)

!control the IRF
numsimIRF = 2700; !number of shocked economies to simulate
lengthIRF = 100; !length of each economy
shockperIRF = 45; !period in which to shock the economy
numdiscIRF = 200; !number of economies to discard

numperIRF = numsimIRF * lengthIRF; !total number of periods in IRF simulation

!control the vf process
vfmaxit=55; !number of VF iterations
vferrortol = 1e-4; !VF error tolerance
accelmaxit = 200; !number of Howard acceleration iterations

!control what is considered a change
changetol = 1e-10; !what is a change for the purposes of AC functions?

!how many parameters to pass to MATLAB
numconstants = 16; !this is relevant for input/output only in pass to MATLAB

!!!!END PARAMETER ENTRY BLOCK

!!!!SOME INITIALIZATIONS/GRID SETUPS

!do allocations for all variable-dimension arrays
allocate(k0(knum),l0(lnum),z0(znum),a0(anum),pr_mat_z(znum,znum,snum),pr_mat_a(anum,anum,snum),pr_mat_s(snum,snum),&
    sigmaagrid(snum),sigmazgrid(snum),MATLABconstants(numconstants),exog_key(numexog,3),exog_pos(numexog,3),endog_key(numendog,2),&
    endog_pos(numendog,2),V(numendog,numexog),Rmat(numendog,numexog,numendog),pr_mat(numexog,numexog),EVmat(numexog,numendog),&
    loopind(numstates,2),polmat(numendog,numexog),Vold(numendog,numexog),polmatold(numendog,numexog),&
    loopRind(numstates*numendog,3),RHSvec(numendog),kprime_key(numendog,numexog),lpol_key(numendog,numexog),&
    kprime_pos(numendog,numexog),lpol_pos(numendog,numexog),asimpos(numper),ssimpos(numper),distzkl(znum,numendog,numper),&
    asimshocks(numper),ssimshocks(numper),Ysim(numper),Ksim(numper),Lsim(numper),Isim(numper),Hsim(numper),ACksim(numper),&
    AClsim(numper),TOPksim(numper),BOTksim(numper),TOPlsim(numper),BOTlsim(numper),ADJksim(numper),ADJlsim(numper),&
    asimposIRF(numperIRF),ssimposIRF(numperIRF),asimshocksIRF(numperIRF),ssimshocksIRF(numperIRF),&
    distzklIRF(znum,numendog,numperIRF),YsimIRF(numperIRF),KsimIRF(numperIRF),LsimIRF(numperIRF),IsimIRF(numperIRF),&
    HsimIRF(numperIRF),ACksimIRF(numperIRF),AClsimIRF(numperIRF),ADJksimIRF(numperIRF),ADJlsimIRF(numperIRF),seedarray(seeddim))

!set up the idiosyncratic capital and labor grids, using the subroutine linspace which replicates MATLAB function
call linspace(k0,log(kmin),log(kmax),knum); k0=exp(k0);
call linspace(l0,log(lmin),log(lmax),lnum); l0=exp(l0);

!set up the sigma or volatility grids and transition matrix
sigmazgrid = (/ sigmaz , zjump*sigmaz /)
sigmaagrid = (/ sigmaa , ajump*sigmaa /)
pr_mat_s(1,:) = (/1.0 - uncfreq , uncfreq /); pr_mat_s(1,:) = pr_mat_s(1,:)/sum(pr_mat_s(1,:))
pr_mat_s(2,:) = (/1.0 - uncpers , uncpers /); pr_mat_s(2,:) = pr_mat_s(2,:)/sum(pr_mat_s(2,:))

!compute the idio & aggr transition matrices, via Tauchen (1986) adapted to stochastical vol case
call unctauchen(pr_mat_z,z0,znum,zmin,zmax,snum,sigmazgrid,rhoz)
call unctauchen(pr_mat_a,a0,anum,amin,amax,snum,sigmaagrid,rhoa)

!index the exogenous variables (z,a,s) & create the unified transition matrix over exogenous variables
ct=0
do zct=1,znum; do act=1,anum; do sct=1,snum
    ct=ct+1;
    
    !indexing the variables
    exog_pos(ct,1) = zct; exog_key(ct,1) = z0(zct); !insert z
    exog_pos(ct,2) = act; exog_key(ct,2) = a0(act); !insert a
    exog_pos(ct,3) = sct; exog_key(ct,3) = dble(sct); !insert s
    
    !creating the transition matrix
    primect=0
    do zprimect=1,znum; do aprimect=1,anum; do sprimect=1,snum
        primect=primect+1
        pr_mat(ct,primect) = pr_mat_z(zct,zprimect,sct) * pr_mat_a(act,aprimect,sct) * pr_mat_s(sct,sprimect)
    end do; end do; end do;
    
    !correction for roundoff error
    pr_mat(ct,:) = pr_mat(ct,:) / sum( pr_mat(ct,:) )
    
end do; end do; end do;

!index the endog variables (k,l_{-1}), noting that the index is used for states and policies
ct=0
do kct=1,knum; do lmin1ct=1,lnum
    ct=ct+1
    endog_pos(ct,1) = kct; endog_key(ct,1) = k0(kct); !insert k
    endog_pos(ct,2) = lmin1ct; endog_key(ct,2) = l0(lmin1ct); !insert l_{-1}
end do; end do;

!indexing for parallelization over V: parallel loops will go over "ct," with
!each thread pulling out endogct or exogct
ct=0
do endogct=1,numendog
do exogct=1,numexog
    ct=ct+1
    loopind(ct,1) = endogct; loopind(ct,2) = exogct
end do !exogct
end do !endogct

!indexing for parallelization over Rmat: similar to above, since parallel loops 
!will go over the "ct" dimension
ct=0
do endogct=1,numendog
do exogct=1,numexog
do polct=1,numendog    
    ct=ct+1
    loopRind(ct,1) = endogct; loopRind(ct,2) = exogct; loopRind(ct,3) = polct
end do !polct
end do !exogct
end do !endogct

!create the current period return array - right now, this is more memory intense
!(factor of snum) than it needs to be, which makes the VFI code elegant but can be relaxed
Rmat(:,:,:) = 0.0

write(*,*) " "
write(13,*) " "

write(*,*) "Doing return matrix setup."
write(13,*) "Doing return matrix setup."

!set up the parallelization for the creation of Rmat, noting the use of the OpenMP 
!"pragmas" starting with "!$omp".  Need to declare which values/arrays are private
!$omp parallel private(ct,endogct,exogct,polct,zval,aval,sval,kval,lmin1val,&
!$omp& kprimeval,lval,yval,ackval,aclval,ival)

!ct is the parallel loop index to create Rmat, as signified by the pragma command
!$omp do
do ct=1,numendog*numexog*numendog
    
    !extract states and policy choices
    endogct = loopRind(ct,1); exogct=loopRind(ct,2); polct=loopRind(ct,3);
    
    !extract exog variables
    zval = exog_key(exogct,1)
    aval = exog_key(exogct,2)
    sval = exog_key(exogct,3)
    
    !extract endog variables
    kval = endog_key(endogct,1)
    lmin1val = endog_key(endogct,2)
    
    !extract policies
    kprimeval = endog_key(polct,1)
    lval = endog_key(polct,2)
    
    !determine RHS stuff
    yval = y(zval,aval,kval,lval,alpha,nu)
    ackval = ACk(zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak)
    aclval = ACl(zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wval)
    ival = kprimeval - (1.0-deltak) * kval
    
    !input the return value into the array
    Rmat(endogct,exogct,polct) = yval - ackval - aclval - ival - wval*lval
    Rmat(endogct,exogct,polct) = pval * Rmat(endogct,exogct,polct)

end do !ct
!$omp end do
!$omp end parallel
!parallel execution stops in the previous line

write(*,"(A,F15.1,A)") "Done with return matrix setup in ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Done with return matrix setup in ",omp_get_wtime()-start," seconds."  

!call the random numbers
do ct=1,seeddim
    seedarray(ct) = seedint+ct
end do !ct
call random_seed(put=seedarray); !this seeds the draws
call random_number(asimshocks); !U(0,1) in each cell of asimshocks
call random_number(ssimshocks); !U(0,1) in each cell of ssimshocks

!write the random numbers
open(8,file="asimshocks.txt")
do ct=1,numper
write(8,*) asimshocks(ct)
end do 
close(8)

open(8,file="ssimshocks.txt")
do ct=1,numper
write(8,*) ssimshocks(ct)
end do 
close(8)

!based on the random shocks drawn, simulate the aggregate exogenous processes
call uncexogsim(ssimshocks,asimshocks,ssimpos,asimpos,pr_mat_s,pr_mat_a,ainit,sinit,snum,anum,numper)

!first, call the random draws for unc and agg prod simulation
call random_number(asimshocksIRF); !U(0,1) in each cell of asimshocksIRF
call random_number(ssimshocksIRF); !U(0,1) in each cell of ssimshocksIRF

!then, based on these shocks, simulate the IRF unc and agg prod processes
call uncexogsimIRF(ssimshocksIRF,asimshocksIRF,ssimposIRF,asimposIRF,pr_mat_a,pr_mat_s,ainit,&
    numperIRF,lengthIRF,numsimIRF,shockperIRF,anum,snum)


!!!!END INITIALIZATIONS/GRID SETUPS

!!!!!VF ITERATION BLOCK

write(*,*) " "
write(13,*) " "


write(*,*) "Doing VFI."
write(13,*) "Doing VFI."

!initialize the VF and policies
Vold(:,:) = 0.0;
V(:,:) = 0.0
polmatold(:,:) = numendog/2
polmat(:,:) = 0
EVmat(:,:) = 0.0

do vfct=1,vfmaxit
    
    !!!Howard acceleration step
    do accelct=1,accelmaxit
        
        !set up the parallelization for the Howard acceleration construction
        !$omp parallel private(ct,endogct,exogct,polstar,exogprimect)
        !$omp do
        do ct=1,numstates
            
            !extract states from loop index matrix
            endogct=loopind(ct,1); exogct=loopind(ct,2)
            
            !extract policy from polmatold
            polstar = polmatold(endogct,exogct)

            !extract current period return
            V(endogct,exogct) = Rmat(endogct,exogct,polstar)

            !add discounted expected continuation value
            do exogprimect=1,numexog
                V(endogct,exogct) = V(endogct,exogct) + beta * pr_mat(exogct,exogprimect) * Vold(polstar,exogprimect)
            end do !exogprimect
            
        end do !ct
        !$omp end do
        !$omp end parallel
        
        !initialize for the next round of the Howard VFI
        Vold = V

    end do !accelct
    
    !!!create EVmat for RHS optimization
    
    !set up the parallelization for the creation of the continuation value
    !$omp parallel private(ct,polct,exogct)
    !$omp do
    do ct=1,numstates
        
        !extract exogenous and policy states
        polct=loopind(ct,1); exogct=loopind(ct,2)
        
        !create expected value for the next period based on policy and exogenous transitions
        EVmat(exogct,polct) = sum(pr_mat(exogct,:) * V(polct,:))
        
    end do !ct
    !$omp end do
    !$omp end parallel
    
    !!!RHS optimization step
    
    !set up the parallelization for the optimization of the Bellman RHS
    !$omp parallel private(ct,endogct,exogct,RHSvec,polstar)
    !$omp do
    do ct=1,numstates
        
        !extract exogenous and endogenous states
        endogct=loopind(ct,1); exogct=loopind(ct,2)
        
        !create RHS vector
        RHSvec = Rmat(endogct,exogct,:) + beta * EVmat(exogct,:)
        
        !extract policies via selection of max payoff
        polstar = maxloc(RHSvec,1)
        polmat(endogct,exogct) = polstar
        V(endogct,exogct) = RHSvec(polstar)
        
    end do !ct
    !$omp end do
    !$omp end parallel
    
    !compute errors and output the info
    vferror = maxval(abs(V-Vold))
    polerror = maxval(abs(polmat-polmatold))
    
    write(13,"(A,I3,A,F10.5,A)") "VF iter = ",vfct," in ",omp_get_wtime()-start," seconds"
    write(13,"(A,I3,A,F10.5)") "VF iter = ",vfct,", VF error = ",vferror
    write(13,"(A,I3,A,F10.5)") "VF iter = ",vfct,", policy error = ",polerror
    write(13,*) " "
    
    write(*,"(A,I3,A,F10.5,A)") "VF iter = ",vfct," in ",omp_get_wtime()-start," seconds"
    write(*,"(A,I3,A,F10.5)") "VF iter = ",vfct,", VF error = ",vferror
    write(*,"(A,I3,A,F10.5)") "VF iter = ",vfct,", policy error = ",polerror
    write(*,*) " "
    
    !exit criterion, then initialize for the next round of VFI
    if (polerror<vferrortol) exit
    Vold = V
    polmatold = polmat
    
end do !vfmaxit

!now that the VF process is over, deallocate memory-intensive unused arrays
deallocate(Rmat,Vold,polmatold,EVmat)

!convert polmat to policies for capital and labor
do endogct=1,numendog
do exogct=1,numexog
    !convert policies to actual capital and labor values
    kprime_key(endogct,exogct) = endog_key(polmat(endogct,exogct),1)
    kprime_pos(endogct,exogct) = endog_pos(polmat(endogct,exogct),1)
    
    !convert policies to capital and labor integer grid points
    lpol_key(endogct,exogct) = endog_key(polmat(endogct,exogct),2)
    lpol_pos(endogct,exogct) = endog_pos(polmat(endogct,exogct),2)
end do !exogct
end do !endogct

!display warning if hit the top of the grid or the bottom of the grid
!for either policy variable

write(*,*) " "
write(13,*) " "

write(*,*) "Hit the top or bottom of grid in theory?"
write(13,*) "Hit the top or bottom of grid in theory?"

if (maxval(kprime_pos)==knum) then 
    write(*,*) "Hit top of capital grid."    
    write(13,*) "Hit top of capital grid."
end if

if (maxval(lpol_pos)==lnum) then 
    write(*,*) "Hit top of labor grid."    
    write(13,*) "Hit top of labor grid."
end if

if (minval(kprime_pos)==1) then 
    write(*,*) "Hit bottom of capital grid."    
    write(13,*) "Hit bottom of capital grid."
end if

if (minval(lpol_pos)==1) then 
    write(*,*) "Hit bottom of labor grid."    
    write(13,*) "Hit bottom of labor grid."
end if

write(*,*) " "
write(13,*) " "

!!!!!END VF ITERATION BLOCK

!!!!!SIMULATION BLOCK
write(*,*) "Doing unconditional simulation."
write(13,*) "Doing unconditional simulation."

!first, initialize the distribution over endogenous variables
distzkl(:,:,:) = 0.0; !set everything to zero
distzkl(:,numendog/2,1) = dble(1.0/znum); !pick a point with initial mass

!then, initialize the aggregate series to 0
Ysim(:) = 0.0; Ksim = Ysim; Lsim = Ysim; Isim = Ysim; Hsim = Ysim; ACksim = Ysim; AClsim = Ysim;
TOPksim = Ysim; BOTksim = Ysim; TOPlsim = Ysim; BOTlsim = Ysim; ADJksim = Ysim; ADJlsim = Ysim;

!now, loop over periods in the simulation, tracking the endogenous
!movement of weight around the grid
do t=1,numper-1
    
    if (mod(t,100)==1) then
        write(*,"(A,F5.1,A)") "Done with ",100.0*t/numper,"% of sim."
        write(13,"(A,F5.1,A)") "Done with ",100.0*t/numper,"% of sim."
    end if
    
    !aggregate states
    act = asimpos(t); sct=ssimpos(t)
    aval = a0(act)
    
    !loop over z and the endogenous variables
    do zct=1,znum
    do endogct=1,numendog
                
        !get implied exogenous state
        exogct = (zct-1)*anum*snum + (act-1)*snum + sct
        
        !extract endogenous policy
        polstar = polmat(endogct,exogct)
        
        !insert distributional weight in appropriate slots next period
        distzkl(:,polstar,t+1) = distzkl(:,polstar,t+1) + pr_mat_z(zct,:,sct)*distzkl(zct,endogct,t)
   
    end do !endogct
    end do !zct
    
    !now, round to make sure that you're ending up with a distribution which makes sense each period
    distzkl(:,:,t+1) = distzkl(:,:,t+1)/sum(distzkl(:,:,t+1))
        
end do !t

write(*,"(A,F15.1,A)") "Finished simulation in ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Finished simulation in ",omp_get_wtime()-start," seconds."    

!now, process the simulation, which doesn't have intertemporal dependencies and can 
!therefore be parallelized over time

!$omp parallel private(t,act,sct,aval,zct,endogct,exogct,polstar,zval,kval,lmin1val,&
!$omp& kprimeval,lval,yval,ackval,aclval,ival,hval)
!$omp do
do t=1,numper-1
    
    !aggregate states
    act = asimpos(t); sct=ssimpos(t)
    aval = a0(act)
    
    !loop over z and the endogenous variables
    do zct=1,znum
    do endogct=1,numendog
                
        !get implied exogenous state
        exogct = (zct-1)*anum*snum + (act-1)*snum + sct
        
        !extract endogenous policy
        polstar = polmat(endogct,exogct)
        
        !get the associated idio values for policies and states
        zval = z0(zct);
        kval = endog_key(endogct,1);
        lmin1val = endog_key(endogct,2)
        kprimeval = endog_key(polstar,1);
        lval = endog_key(polstar,2)
        
        !get associated output, investment, hiring, and AC series
        yval = y(zval,aval,kval,lval,alpha,nu)
        ackval = ACk(zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak)
        aclval = ACl(zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wval)
        ival = kprimeval - (1.0-deltak) * kval
        hval = lval - (1.0-deltan)*lmin1val

        !add the correct weight to the aggregate series next period
        Ysim(t) = Ysim(t) + yval * distzkl(zct,endogct,t); !output
        Ksim(t) = Ksim(t) + kval * distzkl(zct,endogct,t); !capital
        Lsim(t) = Lsim(t) + lval * distzkl(zct,endogct,t); !labor
        Isim(t) = Isim(t) + ival * distzkl(zct,endogct,t); !investment
        Hsim(t) = Hsim(t) + hval * distzkl(zct,endogct,t); !hiring
        ACksim(t) = ACksim(t) + ackval * distzkl(zct,endogct,t); !capital AC
        AClsim(t) = AClsim(t) + aclval * distzkl(zct,endogct,t); !labor AC
        
        !count the fraction hitting the top and the bottom of the grid
        if (kprimeval==k0(knum)) TOPksim(t) = TOPksim(t) + distzkl(zct,endogct,t);
        if (kprimeval==k0(1)) BOTksim(t) = BOTksim(t) + distzkl(zct,endogct,t);
        if (lval == l0(lnum)) TOPlsim(t) = TOPlsim(t) + distzkl(zct,endogct,t);
        if (lval == l0(1)) BOTlsim(t) = BOTlsim(t) + distzkl(zct,endogct,t);
        
        !count the fraction adjusting capital and labor
        if (abs(ival)>changetol) ADJksim(t) = ADJksim(t) + distzkl(zct,endogct,t);
        if (abs(hval)>changetol) ADJlsim(t) = ADJlsim(t) + distzkl(zct,endogct,t);
        
    end do !endogct
    end do !zct
    
end do !t
!$omp end do
!$omp end parallel


write(*,"(A,F15.1,A)") "Finished processing simulation in ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Finished processing simulation in ",omp_get_wtime()-start," seconds."    

write(*,*) " "
write(13,*) " "

!!!!!END SIMULATION BLOCK


!!!!!IRF BLOCK

write(*,*) "Doing IRF simulation."
write(13,*) "Doing IRF simulation."

!now that the exogenous simulations have been created, actually perform the simulation
!first, initialize the distribution over endogenous variables
distzklIRF(:,:,:) = 0.0; !set everything to zero
distzklIRF(:,numendog/2,1) = dble(1.0/znum); !pick a point with initial mass

!then, initialize the aggregate series to 0
YsimIRF(:) = 0.0; KsimIRF = YsimIRF; LsimIRF = YsimIRF; IsimIRF = YsimIRF; HsimIRF = YsimIRF;
ACksimIRF = YsimIRF; AClsimIRF = YsimIRF; ADJksimIRF = YsimIRF; ADJlsimIRF = YsimIRF;

!now, loop over periods in the IRF simulation, tracking the endogenous
!movement of weight around the grid
do t=1,numperIRF-1
    
    if (mod(t,5000)==1) then
        write(*,"(A,F5.1,A)") "Done with ",100.0*t/numperIRF,"% of sim for IRF."
        write(13,"(A,F5.1,A)") "Done with ",100.0*t/numperIRF,"% of sim for IRF."
    end if
    
    !aggregate states
    act = asimposIRF(t); sct=ssimposIRF(t)
    aval = a0(act)
    
    !loop over z and the endogenous variables
    do zct=1,znum
    do endogct=1,numendog
                
        !get implied exogenous state
        exogct = (zct-1)*anum*snum + (act-1)*snum + sct
        
        !extract endogenous policy
        polstar = polmat(endogct,exogct)
        
        !insert distributional weight in appropriate slots next period
        distzklIRF(:,polstar,t+1) = distzklIRF(:,polstar,t+1) + pr_mat_z(zct,:,sct)*distzklIRF(zct,endogct,t)
        
    end do !endogct
    end do !zct
    
    !now, round to make sure that you're ending up with a distribution which makes sense each period
    distzklIRF(:,:,t+1) = distzklIRF(:,:,t+1)/sum(distzklIRF(:,:,t+1))
        
end do !t

write(*,"(A,F15.1,A)") "Finished simulation for IRF in ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Finished simulation for IRF in ",omp_get_wtime()-start," seconds."    

!now, process the simulation, which doesn't have intertemporal dependencies and can 
!therefore be parallelized over time

!$omp parallel private(t,act,sct,aval,zct,endogct,exogct,polstar,zval,kval,lmin1val,&
!$omp& kprimeval,lval,yval,ackval,aclval,ival,hval)

!$omp do
do t=1,numperIRF-1
    
    !aggregate states
    act = asimposIRF(t); sct=ssimposIRF(t)
    aval = a0(act)
    
    !loop over z and the endogenous variables
    do zct=1,znum
    do endogct=1,numendog
                
        !get implied exogenous state
        exogct = (zct-1)*anum*snum + (act-1)*snum + sct
        
        !extract endogenous policy
        polstar = polmat(endogct,exogct)
        
        !get the associated idio values for policies and states
        zval = z0(zct);
        kval = endog_key(endogct,1);
        lmin1val = endog_key(endogct,2)
        kprimeval = endog_key(polstar,1);
        lval = endog_key(polstar,2)
        
        !get associated output, investment, hiring, and AC series
        yval = y(zval,aval,kval,lval,alpha,nu)
        ackval = ACk(zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak)
        aclval = ACl(zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wval)
        ival = kprimeval - (1.0-deltak) * kval
        hval = lval - (1.0-deltan)*lmin1val

        !add the correct weight to the aggregate series next period
        YsimIRF(t) = YsimIRF(t) + yval * distzklIRF(zct,endogct,t); !output
        KsimIRF(t) = KsimIRF(t) + kval * distzklIRF(zct,endogct,t); !capital
        LsimIRF(t) = LsimIRF(t) + lval * distzklIRF(zct,endogct,t); !labor
        IsimIRF(t) = IsimIRF(t) + ival * distzklIRF(zct,endogct,t); !investment
        HsimIRF(t) = HsimIRF(t) + hval * distzklIRF(zct,endogct,t); !hiring
        ACksimIRF(t) = ACksimIRF(t) + ackval * distzklIRF(zct,endogct,t); !capital AC
        AClsimIRF(t) = AClsimIRF(t) + aclval * distzklIRF(zct,endogct,t); !labor AC
 
        !count the fraction adjusting capital and labor
        if (abs(ival)>changetol) ADJksimIRF(t) = ADJksimIRF(t) + distzklIRF(zct,endogct,t);
        if (abs(hval)>changetol) ADJlsimIRF(t) = ADJlsimIRF(t) + distzklIRF(zct,endogct,t);
        
    end do !endogct
    end do !zct
    
end do !t
!$omp end do
!$omp end parallel

write(*,"(A,F15.1,A)") "Finished processing simulation for IRF in ",omp_get_wtime()-start," seconds."    
write(13,"(A,F15.1,A)") "Finished processing simulation for IRF in ",omp_get_wtime()-start," seconds."    


write(*,*) " "
write(13,*) " "

!!!!!END IRF BLOCK

!record time before beginning file output
write(*,"(A,F15.1,A)") "Finished calculations at ",omp_get_wtime()-start," seconds"
write(13,"(A,F15.1,A)") "Finished calculations at ",omp_get_wtime()-start," seconds"

write(*,*) " "
write(13,*) " "

!!!!FILE OUTPUT BLOCK

write(*,*) "Writing output to .txt files."
write(13,*) "Writing output to .txt files."

write(*,*) " "
write(13,*) " "

!constants
MATLABconstants = (/ dble(znum),dble(knum),dble(lnum),dble(anum),dble(snum),&
    dble(numper),dble(numdiscard),dble(numendog),dble(numexog),dble(numstates),&
    deltak,deltan,dble(numsimIRF),dble(lengthIRF),dble(shockperIRF),&
    dble(numdiscIRF)/)

open(8,file=basedir//"\MATLABconstants.txt")
do ct=1,numconstants
write(8,*) MATLABconstants(ct)
end do !ct
close(8)

!labor grid
open(8,file=basedir//"\l0.txt")
do ct=1,lnum
write(8,*) l0(ct)
end do
close(8)

!capital grid
open(8,file=basedir//"\k0.txt")
do ct=1,knum
write(8,*) k0(ct)
end do
close(8)

!idio prod grid
open(8,file=basedir//"\z0.txt")
do ct=1,znum
write(8,*) z0(ct)
end do
close(8)

!agg prod grid
open(8,file=basedir//"\a0.txt")
do ct=1,anum
write(8,*) a0(ct)
end do
close(8)

!VF
open(8,file=basedir//"\V.txt")
do endogct=1,numendog
do exogct=1,numexog
write(8,*) V(endogct,exogct)        
end do !exogct
end do !endogct
close(8)

!policy matrix
open(8,file=basedir//"\polmat.txt")
do endogct=1,numendog
do exogct=1,numexog
write(8,*) polmat(endogct,exogct)        
end do !exogct
end do !endogct
close(8)

!kprime policy key
open(8,file=basedir//"\kprime_key.txt")
do endogct=1,numendog
do exogct=1,numexog
write(8,*) kprime_key(endogct,exogct)        
end do !exogct
end do !endogct
close(8)

!labor policy key
open(8,file=basedir//"\lpol_key.txt")
do endogct=1,numendog
do exogct=1,numexog
write(8,*) lpol_key(endogct,exogct)        
end do !exogct
end do !endogct
close(8)

!agg prod series
open(8,file=basedir//"\asimpos.txt")
do t=1,numper
write(8,*) asimpos(t)
end do !t
close(8)

!unc series
open(8,file=basedir//"\ssimpos.txt")
do t=1,numper
write(8,*) ssimpos(t)
end do !t
close(8)

!output series
open(8,file=basedir//"\Ysim.txt")
do t=1,numper
write(8,*) Ysim(t)
end do !t
close(8)

!capital series
open(8,file=basedir//"\Ksim.txt")
do t=1,numper
write(8,*) Ksim(t)
end do !t
close(8)

!labor series
open(8,file=basedir//"\Lsim.txt")
do t=1,numper
write(8,*) Lsim(t)
end do !t
close(8)

!investment series
open(8,file=basedir//"\Isim.txt")
do t=1,numper
write(8,*) Isim(t)
end do !t
close(8)

!hiring series
open(8,file=basedir//"\Hsim.txt")
do t=1,numper
write(8,*) Hsim(t)
end do !t
close(8)

!capital AC series
open(8,file=basedir//"\ACksim.txt")
do t=1,numper
write(8,*) ACksim(t)
end do !t
close(8)

!labor AC series
open(8,file=basedir//"\AClsim.txt")
do t=1,numper
write(8,*) AClsim(t)
end do !t
close(8)

!hit top of capital
open(8,file=basedir//"\TOPksim.txt")
do t=1,numper
write(8,*) TOPksim(t)
end do !t
close(8)

!hit bottom of capital
open(8,file=basedir//"\BOTksim.txt")
do t=1,numper
write(8,*) BOTksim(t)
end do !t
close(8)

!hit top of labor
open(8,file=basedir//"\TOPlsim.txt")
do t=1,numper
write(8,*) TOPlsim(t)
end do !t
close(8)

!hit bottom of labor
open(8,file=basedir//"\BOTlsim.txt")
do t=1,numper
write(8,*) BOTlsim(t)
end do !t
close(8)

!adjust capital
open(8,file=basedir//"\ADJksim.txt")
do t=1,numper
write(8,*) ADJksim(t)
end do !t
close(8)

!adjust labor
open(8,file=basedir//"\ADJlsim.txt")
do t=1,numper
write(8,*) ADJlsim(t)
end do !t
close(8)

!endogenous variable key
open(8,file=basedir//"\endog_key.txt")
do endogct=1,numendog
write(8,*) endog_key(endogct,:)
end do !t
close(8)

!exogenous variable key
open(8,file=basedir//"\exog_key.txt")
do exogct=1,numexog
write(8,*) exog_key(exogct,:)
end do !t
close(8)

!IRF unc series
open(8,file=basedir//"\ssimposIRF.txt")
do ct=1,numperIRF
write(8,*) ssimposIRF(ct)
end do !ct
close(8)

!IRF agg prod series
open(8,file=basedir//"\asimposIRF.txt")
do ct=1,numperIRF
write(8,*) asimposIRF(ct)
end do !ct
close(8)

!IRF output series
open(8,file=basedir//"\YsimIRF.txt")
do ct=1,numperIRF
write(8,*) YsimIRF(ct)
end do !ct
close(8)

!IRF capital series
open(8,file=basedir//"\KsimIRF.txt")
do ct=1,numperIRF
write(8,*) KsimIRF(ct)
end do !ct
close(8)

!IRF labor series
open(8,file=basedir//"\LsimIRF.txt")
do ct=1,numperIRF
write(8,*) LsimIRF(ct)
end do !ct
close(8)

!IRF investment series
open(8,file=basedir//"\IsimIRF.txt")
do ct=1,numperIRF
write(8,*) IsimIRF(ct)
end do !ct
close(8)

!IRF hiring series
open(8,file=basedir//"\HsimIRF.txt")
do ct=1,numperIRF
write(8,*) HsimIRF(ct)
end do !ct
close(8)

!IRF capital AC series
open(8,file=basedir//"\ACksimIRF.txt")
do ct=1,numperIRF
write(8,*) ACksimIRF(ct)
end do !ct
close(8)

!IRF labor AC series
open(8,file=basedir//"\AClsimIRF.txt")
do ct=1,numperIRF
write(8,*) AClsimIRF(ct)
end do !ct
close(8)

!IRF capital adjustment series
open(8,file=basedir//"\ADJksimIRF.txt")
do ct=1,numperIRF
write(8,*) ADJksimIRF(ct)
end do !ct
close(8)

!IRF labor adjustment series
open(8,file=basedir//"\ADJlsimIRF.txt")
do ct=1,numperIRF
write(8,*) ADJlsimIRF(ct)
end do !ct
close(8)

!!!!END OUTPUT BLOCK

!call the clock to get the end time and closes log file
finish = omp_get_wtime()
write(*,"(A,F15.1,A)") "Finished completely in ",finish-start," seconds."
write(13,"(A,F15.1,A)") "Finished completely in ",finish-start," seconds."
close(13)

!!!!INTERNAL FUNCTION AND SUBROUTINES BLOCK
contains 

subroutine unctauchen(transarray,grid,znum,zmin,zmax,snum,sigmagrid,rho)
implicit none 

!this routine discretizes the AR(1) productivity processes, subject to 
!unc shocks

!input/output declarations
integer :: znum,snum
double precision :: zmin,zmax,rho
double precision :: transarray(znum,znum,snum),grid(znum),sigmagrid(snum)

!other declarations
integer :: zct,sct,zprimect
double precision :: log_grid(znum),standdev,gridinc

!create grid
call linspace(log_grid,log(zmin),log(zmax),znum)
grid = exp(log_grid)
gridinc = log_grid(2)-log_grid(1)

!loop over unc states
do sct=1,snum
    standdev = sigmagrid(sct)
    do zct=1,znum
        !middle intervals
        do zprimect=2,(znum-1)
            transarray(zct,zprimect,sct) = &
                normcdf(log_grid(zprimect)+gridinc/2.0,rho*log_grid(zct),standdev) - &
                normcdf(log_grid(zprimect)-gridinc/2.0,rho*log_grid(zct),standdev)
        end do !zprimect
        !first interval & last interval take remainder of mass
        transarray(zct,1,sct) = normcdf(log_grid(1)+gridinc/2.0,rho*log_grid(zct),standdev)
        transarray(zct,znum,sct) = 1.0 - normcdf(log_grid(znum)-gridinc/2.0,rho*log_grid(zct),standdev)
    end do !zct
    
    !impose that everything sums to 1 with rounding
    do zct=1,znum
        transarray(zct,:,sct) = transarray(zct,:,sct) / sum(transarray(zct,:,sct))
    end do !zct
    
end do !sct

end subroutine unctauchen

subroutine uncexogsimIRF(ssimshocksIRF,asimshocksIRF,ssimposIRF,asimposIRF,pr_mat_a,pr_mat_s,ainit,&
    numperIRF,lengthIRF,numsimIRF,shockperIRF,anum,snum)
implicit none

!this subroutine simulates the unc and agg prod processes in the IRF experiment

!input/output declarations
integer :: ainit,numperIRF,lengthIRF,numsimIRF,shockperIRF,asimposIRF(numperIRF),ssimposIRF(numperIRF),&
    anum,snum
double precision :: ssimshocksIRF(numperIRF),asimshocksIRF(numperIRF),pr_mat_a(anum,anum,snum),&
    pr_mat_s(snum,snum)

!other declarations
integer :: t,simct,perct,sprimect,ct,aprimect
double precision :: ssimgrid(snum),asimgrid(anum)

!initialize the simulation for agg prod - unc is initialized below
asimposIRF(1) = ainit

!now, simulate the unc process, according to the IRF control values
perct=0
do simct=1,numsimIRF
    do t=1,lengthIRF
    
    perct=perct+1
    
    !first, set the unc process to low for the first shockperIRF-1 periods
    if (t==1) then
        
        ssimposIRF(perct) = sinit; !1 is low unc, 2 is high unc
        
    !then, for the shock period, set unc to a high level
    else if (t==shockperIRF) then
        
        ssimposIRF(perct) = 2;
    
    !then, set unc based on the shocks: this block is like the one in uncexogsim, up until
    !the unc spell ends
    else 
        
        sprimect=0; !this will store new value
   
        !create vector of cumulative probability thresholds from transition mat
        ssimgrid(1) = pr_mat_s(ssimposIRF(perct-1),1)
        do ct=2,snum
            ssimgrid(ct) = ssimgrid(ct-1) + pr_mat_s(ssimposIRF(perct-1),ct)
        end do !ct

        !compare the uniform shock to the thresholds, and decide which bin
        if (ssimshocksIRF(perct)<ssimgrid(1)) then
            sprimect = 1
        else 
            do ct=2,snum
                if (ssimgrid(ct-1)<=ssimshocksIRF(perct).and.ssimshocksIRF(perct)<ssimgrid(ct)) then
                    sprimect = ct
                end if
            end do !ct
        end if

        !fill in the simulated unc position
        ssimposIRF(perct) = sprimect
        

    end if
    
end do !t
end do !simct

!now, simulate the agg prod process, according to the simulated unc process from above
do t=2,numperIRF; !note that this counter is over all the periods, not differentiating across simulations
    aprimect=0; !this will store the new value of agg prod
    
    !create vector of cumulative probability thresholds from transition mat
    !note that the third entry takes uncertainty into account
    asimgrid(1) = pr_mat_a(asimposIRF(t-1),1,ssimposIRF(t-1))
    do ct=2,anum
        asimgrid(ct) = asimgrid(ct-1) + pr_mat_a(asimposIRF(t-1),ct,ssimposIRF(t-1))
    end do !ct
    
    !compare the uniform shock to the thresholds, and decide which bin
    if (asimshocksIRF(t)<asimgrid(1)) then
        aprimect = 1
    else 
        do ct=2,anum
            if (asimgrid(ct-1)<=asimshocksIRF(t).and.asimshocksIRF(t)<asimgrid(ct)) then
                aprimect = ct
            end if
        end do !ct
    end if
    
    !fill in the simulated unc position
    asimposIRF(t) = aprimect
end do !t

end subroutine uncexogsimIRF

subroutine uncexogsim(ssimshocks,asimshocks,ssimpos,asimpos,pr_mat_s,pr_mat_a,ainit,sinit,snum,anum,numper)
implicit none

!this subroutine simulates the uncertainty and aggregate productivity processes

!input/output declarations
integer :: numper,ainit,sinit,snum,anum,ssimpos(numper),asimpos(numper)
double precision :: ssimshocks(numper),asimshocks(numper),pr_mat_s(snum,snum),&
    pr_mat_a(anum,anum,snum)
    
!other declarations
integer :: t,ct,sprimect,aprimect
double precision :: ssimgrid(snum),asimgrid(anum)

!initialize the simulation for uncertainty and for agg prod
ssimpos(1) = sinit
asimpos(1) = ainit

!now simulate the uncertainty process
do t=2,numper
    sprimect=0; !this will store new value
   
    !create vector of cumulative probability thresholds from transition mat
    ssimgrid(1) = pr_mat_s(ssimpos(t-1),1)
    do ct=2,snum
        ssimgrid(ct) = ssimgrid(ct-1) + pr_mat_s(ssimpos(t-1),ct)
    end do !ct

    !compare the uniform shock to the thresholds, and decide which bin
    if (ssimshocks(t)<ssimgrid(1)) then
        sprimect = 1
    else 
        do ct=2,snum
            if (ssimgrid(ct-1)<=ssimshocks(t).and.ssimshocks(t)<ssimgrid(ct)) then
                sprimect = ct
            end if
        end do !ct
    end if

    !fill in the simulated unc position
    ssimpos(t) = sprimect
end do !t

!given the uncertainty process, simulate the aggregate productivity process
do t=2,numper
    aprimect=0; !this will store the new value of agg prod
    
    !create vector of cumulative probability thresholds from transition mat
    !note that the third entry takes uncertainty into account
    asimgrid(1) = pr_mat_a(asimpos(t-1),1,ssimpos(t-1))
    do ct=2,anum
        asimgrid(ct) = asimgrid(ct-1) + pr_mat_a(asimpos(t-1),ct,ssimpos(t-1))
    end do !ct
    
    !compare the uniform shock to the thresholds, and decide which bin
    if (asimshocks(t)<asimgrid(1)) then
        aprimect = 1
    else 
        do ct=2,anum
            if (asimgrid(ct-1)<=asimshocks(t).and.asimshocks(t)<asimgrid(ct)) then
                aprimect = ct
            end if
        end do !ct
    end if
    
    !fill in the simulated unc position
    asimpos(t) = aprimect
end do !t
end subroutine uncexogsim

double precision function y(zval,aval,kval,lval,alpha,nu)
implicit none

!this function evaluates output given parameters and states

double precision :: zval,aval,kval,lval,alpha,nu

y = zval * aval * (kval ** alpha) * (lval ** nu)
    
end function

double precision function ACk(zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak)
implicit none

!this function evaluates capital AC given parameters and states

!input parameters
double precision :: zval,aval,kval,lval,alpha,nu,kprimeval,capirrev,capfix,deltak

!other parameters
double precision :: yval,changetol,ival

changetol = 1.0e-10
ival = kprimeval - (1.0-deltak) * kval

!start the AC at 0
ACk = 0.0

!take out the partial irreversibility costs
if (ival<-changetol) then
    ACk = ACk - ival * capirrev
end if

!take out the fixed disruption costs
if (abs(ival)>changetol) then
    yval = y(zval,aval,kval,lval,alpha,nu)
    ACk = ACk + yval * capfix
end if

end function ACk

double precision function ACl(zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wval)
implicit none

!this function evaluates labor AC given parameters and states

!input parameters
double precision :: zval,aval,kval,lval,alpha,nu,lmin1val,hirelin,firelin,labfix,deltan,wval

!other parameters
double precision :: hval,changetol,yval

changetol = 1.0e-10

hval = lval - (1.0-deltan)*lmin1val

!take out the fixed and linear costs if you changed costs
ACl = 0.0
if (abs(hval)>changetol) then
    yval = y(zval,aval,kval,lval,alpha,nu)
    ACl = ACl + labfix * yval; !the fixed costs
    if (hval<-changetol) ACl = ACl - hval * wval * firelin; !the linear firing costs
    if (hval>changetol) ACl = ACl + hval * wval * hirelin; !the linear hiring costs
end if
end function ACl

end program RUBC_pe