/* This file generates the figures in the paper and the numbers which get reported in the text.*/ 

clear all  
set mem 2000m

cd "$output"
graph set eps fontface times



*=============================
* SECTION 1 - Introduction  
*=============================

**********************************************************
* Back of the envelope calculations for the intrudoction  
**********************************************************

use data_for_accounting, clear
tsset year

gen P_M_OWN = M_OWN/l1.OWN
gen P_M_OWN_COUNTY = M_OWN_OUT_COUNT/l1.OWN
gen P_MJ_M_OWN = M_OWN_J/M_OWN
gen P_MJ_OWN = P_M_OWN*P_MJ_M_OWN
gen P_M_RENT = M_RENT/l1.RENT		
gen P_OWN = OWN/POP		
gen u_OWN = U_OWN/OWN
gen share_OWN = OWN/POP

list  year P_M_OWN_COUNTY u_OWN share_OWN

* Log file contains the back of the envelope calculation
cap log close
log using "$results\envelope", replace t

* Introduction example
di round(P_M_OWN_COUNTY[2],0.001)
di round(P_M_OWN_COUNTY[5],0.001)

local mc_m = round(P_M_OWN_COUNTY[5],0.001)-round(P_M_OWN_COUNTY[2],0.001)
di 100*`mc_m'
di round(P_M_OWN_COUNTY[6],0.001)/round(P_M_OWN_COUNTY[2],0.001)

di u_OWN[5]
di share_OWN[5]

local uh = u_OWN[5]*share_OWN[5]
di "The difference in unemployment rate in 2010 is " round(100*`mc_m'*`uh',0.001)
cap log close



*=============================
* SECTION 2 - MOBILITY TRENDS  
*=============================

/* 	Note:
	====
	Graphs are different than rest of the paper in three aspects:
	(1) Presentationally, it is showing t to t+1 (rather than t-1 to t). 
	(2) It drops imputed values. This is important for inter-county long trends, but not needed for the rest of the paper that is focused
		on 2006 and on, where the imputation procedure is corrected by census (see Kaplan and SW). 
	(3) Because we drop impoted, we divide by homeownership and renting at t+1. Otherwise the weights are problematic (as dropping different number
	    of observations with different weights every year). 
*/




************
* Figure 1 *
************

u data4figure1_noimpute.dta, clear


tsset year
gen own = OWN/POP
gen P_M_OWN = 100*M_OWN/OWN				/* Note that need to devide by the population of owners at t-1, since moving is from t-1 to t */ 
gen P_M_OWN_J = 100*M_OWN_J/OWN
gen P_MJ_given_M_OWN = 100*M_OWN_J/M_OWN	 
gen P_M_OWN_COUNTY = 100*M_OWN_COUNTY/OWN

gen rent = RENT/POP
gen P_M_RENT = 100*M_RENT/RENT				/* Note that need to devide by the population of owners at t-1, since moving is from t-1 to t */ 
gen P_M_RENT_J = 100*M_RENT_J/RENT
gen P_MJ_given_M_RENT = 100*M_RENT_J/M_RENT
gen P_M_RENT_COUNTY = 100*M_RENT_COUNTY/RENT

gen P_M_J = 100*(M_OWN_J+M_RENT_J)/(OWN+RENT)
gen P_M = 100*(M_OWN+M_RENT)/(OWN+RENT)
gen P_MJ_given_M = 100*(M_OWN_J+M_RENT_J)/(M_OWN + M_RENT)


label var P_M_RENT "Moving Rate - Renters"
label var P_M_OWN "Moving Rate - Owners"
label var P_M_RENT_J "Moving rate for job reasosn - renters"
label var P_M_OWN_J "Moving rate for job reasosn - owners"
label var P_M_RENT_COUNTY "Moving rate out of county - renters"
label var P_M_OWN_COUNTY "Moving rate out of county  - owners"
label var P_MJ_given_M_OWN "Share job related mobility - owners"  
label var P_MJ_given_M_RENT "Share job related mobility - renters"
label var own "Home Ownership Rate (in the LF)"

/* Merging with recession dummy. Note that I tried to make recession dummy consistent with March to March years */
merge year using "$do\march_rec.dta"
drop if _merge~=3
drop _merge
ren march_rec rec_dummy 

sort year
merge year using data4trend_HOcorr
keep if _m==3
drop _m

gen P_Mcorr_OWN =  100*age1665_M_OWM_corr/ age1665_OWN				
gen P_Mcorr_RENT =  100*age1665_M_RENT_corr/ age1665_RENT
label var P_Mcorr_RENT "Moving rate - renters, HO correction applied"
label var P_Mcorr_OWN "Moving rate - owners, HO correction applied"

gen P_Mcorr2_OWN =  100*age1665_M_OWM_corr2/ age1665_OWN				
gen P_Mcorr2_RENT =  100*age1665_M_RENT_corr2/ age1665_RENT
label var P_Mcorr2_RENT "Moving rate - renters, HO correction 2 applied"
label var P_Mcorr2_OWN "Moving rate - owners, HO correction 2  applied"

/* 	Changing timing: The March recessions bars are generated so that if there was a recession between march t-1 and march t, there is 
	a bar at t (since the question at t referes to moving between t-1 and t). This is correct but rather weird on the graphs since it 
	looks like the recessions are in the wrong year... 
	Below we do the following: 1) change the timing on mobility question to be moved between t-1 to t reported at t-1. 2) Move the recession bars
	back one year to match the new definition of moving (and the normal recession dates...)
	I also construct an average annual UE series. 
	*/ 

keep own P_M P_M_RENT P_M_RENT_COUNTY P_M_OWN P_M_OWN_COUNTY P_MJ_given_M_RENT P_M_RENT_J P_M_OWN_J P_MJ_given_M_OWN year rec_dummy M_OWN_LOOK_WORK M_RENT_LOOK_WORK OWN_U RENT_U P_Mcorr*
replace year = year-1
label var P_M_RENT "Moving Rate (t to t+1) - Renters (%)"
label var P_M_OWN "Moving Rate (t to t+1) - Owners (%)"
label var P_M_RENT_J "Moving rate for job reasosn (t to t+1) - renters (%)"
label var P_M_OWN_J "Moving rate for job reasosn (t to t+1) - owners (%)"


*** All moves (panel 1)
cap drop recg*
gen recg=.
gen max_for_graph = 43
gen min_for_graph = 0
su max_for_graph
local height=r(max)
su min_for_graph
local base=r(min)
drop max_for_graph min_for_graph
replace recg=rec_dummy*`height'
replace recg=. if recg==0
label var recg "Recession" 
	
#delimit ;
tw 	(bar recg year if year>=1980, color(ebg) cmiss(n) xtitle(""))
	(connected P_M_RENT year if year<=2007 & year>=1980, color(dkgreen)  ms(Dh)  msize(medlarge) lwidth(medthick) cmissing(n))
	(connected P_M_OWN year if year<=2007 & year>=1980, color(ebblue)  ms(Th)  msize(medlarge) lwidth(medthick) cmissing(n))
	(connected P_M_RENT year if year>=2007 & year>=1980, color(dkgreen)  ms(D)  msize(medlarge) lwidth(medthick) cmissing(n)  ylabel(0(10)43, norescale labs(large)))
	(connected P_M_OWN year if year>=2007 & year>=1980, color(ebblue)  ms(T)  msize(medlarge) lwidth(medthick) cmissing(n) ),
	tlabel(1980(5)2010, labs(large))
	graphregion(color(white)) legend(order(2 3) rows(1)) name(M, replace)
	title("All Moves", size(vlarge) color(black));
#delimit cr


*** Job related moves (panel 2)
cap drop recg*
gen recg=.
gen max_for_graph = 8
gen min_for_graph = 0
su max_for_graph
local height=r(max)
su min_for_graph
local base=r(min)
drop max_for_graph min_for_graph
replace recg=rec_dummy*`height'
replace recg=. if recg==0
label var recg "Recession" 
		
#delimit ;
tw 	(bar recg year if year>=1998, color(ebg) cmiss(n) xtitle(""))
	(connected P_M_RENT_J year if year<=2007 & year>=1998, color(dkgreen)  ms(Dh)  msize(medlarge) lwidth(medthick) cmissing(n) )
	(connected P_M_OWN_J year if year<=2007 & year>=1998, color(ebblue)  ms(Th)  msize(medlarge) lwidth(medthick) cmissing(n))
	(connected P_M_RENT_J year if year>=2007 & year>=1998, color(dkgreen)  ms(D)  msize(medlarge) lwidth(medthick) cmissing(n)  ylabel(0(2)8,  norescale labs(large)))
	(connected P_M_OWN_J year if year>=2007 & year>=1998, color(ebblue)  ms(T)  msize(medlarge) lwidth(medthick) cmissing(n) ),
	tlabel(1998(4)2010, labs(large))
	graphregion(color(white)) legend(order(2 3) rows(1)) name(MJ, replace)
	title("Self-Reported Job-Related Moves", size(vlarge) color(black));
#delimit cr

	
*** Inter-county moves (panel 3)
cap drop recg*
gen recg=.
gen max_for_graph = 17
gen min_for_graph = 0
su max_for_graph
local height=r(max)
su min_for_graph
local base=r(min)
drop max_for_graph min_for_graph
replace recg=rec_dummy*`height'
replace recg=. if recg==0
label var recg "Recession" 
	
#delimit ;
tw 	(bar recg year if year>=1980, color(ebg) cmiss(n) xtitle(""))
	(connected P_M_RENT_C year if year<=2007 & year>=1980, color(dkgreen)  ms(Dh)  msize(medlarge) lwidth(medthick) cmissing(n) )
	(connected P_M_OWN_C year if year<=2007 & year>=1980, color(ebblue)  ms(Th)  msize(medlarge) lwidth(medthick) cmissing(n) )
	(connected P_M_RENT_C year if year>=2007 & year>=1980, color(dkgreen)  ms(D)  msize(medlarge) lwidth(medthick) cmissing(n)  ylabel(0(4)16, norescale labs(large)))
	(connected P_M_OWN_C year if year>=2007 & year>=1980, color(ebblue)  ms(T)  msize(medlarge) lwidth(medthick) cmissing(n) ),
	tlabel(1980(5)2010, labs(large))
	graphregion(color(white)) legend(order(2 3) rows(1)) name(MC, replace)
	title("Inter-County Moves", size(vlarge) color(black));
#delimit cr

grc1leg M MJ MC, rows(1) graphregion(color(white)) 
graph display, xsize(12) ysize(5)
graph export "$graphs\mobility_HO_R_new.eps", as(eps) replace
graph export "$graphs\mobility_HO_R_new.png", as(png) width(2400) replace



******************************************************************************
* Log file contains the numbers which go into the text describing the graphs *
******************************************************************************
cap log close
log using "$results\mobility_trends", replace t

su P_M_OWN if year==2006
local p_m_own_2006 = r(mean)
su P_M_OWN if year==2011
local p_m_own_2011 = r(mean)
di "Percentage points diff: "`p_m_own_2011' - `p_m_own_2006'
di "Percent diff: "100*(`p_m_own_2011'/`p_m_own_2006'-1)

su P_M_RENT if year==2006
local p_m_rent_2006 = r(mean)
su P_M_RENT if year==2011
local p_m_rent_2011 = r(mean)
di "Percentage points diff: "`p_m_rent_2011'-`p_m_rent_2006'
di "Percent diff: "100*(`p_m_rent_2011'/`p_m_rent_2006'-1)

su P_M_OWN_J if year==2006
local p_m_own_j_2006 = r(mean)
su P_M_OWN_J if year==2011
local p_m_own_j_2011 = r(mean)
di "Percentage points diff: "`p_m_own_j_2011'-`p_m_own_j_2006'
di "Percent diff: "100*(`p_m_own_j_2011'/`p_m_own_j_2006'-1)

su P_M_RENT_J if year==2006
local p_m_rent_j_2006 = r(mean)
su P_M_RENT_J if year==2011
local p_m_rent_j_2011 = r(mean)
di "Percentage points diff: "`p_m_rent_j_2011'-`p_m_rent_j_2006'
di "Percent diff: "100*(`p_m_rent_j_2011'/`p_m_rent_j_2006'-1)


su P_M_OWN_COUNTY if year==2006
local p_m_own_county_2006 = r(mean)
su P_M_OWN_COUNTY if year==2011
local p_m_own_county_2011 = r(mean)
di "Percentage points diff: "`p_m_own_county_2011'-`p_m_own_county_2006'
di "Percent diff: "100*(`p_m_own_county_2011'/`p_m_own_county_2006'-1)

su P_M_RENT_COUNTY if year==2006
local p_m_rent_county_2006 = r(mean)
su P_M_RENT_COUNTY if year==2011
local p_m_rent_county_2011 = r(mean)
di "Percentage points diff:" `p_m_rent_county_2011'-`p_m_rent_county_2006'
di "Percent diff: "100*(`p_m_rent_county_2011'/`p_m_rent_county_2006'-1)

su P_Mcorr_OWN if year==2006
local p_m_own_corr_2006 = r(mean)
su P_Mcorr_OWN if year==2011
local p_m_own_corr_2011 = r(mean)
di "Percentage points diff: "`p_m_own_corr_2011'-`p_m_own_corr_2006'
di "Percent diff: "100*(`p_m_own_corr_2011'/`p_m_own_corr_2006'-1)

su P_Mcorr_RENT if year==2006
local p_m_rent_corr_2006 = r(mean)
su P_Mcorr_RENT if year==2011
local p_m_rent_corr_2011 = r(mean)
di "Percentage points diff:" `p_m_rent_corr_2011'-`p_m_rent_corr_2006'
di "Percent diff: "100*(`p_m_rent_corr_2011'/`p_m_rent_corr_2006'-1)

cap log close


****************************************************************************************
* Log file contains the regressions for renters below trend - also generates table A.1 *
****************************************************************************************

keep P_M_OWN P_M_RENT  P_Mcorr*_OWN P_Mcorr*_RENT P_M_*_COUNTY year rec*

reshape long P_M@ P_Mcorr@ P_Mcorr2@ P_M@_COUNTY,i(year) j (own_s) s
tab own_s, gen(own)
ren own1 own
ren own2 rent
drop own_s

foreach var of varlist  P_M P_M_COUNTY P_Mcorr P_Mcorr2 {
	gen ln_`var' = log(`var')
	}

gen cur_rec = 0
replace cur_rec = 1 if year>=2007	/* Note that mobility is from t to t-1, so recession starts 2007 */

gen ownXtrend = own*year
gen rentXtrend = rent*year

gen ownXcur_rec = own*cur_rec
gen rentXcur_rec = rent*cur_rec

reg  P_M year if own==1&year<2007
reg  P_M year if own==0&year<2007
reg  P_M own rent ownXtrend rentXtrend if year<2007, noc

cap log close
log using "$results\mobility_regs", replace t

* All moves, percentage points, no HO corr
reg P_M own rent ownXtrend rentXtrend ownXcur_rec rentXcur_rec, noc rob
test ownXtrend = rentXtrend 
local Ftrend = r(p)
test ownXcur_rec=rentXcur_rec
local Frec = r(p)
outreg2 using "$results\ho_rent_below_trend.xml", addstat(Test trend pvalue, `Ftrend', Test recession pvalue, `Frec') excel dec(3) replace 


* All moves, growth rates, no HO corr
reg ln_P_M own rent ownXtrend rentXtrend ownXcur_rec rentXcur_rec, noc rob
test ownXtrend = rentXtrend 
local Ftrend = r(p)
test ownXcur_rec=rentXcur_rec
local Frec = r(p)
outreg2 using "$results\ho_rent_below_trend.xml", addstat(Test trend pvalue, `Ftrend', Test recession pvalue, `Frec') excel dec(3) append

* Intercounty moves, percentage points, no HO corr
reg P_M_COUNTY own rent ownXtrend rentXtrend ownXcur_rec rentXcur_rec, noc rob
test ownXtrend = rentXtrend 
local Ftrend = r(p)
test ownXcur_rec=rentXcur_rec
local Frec = r(p)
outreg2 using "$results\ho_rent_below_trend.xml", addstat(Test trend pvalue, `Ftrend', Test recession pvalue, `Frec') excel dec(3) append

* Intercounty moves, growth rates, no HO corr
reg ln_P_M_COUNTY own rent ownXtrend rentXtrend ownXcur_rec rentXcur_rec, noc rob
test ownXtrend = rentXtrend 
local Ftrend = r(p)
test ownXcur_rec=rentXcur_rec
local Frec = r(p)
outreg2 using "$results\ho_rent_below_trend.xml", addstat(Test trend pvalue, `Ftrend', Test recession pvalue, `Frec') excel dec(3) append

* All moves, percentage points, HO corr 1
reg P_Mcorr own rent ownXtrend rentXtrend ownXcur_rec rentXcur_rec, noc rob
test ownXtrend = rentXtrend 
local Ftrend = r(p)
test ownXcur_rec=rentXcur_rec
local Frec = r(p)
outreg2 using "$results\ho_rent_below_trend.xml", addstat(Test trend pvalue, `Ftrend', Test recession pvalue, `Frec') excel dec(3) append

* All moves, growth rates, HO corr 1
reg ln_P_Mcorr own rent ownXtrend rentXtrend ownXcur_rec rentXcur_rec, noc rob
test ownXtrend = rentXtrend 
local Ftrend = r(p)
test ownXcur_rec=rentXcur_rec
local Frec = r(p)
outreg2 using "$results\ho_rent_below_trend.xml", addstat(Test trend pvalue, `Ftrend', Test recession pvalue, `Frec') excel dec(3) append

cap log close


*=====================================================================================================
* Section 5, Comparing trends pre recession (the results are very similar to the results in table A.1
*=====================================================================================================
cap log close
log using "$results\comparing_trends_pre_rec", replace t
* All moves, percentage points, no HO corr
reg P_M rent year rentXtrend if year<2007, rob

* All moves, growth rates, no HO corr
reg ln_P_M rent year rentXtrend if year<2007, rob

* Intercounty moves, percentage points, no HO corr
reg P_M_COUNTY rent year rentXtrend if year<2007, rob

* Intercounty moves, growth rates, no HO corr
reg ln_P_M_COUNTY rent year rentXtrend if year<2007, rob

* All moves, percentage points, HO corr 1
reg P_Mcorr rent year rentXtrend if year<2007, rob

* All moves, growth rates, HO corr 1
reg ln_P_Mcorr rent year rentXtrend if year<2007, rob
log close


*======================
* Section 5, 5 States *
*======================

cap log close
u 5states_county, clear
log using "$results\5states_exercise", replace t

l fstates_P_M_OWN_J fstates_P_M_RENT_J
tsset year
gen fstates_P_M_OWN_J_rewt =  fstates_M_OWN_reweight/l1.fstates_OWN

di "% change in P_M_OWN_J " fstates_P_M_OWN_J[7]/fstates_P_M_OWN_J[2]-1
di "% change in P_M_RENT_J " fstates_P_M_RENT_J[7]/fstates_P_M_RENT_J[2]-1
di "% change in P_M_RENT_J " fstates_P_M_OWN_J_rewt[7]/fstates_P_M_OWN_J_rewt[2]-1

cap log close


*=========================================
* Section 7 - match-based trend and noise
*=========================================
u matched_mobility, clear
gen year = year(dofm(date))
tsset date

gen P_M_OWN_UE =  l1.matched_M_OWN_UE/l1.matched_U_OWN		/* l1 to match the timing from t-1 to t */
gen P_M_RENT_UE =  l1.matched_M_RENT_UE/l1.matched_U_RENT	/* l1 to match the timing from t-1 to t */
gen P_M_OWN =  l1.matched_M_OWN/l1.matched_OWN		/* l1 to match the timing from t-1 to t */
gen P_M_RENT =  l1.matched_M_RENT/l1.matched_RENT	/* l1 to match the timing from t-1 to t */

/* No 2012, as has only 3 months */
tabout year if year<2012 using "$results\match_base_own_u.csv", c(mean  P_M_OWN_UE median  P_M_OWN_UE sd  P_M_OWN_UE iqr  P_M_OWN_UE) style(csv) sum format(4) replace
tabout year if year<2012 using "$results\match_base_rent_u.csv", c(mean  P_M_RENT_UE median  P_M_RENT_UE sd  P_M_RENT_UE iqr  P_M_RENT_UE) style(csv) sum format(4) replace
tabout year if year<2012 using "$results\match_base_own_all.csv", c(mean  P_M_OWN median  P_M_OWN sd  P_M_OWN iqr  P_M_OWN)  style(csv) sum format(4) replace
tabout year if year<2012 using "$results\match_base_rent_all.csv", c(mean  P_M_RENT median  P_M_RENT sd  P_M_RENT iqr  P_M_RENT) style(csv) sum format(4) replace

cd "$do"
