/* 	This file generates the mobility and counterfactual mobility rates covariance matrix which 
	is later used in the calculation of the standard errors. The cases are as follows:
	(1) Table 1 (A1 A2 B1 B2 C1 C2) Job related mobility (j) is taken from the data. The mobility rates 
	    and the job related mobility rates variances are calcaulted using REPWT, and the off diagonals are 
		assigned for zero. Delta method is applied to get a covariance matrix with only mobility rates and 
		counterfactual mobility rates multiplied by job related mobility. Note that after applying delta, 
		the covariance between mobility rate and counterfactual mobility rate is not zero. 
	(2) Table 1 (A3 A4 B3 B4) Variance of counterfactual mobility rates is a function of homeowner job related 
		mobility in 2007, and renter job-related mobility in 2008 to 2012. The variance of these variables is
		computed using repwts and these are combined using Delta method to get to variance of counterfactual
		mobility rates.
	(3) Table 2 (Col1 Col2) Job related moblity is either deterministic, or is 
	    assigned also in counterfactual. For these cases, can directly calculate the variances of 
		mobility rateXjob realted mobility (observed and counterfactul) using REPWT, and assign diagonal 
		to zero. 
	(4) Table 2 (Col3) Historically high variance need to be calculated from the data using regressions
	    (as there are no REPWT for 1981). The delta method is then applied as in 1. 
	(5) Table 2 (Col4) Historically high variance need to be calculated from the data using regressions
	    (as there are no REPWT for 1981). No need to apply delta method. 
		
	Note: whenever a single counterfactual mobility is used for all years, the covariance matrix of 
	counterfactual mobility rates (m^C_2008 to M^C_2012) is block with the variances also on the off
	diagonal. 
		
	*/ 

   
clear all
set more off
set mem 2g
cd "$output"

*====================================================
* Part 1a: Calculate variance for mobility rates (m)
*====================================================
* Table 1 (A1 A2 B1 B2 C1 C2) Table 2 (Col1 Col3 Col4)
local unemp_boost = 2

use data_for_accounting_monthly, clear
egen tag = tag(year)
keep if tag==1
keep if year>=2008
keep year monthly_avg_OWN age1665_monthly_avg_OWN
save temp.dta, replace

forvalues i = 0(1)160 {

	if `i' == 0 {
		use data_for_accounting, clear
	}
	if `i'>0 {
		u "$output\rep`i'\\data_for_accounting", clear
	}
	
	keep if year>=2008
	qui merge 1:1 year using temp.dta

	*Table 1 A1 and A2
	gen P_M_OWN = M_OWN/monthly_avg_OWN/12
	gen P_M_OWN_UE_1 = `unemp_boost'*P_M_OWN
	gen P_M_OWN_UE_2 = P_M_OWN_UE_1
		
	*Table 1 B1 and B2 and Table 2 Col1, Col3, Col4
	drop P_M_OWN 
	gen P_M_OWN = M_OWN_OUT_COUNTY/monthly_avg_OWN/12
	gen P_M_OWN_UE_5 = `unemp_boost'*P_M_OWN
	gen P_M_OWN_UE_6 = P_M_OWN_UE_5
	gen P_M_OWN_UE_11 = P_M_OWN_UE_5
	gen P_M_OWN_UE_13 = P_M_OWN_UE_5
	gen P_M_OWN_UE_14 = P_M_OWN_UE_5
	
	*Table 1 C1 and C2
	drop P_M_OWN
	gen P_M_OWN =  age1665_M_OWM_corr/age1665_monthly_avg_OWN/12
	gen P_M_OWN_UE_9 = `unemp_boost'*P_M_OWN
	gen P_M_OWN_UE_10 = P_M_OWN_UE_9
	
	keep year  P_M_OWN_UE_*
	sort year
	gen rep = `i'
	if `i' == 0 {
		save m_reps.dta, replace
	}
	if `i'>0 {
		append using m_reps.dta
		save m_reps.dta, replace
	}

}

foreach var of varlist  P_M_OWN_UE_* {
	gen `var'_rep0_temp = `var' if rep==0
	egen `var'_rep0 = min(`var'_rep0_temp), by(year)
	gen sq_dev_`var' = (`var' - `var'_rep0)^2
	egen var_`var' = sum(4/160*sq_dev_`var'),by(year)
}

keep var_* year
egen tag = tag(year)
keep if tag==1
drop tag

sort year
foreach i in 1 2 5 6 9 10 11 13 14  {
	mkmat   var_P_M_OWN_UE_`i', matrix(temp)
	matrix cov_m_`i' = diag(temp)
}

foreach i in 1 2 5 6 9 10 11 13 14  {
	clear
	svmat cov_m_`i'
	save cov_m_`i', replace
}



*==============================================================
* Part 1b: Calculate the variance for job related mobility (j)
*==============================================================
* Table 1 (A1 A2 B1 B2 C1 C2) Table 2 (Col1 Col3 Col4)	(Set for 1 for Col1 and Col4 of Table 2)
local unemp_boost = 2

use data_for_accounting_monthly, clear
egen tag = tag(year)
keep if tag==1
keep if year>=2008
keep year monthly_avg_OWN age1665_monthly_avg_OWN
save temp.dta, replace

forvalues i = 0(1)160 {

	if `i' == 0 {
		use data_for_accounting, clear
	}
	if `i'>0 {
		u "$output\rep`i'\\data_for_accounting", clear
	}
	
	keep if year>=2008
	qui merge 1:1 year using temp.dta

	*Table 1 A1, A2
	gen g_1=M_OWN_J/M_OWN
	gen g_2 = g_1	
		
	*Table 1 B1 and B2 and Table 2 Col3
	gen g_5=M_OWN_J_OUT_COUNTY/M_OWN_OUT_COUNTY
	gen g_6 = g_5	
	gen g_13 = g_5	

	*Table 1 C1, C2
	gen g_9=age1665_M_OWN_J/age1665_M_OWN
	gen g_10 = g_9	

	*Table 2 Col1 and C4
	gen g_11 = 1	
	gen g_14 = 1
	
	keep year  g_*
	sort year
	gen rep = `i'
	if `i' == 0 {
		save j_reps.dta, replace
	}
	if `i'>0 {
		append using j_reps.dta
		save j_reps.dta, replace
	}

}

foreach var of varlist  g_* {
	gen `var'_rep0_temp = `var' if rep==0
	egen `var'_rep0 = min(`var'_rep0_temp), by(year)
	gen sq_dev_`var' = (`var' - `var'_rep0)^2
	egen var_`var' = sum(4/160*sq_dev_`var'),by(year)
}

keep var_* year
egen tag = tag(year)
keep if tag==1
drop tag

sort year
foreach i in 1 2 5 6 9 10 11 13 14  {
	mkmat   var_g_`i', matrix(temp)
	matrix cov_j_`i' = diag(temp)
}

foreach i in 1 2 5 6 9 10 11 13 14  {
	clear
	svmat cov_j_`i'
	save cov_j_`i', replace
}

*==================================================================
* Part 1c: Calculate the variance for mobility rates for job (m*j)
*==================================================================
* Table 1 (A3 A4 B3 B4) Table 2 (Col2)
local unemp_boost = 2

use data_for_accounting_monthly, clear
egen tag = tag(year)
keep if tag==1
keep if year>=2008
keep year monthly_avg_OWN age1665_monthly_avg_OWN
save temp.dta, replace

forvalues i = 0(1)160 {

	if `i' == 0 {
		use data_for_accounting, clear
	}
	if `i'>0 { 
		u "$output\rep`i'\\data_for_accounting", clear
	}
	
	keep if year>=2008
	qui merge 1:1 year using temp.dta

	*Table 1 A3 and A4
	gen g_N=M_OWN_J/M_OWN
	gen P_M_OWN = M_OWN/monthly_avg_OWN/12
	gen P_M_OWN_UE = `unemp_boost'*P_M_OWN
	gen P_M_OWN_J_UE_3 = g_N*P_M_OWN_UE
	gen P_M_OWN_J_UE_4 = P_M_OWN_J_UE_3
		
	*Table 1 B3 and B4 and Table 2 Col2
	drop g_N P_M_OWN P_M_OWN_UE
	gen g_N=M_OWN_J_OUT_COUNTY/M_OWN_OUT_COUNTY
	gen P_M_OWN = M_OWN_OUT_COUNTY/monthly_avg_OWN/12
	gen P_M_OWN_UE = `unemp_boost'*P_M_OWN
	gen P_M_OWN_J_UE_7 = g_N*P_M_OWN_UE
	gen P_M_OWN_J_UE_8 = P_M_OWN_J_UE_7
	gen P_M_OWN_J_UE_12 = P_M_OWN_J_UE_7
	
	keep year  P_M_OWN_J_UE_*
	sort year
	gen rep = `i'
	if `i' == 0 {
		save mj_reps.dta, replace
	}
	if `i'>0 {
		append using mj_reps.dta
		save mj_reps.dta, replace
	}

}

foreach var of varlist  P_M_OWN_J_UE_* {
	gen `var'_rep0_temp = `var' if rep==0
	egen `var'_rep0 = min(`var'_rep0_temp), by(year)
	gen sq_dev_`var' = (`var' - `var'_rep0)^2
	egen var_`var' = sum(4/160*sq_dev_`var'),by(year)
}

keep var_* year
egen tag = tag(year)
keep if tag==1
drop tag

sort year
foreach i in 3 4 7 8 12  {
	mkmat   var_P_M_OWN_J_UE_`i', matrix(temp)
	matrix cov_mj_`i' = diag(temp)
}

foreach i in 3 4 7 8 12  {
	clear
	svmat cov_mj_`i'
	save cov_mj_`i', replace
}



*========================================================================
* Part 2a: Calculate the variance for counterfactual mobility rates (mc)
*========================================================================
* Table 1 (A1 A2 B1 B2 C1 C2) Table 2 (Col1)
local unemp_boost = 2
local unemp_rent = 1.2

use data_for_accounting_monthly, clear
egen tag = tag(year)
keep if tag==1
keep year monthly_avg_OWN age1665_monthly_avg_OWN
save temp.dta, replace

forvalues i = 0(1)160 {

	if `i' == 0 {
		use data_for_accounting, clear
	}
	if `i'>0 {
		u "$output\rep`i'\\data_for_accounting", clear
	}
	
	qui merge 1:1 year using temp.dta

	*Table 1 A1 
	gen P_M_OWN = M_OWN/monthly_avg_OWN/12
	gen P_M_OWN_UE = `unemp_boost'*P_M_OWN
	gen P_M_OWN_hat_temp = P_M_OWN_UE if year==2007
	egen P_M_OWN_hat_1 = min(P_M_OWN_hat_temp)
		
	*Table 1 A2
	gen P_M_OWN_hat_2=0
	
	*Table 1 B1 Table 2 Col1
	drop P_M_OWN P_M_OWN_UE P_M_OWN_hat_temp
	gen P_M_OWN = M_OWN_OUT_COUNTY/monthly_avg_OWN/12
	gen P_M_OWN_UE = `unemp_boost'*P_M_OWN
	gen P_M_OWN_hat_temp = P_M_OWN_UE if year==2007
	egen P_M_OWN_hat_5 = min(P_M_OWN_hat_temp)
	gen P_M_OWN_hat_11=P_M_OWN_hat_5
	
	*Table 1 B2
	gen P_M_OWN_hat_6=0
	
	*Table 1 C1
	drop P_M_OWN P_M_OWN_UE P_M_OWN_hat_temp
	gen P_M_OWN =  age1665_M_OWM_corr/age1665_monthly_avg_OWN/12
	gen P_M_OWN_UE = `unemp_boost'*P_M_OWN
	gen P_M_OWN_hat_temp = P_M_OWN_UE if year==2007
	egen P_M_OWN_hat_9 = min(P_M_OWN_hat_temp)
	
	*Table 1 C2
	gen P_M_OWN_hat_10 = 0

	drop *_temp
	keep year  P_M_OWN_hat_*
	sort year
	keep if year>=2008
	gen rep = `i'
	if `i' == 0 {
		save mc_reps.dta, replace
	}
	if `i'>0 {
		append using mc_reps.dta
		save mc_reps.dta, replace
	}

}

foreach var of varlist  P_M_OWN_hat_* {
	gen `var'_rep0_temp = `var' if rep==0
	egen `var'_rep0 = min(`var'_rep0_temp), by(year)
	gen sq_dev_`var' = (`var' - `var'_rep0)^2
	egen var_`var' = sum(4/160*sq_dev_`var'),by(year)
}

keep var_* year
egen tag = tag(year)
keep if tag==1
drop tag

sort year
foreach i in 1 2 5 6 9 10 11 {
	mkmat   var_P_M_OWN_hat_`i', matrix(temp)
	matrix cov_mc_`i' = temp,temp,temp,temp,temp
}

foreach i in 1 2 5 6 9 10 11 {
	clear
	svmat cov_mc_`i'
	save cov_mc_`i', replace
}



*======================================================================================================
* Compute variance of variables which get used in computing variance of counterfactual mobility (mc*j)*
*======================================================================================================
* Table 1 (A3 A4 B3 B4)

local unemp_boost = 2
local unemp_rent = 1.2

use data_for_accounting_monthly, clear
egen tag = tag(year)
keep if tag==1
keep year monthly_avg_OWN age1665_monthly_avg_OWN
save temp.dta, replace

forvalues i = 0(1)160 {

	if `i' == 0 {
		use data_for_accounting, clear
	}
	if `i'>0 {
		u "$output\rep`i'\\data_for_accounting", clear
	}
	
	qui merge 1:1 year using temp.dta

	*Table 1 A3 
	gen g=M_OWN_J/M_OWN
	gen P_M_OWN = M_OWN/monthly_avg_OWN/12
	gen P_M_OWN_UE = `unemp_boost'*P_M_OWN
	cap drop _merge
	if `i' == 0 {
		merge 1:1 year using M_RENT_reweight_baseline
	}
	if `i' > 0 {
		merge 1:1 year using "$output\rep`i'\\M_RENT_reweight_baseline"
	}
	tsset year
	gen P_M_RENT_J_3 = M_OWN_J_reweight/monthly_avg_OWN/12
	gen P_M_OWN_J_UE_hat_3 = g*P_M_OWN_UE if year==2007
	
	* Table 1 A4
	gen P_M_RENT_J_4 = P_M_RENT_J_3
	gen P_M_OWN_J_UE_hat_4 = P_M_OWN_J_UE_hat_3
	
	*Table 1 B3 
	drop g P_M_OWN P_M_OWN_UE
	gen g=M_OWN_J_OUT_COUNTY/M_OWN_OUT_COUNTY
	gen P_M_OWN = M_OWN_OUT_COUNTY/monthly_avg_OWN/12
	gen P_M_OWN_UE = `unemp_boost'*P_M_OWN
	cap drop _merge
	if `i' == 0 {
		merge 1:1 year using M_RENT_reweight_county, update replace
	}
	if `i' > 0 {
		merge 1:1 year using "$output\rep`i'\\M_RENT_reweight_county", update replace
	}
	tsset year
	gen P_M_RENT_J_7 = M_OWN_J_reweight/monthly_avg_OWN/12
	gen P_M_OWN_J_UE_hat_7 = g*P_M_OWN_UE if year==2007
	
	* Table 1 C4
	gen P_M_RENT_J_8 = P_M_RENT_J_7
	gen P_M_OWN_J_UE_hat_8 = P_M_OWN_J_UE_hat_7
	
	keep year  P_M_OWN_J_UE_hat_* P_M_RENT_J_*
	sort year
	keep if year>=2007
	gen rep = `i'
	if `i' == 0 {
		save mcj_reps_test.dta, replace
	}
	if `i'>0 {
		append using mcj_reps_test.dta
		save mcj_reps_test.dta, replace
	}

}

foreach var of varlist  P_M_OWN_J_UE_hat_* P_M_RENT_J_* {
	gen `var'_rep0_temp = `var' if rep==0
	egen `var'_rep0 = min(`var'_rep0_temp), by(year)
	gen sq_dev_`var' = (`var' - `var'_rep0)^2
	egen var_`var' = sum(4/160*sq_dev_`var'),by(year)
}

keep var_* year
egen tag = tag(year)
keep if tag==1
drop tag

sort year
foreach i in 3 4 7 8 {
	mkmat  var_P_M_OWN_J_UE_hat_`i', matrix(temp)
	mkmat  var_P_M_RENT_J_`i', matrix(temp1)
	matrix temp3 = temp[1,1]\temp1
	matrix cov_mcj_inputs_`i' = diag(temp3)
}

foreach i in 3 4 7 8 {
	clear
	svmat cov_mcj_inputs_`i'
	save cov_mcj_inputs_`i', replace
}


*==================================================================================
* Part 2b: Calculate the variance for counterfactual mobility rates for job (mc*j)
*==================================================================================
* Table 2 (Col2)
local unemp_boost = 2
local unemp_rent = 1.2

use data_for_accounting_monthly, clear
egen tag = tag(year)
keep if tag==1
keep year monthly_avg_OWN age1665_monthly_avg_OWN
save temp.dta, replace

forvalues i = 0(1)160 {

	if `i' == 0 {
		use data_for_accounting, clear
	}
	if `i'>0 {
		u "$output\rep`i'\\data_for_accounting", clear
	}
	
	qui merge 1:1 year using temp.dta

	*Table 2 Col2
	cap drop _merge
	if `i' == 0 {
		merge 1:1 year using M_RENT_reweight_county
	}
	if `i' > 0 {
		merge 1:1 year using "$output\rep`i'\\M_RENT_reweight_county", update replace
	}
	gen P_M_RENT_J = `unemp_rent'*M_OWN_J_reweight/monthly_avg_OWN/12
	gen P_M_OWN_J_UE_hat_12 = P_M_RENT_J if year>2007
		
	keep year  P_M_OWN_J_UE_hat_*
	sort year
	keep if year>=2008
	gen rep = `i'
	if `i' == 0 {
		save mcj_reps.dta, replace
	}
	if `i'>0 {
		append using mcj_reps.dta
		save mcj_reps.dta, replace
	}

}

foreach var of varlist  P_M_OWN_J_UE_hat_* {
	gen `var'_rep0_temp = `var' if rep==0
	egen `var'_rep0 = min(`var'_rep0_temp), by(year)
	gen sq_dev_`var' = (`var' - `var'_rep0)^2
	egen var_`var' = sum(4/160*sq_dev_`var'),by(year)
}

keep var_* year
egen tag = tag(year)
keep if tag==1
drop tag

sort year
foreach i in 12 {
	mkmat  var_P_M_OWN_J_UE_hat_`i', matrix(temp)
	matrix cov_mcj_`i' = diag(temp)
}

foreach i in 12  {
	clear
	svmat cov_mcj_`i'
	save cov_mcj_`i', replace
}



*=============================================================================================
* Part 2c: Calculate the variance for counterfactual mobility rates for historically high (mc)
*=============================================================================================
local unemp_boost = 2

*** Get that variance of annual mobility rates from the regression using 1981 data
u "cps_data" if year==1981|year==1980, clear

egen double max_POP = sum(weight) if year==1981
su max_POP
local max_POP = r(mean)
egen double l1_OWN = sum(weight) if own==10 & empstat<30& year==1980
su l1_OWN
local l1_OWN = r(mean)

gen mig_dummy = migrate1>3 & own==10 & empstat<30 & migrate1!=.
gen msa = metarea
replace msa = statefip if msa>=9997 /* The number of people whose MSA is not identified is huge - approx 70% */

reg mig_dummy [pw=weight] if year==1981, cluster(msa)
matrix var_max_P_M  = e(V)
local var_max_P_M = var_max_P_M[1,1]
matrix max_P_M  = e(b)
local max_P_M = max_P_M[1,1]
 
* Turn into homeowner moblity rate  
use data_for_accounting_monthly, clear
sort year
merge year using data_for_accounting
tab _merge
drop _merge
sort date
tsset date 
local var_mcreg = `var_max_P_M'*(`max_POP'/`l1_OWN')^2
matrix var_mcreg = J(5,5,`var_mcreg')

* Used homeownership in 2008-2012 to turn to monthly
gen w = `unemp_boost '*l1.OWN/monthly_avg_OWN/12 if month==4 
keep if w!=.
keep if year>=2008
mkmat w, matrix(w_temp)
matrix w=diag(w_temp)

matrix cov_mc_13 = w*var_mcreg*w'
matrix cov_mc_14 = cov_mc_13

foreach i in 13 14 {
	clear
	svmat cov_mc_`i'
	save cov_mc_`i', replace
}

cd "$do"
