clear all  
set mem 2000m

cd "$output"

*********************************************************************************************************
* This files prepares the projections of mobility from a regression on trend and recession dummies      *
* The projections are done for 3 optional samples: 													    *
* 1) Baseline (in the labor force)				   													    *
* 2) Age 16-65 						   													    			*
* 3) Age 16-65 with homeownership correction. 	   													    *
* 4) Inter-county 																						*
* 												   													    *
* This is conducted outside of the file accounting_HO_prepare_data since it is runs over all years (and *
* not only 2007 and on) 						   													    *
*																										*
* It generates files called P_M_OWN_hat_*.dta for each case, which conatin the projected mobility rates *
* and the historically high mobility rate for each case.												*
*********************************************************************************************************



*==========================*
* (1) Basline: Labor force *
*==========================*

u year weight state migrate1 own empstat qmigrat1 wks* weeks* whymove using "cps_data", clear

*** Sample correction criteria
keep if empst<30 		/* We focus on in the labor force status */ 

/* Use CPS Prepare the aggregate level sums and collapse by year */
egen POP = sum(weight), by(year) 
label var POP "Labor force level"
egen OWN = sum(weight) if own==10, by(year)
label var OWN "Home owners level"
egen M_OWN = sum(weight) if migrate>1 & own==10 & migrate!=., by(year)
label var M_OWN "Home owners who moved level"
collapse POP OWN M_OWN, by(year)

/* Finding move probabilities */
tsset year
gen own = OWN/POP
gen P_M_OWN = M_OWN/l1.OWN				/* Note that need to devide by the population of owners at t-1, since moving is from t-1 to t */ 

label var P_M_OWN "Moving Rate - Owners"

/* Merging with recession dummy. Note that I tried to make recession dummy consistent with March to March years */
merge year using "$do\march_rec.dta"
drop if _merge~=3
drop _merge
ren march_rec rec_dummy 

keep P_M_OWN year rec_dummy
	
*** Regressions for homeowners' mobility
sort year
gen t=1
replace t=t[_n-1]+1 if t[_n-1]!=.
gen log_P_M_OWN = log(P_M_OWN)
label var log_P_M_OWN "Log homeowners mobility rate"
label var rec_dummy "Recession dummy"

reg log_P_M_OWN t rec_dummy if year<=2007 & year>1984
predict log_P_M_OWN_hat, xb

gen P_M_OWN_hat=exp(log_P_M_OWN_hat)

egen max_P_M_OWN = max(P_M_OWN)
keep year P_M_OWN_hat max_P_M_OWN
sort year

save "$output\P_M_OWN_hat_baseline", replace


*===========================*
* (2) Basline: 16-65 sample *
*===========================*

u year age weight state migrate1 own empstat qmigrat1 wks* weeks* whymove using "cps_data", clear

*** Sample correction criteria
keep if age>15 & age<66 		/* We focus on in the labor force status */ 

/* Use CPS Prepare the aggregate level sums and collapse by year */
egen POP = sum(weight), by(year) 
label var POP "Labor force level"
egen OWN = sum(weight) if own==10, by(year)
label var OWN "Home owners level"
egen M_OWN = sum(weight) if migrate>1 & own==10 & migrate!=., by(year)
label var M_OWN "Home owners who moved level"
collapse POP OWN M_OWN, by(year)

/* Finding move probabilities */
tsset year
gen own = OWN/POP
gen P_M_OWN = M_OWN/l1.OWN				/* Note that need to devide by the population of owners at t-1, since moving is from t-1 to t */ 

label var P_M_OWN "Moving Rate - Owners"

/* Merging with recession dummy. Note that I tried to make recession dummy consistent with March to March years */
merge year using "$do\march_rec.dta"
drop if _merge~=3
drop _merge
ren march_rec rec_dummy 

keep P_M_OWN year rec_dummy
	
*** Regressions for homeowners' mobility
sort year
gen t=1
replace t=t[_n-1]+1 if t[_n-1]!=.
gen log_P_M_OWN = log(P_M_OWN)
label var log_P_M_OWN "Log homeowners mobility rate"
label var rec_dummy "Recession dummy"

reg log_P_M_OWN t rec_dummy if year<=2007 & year>1984
predict log_P_M_OWN_hat, xb

gen P_M_OWN_hat=exp(log_P_M_OWN_hat)

egen max_P_M_OWN = max(P_M_OWN)
keep year P_M_OWN_hat max_P_M_OWN
sort year

save "$output\P_M_OWN_hat_16_65", replace


*==============================================*
* (3) Basline: 16-65 sample with HO correction *
*==============================================*

u year age weight state migrate1 own empstat qmigrat1 wks* weeks* whymove using "cps_data", clear

*** Sample correction criteria
keep if age>15 & age<66 		/* We focus on in the labor force status */ 

/* Use CPS Prepare the aggregate level sums and collapse by year */
egen POP = sum(weight), by(year) 
label var POP "Labor force level"
egen OWN = sum(weight) if own==10, by(year)
label var OWN "Home owners level"
egen OWN_16 = sum(weight) if own==10 & age==16, by(year) 
label var OWN_16 "Home owners age 16 level"
egen OWN_65 = sum(weight) if own==10 & age==65, by(year) 
label var OWN_65 "Home owners age 65 level"
egen M_OWN = sum(weight) if migrate>1 & own==10 & migrate!=., by(year)
label var M_OWN "Home owners who moved level"
egen RENT = sum(weight) if own>10, by(year)
label var RENT "Home renters 16-65 level"
egen RENT_16 = sum(weight) if own>10 & age==16, by(year) 
label var RENT_16 "Renters age 16 level"
egen RENT_65 = sum(weight) if own>10 & age==65, by(year) 
label var RENT_65 "Renters 65 level"


collapse POP RENT* OWN* M_OWN, by(year)

/* Finding move probabilities */
tsset year
foreach name in OWN RENT {
	ren `name'_65 `name'_65_old 
	gen `name'_65=l1.`name'_65_old 
	label var `name'_65 "Home owners age 65 at t-1, level"
}
drop *_old

gen NR= RENT_16-RENT_65
gen dRENT=RENT-l1.RENT

gen M_OWM_corr=M_OWN+dRENT-NR	/* This is the one we use in the counterfactual exercise*/ 
gen P_M_OWN =  M_OWM_corr/l1.OWN

label var P_M_OWN "Moving Rate - Owners"
save data_HO_correction, replace

/* Merging with recession dummy. Note that I tried to make recession dummy consistent with March to March years */
merge year using "$do\march_rec.dta"
drop if _merge~=3
drop _merge
ren march_rec rec_dummy 

keep P_M_OWN year rec_dummy
	
*** Regressions for homeowners' mobility
sort year
gen t=1
replace t=t[_n-1]+1 if t[_n-1]!=.
gen log_P_M_OWN = log(P_M_OWN)
label var log_P_M_OWN "Log homeowners mobility rate"
label var rec_dummy "Recession dummy"

reg log_P_M_OWN t rec_dummy if year<=2007 & year>1984
predict log_P_M_OWN_hat, xb

gen P_M_OWN_hat=exp(log_P_M_OWN_hat)

egen max_P_M_OWN = max(P_M_OWN)
keep year P_M_OWN_hat max_P_M_OWN
sort year

save "$output\P_M_OWN_hat_HO_corr", replace



*================================================================*
* (3a) Basline: 16-65 sample with HO correction (second equation)*
*================================================================*

u year age weight state migrate1 own empstat qmigrat1 wks* weeks* whymove using "cps_data", clear

*** Sample correction criteria
keep if age>15 & age<66 		/* We focus on in the labor force status */ 

/* Use CPS Prepare the aggregate level sums and collapse by year */
egen POP = sum(weight), by(year) 
label var POP "Labor force level"
egen OWN = sum(weight) if own==10, by(year)
label var OWN "Home owners level"
egen OWN_16 = sum(weight) if own==10 & age==16, by(year) 
label var OWN_16 "Home owners age 16 level"
egen OWN_65 = sum(weight) if own==10 & age==65, by(year) 
label var OWN_65 "Home owners age 65 level"
egen M_OWN = sum(weight) if migrate>1 & own==10 & migrate!=., by(year)
label var M_OWN "Home owners who moved level"
egen RENT = sum(weight) if own>10, by(year)
label var RENT "Home renters 16-65 level"
egen RENT_16 = sum(weight) if own>10 & age==16, by(year) 
label var RENT_16 "Renters age 16 level"
egen RENT_65 = sum(weight) if own>10 & age==65, by(year) 
label var RENT_65 "Renters 65 level"


collapse POP RENT* OWN* M_OWN, by(year)

/* Finding move probabilities */
tsset year
foreach name in OWN RENT {
	ren `name'_65 `name'_65_old 
	gen `name'_65=l1.`name'_65_old 
	label var `name'_65 "Home owners age 65 at t-1, level"
}
drop *_old

gen NH= OWN_16-OWN_65
gen NR= RENT_16-RENT_65
gen dRENT=RENT-l1.RENT
gen dOWN=OWN-l1.OWN

gen M_OWM_corr=M_OWN-dOWN+NH
gen P_M_OWN =  M_OWM_corr/l1.OWN

label var P_M_OWN "Moving Rate - Owners"

/* Merging with recession dummy. Note that I tried to make recession dummy consistent with March to March years */
merge year using "$do\march_rec.dta"
drop if _merge~=3
drop _merge
ren march_rec rec_dummy 

keep P_M_OWN year rec_dummy
	
*** Regressions for homeowners' mobility
sort year
gen t=1
replace t=t[_n-1]+1 if t[_n-1]!=.
gen log_P_M_OWN = log(P_M_OWN)
label var log_P_M_OWN "Log homeowners mobility rate"
label var rec_dummy "Recession dummy"

reg log_P_M_OWN t rec_dummy if year<=2007 & year>1984
predict log_P_M_OWN_hat, xb

gen P_M_OWN_hat=exp(log_P_M_OWN_hat)

egen max_P_M_OWN = max(P_M_OWN)
keep year P_M_OWN_hat max_P_M_OWN
sort year

save "$output\P_M_OWN_hat_HO_corr2", replace


*=================================*
* (4) Mobility outside the county *
*=================================*

u year weight state migrate1 own empstat qmigrat1 wks* weeks* whymove using "cps_data", clear

*** Sample correction criteria
keep if empst<30 		/* We focus on in the labor force status */ 

/* Use CPS Prepare the aggregate level sums and collapse by year */
egen POP = sum(weight), by(year) 
label var POP "Labor force level"
egen OWN = sum(weight) if own==10, by(year)
label var OWN "Home owners level"
egen M_OWN = sum(weight) if migrate>3 & own==10 & migrate!=., by(year)
label var M_OWN "Home owners who moved level"
collapse POP OWN M_OWN, by(year)

/* Finding move probabilities */
tsset year
gen own = OWN/POP
gen P_M_OWN = M_OWN/l1.OWN				/* Note that need to devide by the population of owners at t-1, since moving is from t-1 to t */ 

label var P_M_OWN "Moving Rate - Owners"

/* Merging with recession dummy. Note that I tried to make recession dummy consistent with March to March years */
merge year using "$do\march_rec.dta"
drop if _merge~=3
drop _merge
ren march_rec rec_dummy 

keep P_M_OWN year rec_dummy
	
*** Regressions for homeowners' mobility
sort year
gen t=1
replace t=t[_n-1]+1 if t[_n-1]!=.
gen log_P_M_OWN = log(P_M_OWN)
label var log_P_M_OWN "Log homeowners mobility rate"
label var rec_dummy "Recession dummy"

reg log_P_M_OWN t rec_dummy if year<=2007 & year>1984
predict log_P_M_OWN_hat, xb

gen P_M_OWN_hat=exp(log_P_M_OWN_hat)

egen max_P_M_OWN = max(P_M_OWN)
keep year P_M_OWN_hat max_P_M_OWN
sort year

save "$output\P_M_OWN_hat_county", replace

cd "$do"
