/* This file conducts the counterfactual exercise for different configurations (different assumptions on mobility and counterfactual mobility rates)
   For each set of parameters, the file calls "accounting_HO.do" and stores the results in the form of different stata datasets in the "results" folder. 
   This sequence of configurations generates results reported in table 1 and 2 in the paper as well as all the other counterfactual results reported in the text. */ 

clear all
set mem 1500m
cd "$output"


cap postclose table_1
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_1", replace



*=========
* Table 1 
*=========
*** Column A1
* Measure: baseline
* Counterfactual: 2006 homeowners as control

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the homeownership correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always have counterfactual on M
									   1: Use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession (not reweighted)
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 0			/* 0: Don't do counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 1						/* 1 to 14 are the specifications in tables 1 and 2 in the paper 
									   100: Calculate the covariance matrix between unemployment / lf / mobility /cf mobility / j, and does not assign any zeros on off diagonal
									   101: Calculate the covariance matrix between unemployment / lf / mobility /cf mobility / j, and assigns zeros on off diagonal 
											for mobility with everything, but keeping the block of cf mobility with non-zeros on the diagonal. This is equivalent to the REPWT spec, calculated using regression.
									   102: Same as 101, only assigning the covariance between mobility and cf mobility to zero AFTER applying the delta with js. */ 

do "$do\accounting_HO"




*** Column A2
* Measure: baseline
* Counterfactual: homeowners in previous recessions

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 3			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */
									   
global se = 2

do "$do\accounting_HO"


*** Column A3
* Measure: baseline
* Counterfactual: Renters Growth Rates Reweighted

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 6			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */
									   
global se = 3

do "$do\accounting_HO"



*** Column A4
* Measure: baseline
* Counterfactual: Renters Rates Reweighted - percentage points difference

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 5			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 4

do "$do\accounting_HO"



*** Column B1
* Measure: inter-county
* Counterfactual: 2006 homeowners as control

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 5

do "$do\accounting_HO"

*** Column B2
* Measure: inter-county
* Counterfactual: homeowners in previous recessions

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 3			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 6

do "$do\accounting_HO"


*** Column B3
* Measure: inter-county
* Counterfactual: Renters Growth Rates - Reweighted

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 6			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 7

do "$do\accounting_HO"



*** Column B4
* Measure: inter-county
* Counterfactual: Renters Percentage point differences - Reweighted

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 5			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */
									   
global se = 8

do "$do\accounting_HO"

*** Column C1
* Measure: HO correction
* Counterfactual: 2006 homeowners as control

global POP		=	2				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   1				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 9
									   
do "$do\accounting_HO"

*** Column C2
* Measure: HO correction
* Counterfactual: homeowners in previous recessions

global POP		=	2				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   1				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 3			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 10
									   
do "$do\accounting_HO"



postclose table_1


*=========
* Table 2 
*=========

cap postclose table_1
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_3", replace
/* Note that the post keeps calling it table 1, but the file is generated as table 3 */

*** Column 1
* Measure: inter-county
* Counterfactual: Pre-recession, with all moves considered to be move for job. 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	1			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */


global se = 11
									   
do "$do\accounting_HO"



*** Column 2
* Measure: inter-county
* Counterfactual: renters re-weighted LEVELS. 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 7			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 12
									   
do "$do\accounting_HO"

*** Column 3
* Measure: inter-county
* Counterfactual: historically high with j from data. 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 4			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 13
									   
do "$do\accounting_HO"


*** Column 4
* Measure: inter-county
* Counterfactual: historically high, with all moves considered to be move for job. 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
										
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	1			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 4			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 14
									   
do "$do\accounting_HO"

postclose table_1



*===================================
* Seasonality for Panel A, column 1 
*===================================
cap postclose table_1
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_seasonality", replace

* Measure: baseline
* Counterfactual: 2006 homeowners as control - SEASONALITY FROM SIPP

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 1				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"

postclose table_1


*====================================
* 5 states - Note in end of section 5
*====================================

cap postclose table_seasonality
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_nevada", replace

* Measure: Nevada Arizona Florida Michigan California inter-county
* Counterfactual: Renters Growth Gates (also on G) 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 9			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"



* Measure: Nevada Arizona Florida Michigan California inter-county
* Counterfactual: Renters Growth Gates (also on G) - REWEIGHTED

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 11			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"


postclose table_1



*============================
* Moving rate from matching 
*============================
cap postclose table_1
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_matched", replace

*** Column 1
* Measure: Matched - scaled to match annual matches with a single move. 
* Counterfactual: pre-recession 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 8			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 2				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */


global se = 15

do "$do\accounting_HO"


*** Column 2
* Measure: Matched - scaled to match annual matches with a single move. 
* Counterfactual: renters 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 2			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 2				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"




*** Column 3
* Measure: Matched - Assumes only half of matches are mobility 
* Counterfactual: pre-recession 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 8			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 3				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"


*** Column 4
* Measure: Matched - Assumes only half of matches are mobility 
* Counterfactual:renters growth 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 2			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 3				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"



postclose table_1




*=============
* Separations  
*=============
cap postclose table_1
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_sep", replace

*** Case 1: d!=1
* Measure: baseline 
* Counterfactual: pre-recession 

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */
									   
global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 1				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global use_d = 0					/* 0: Use destruction rate of new movers from data
									   1: Use destruction rate of new movers of 1 */

global use_g_E = 0					/* 0: Only include the category "New job or job transfer"
									   1: Also include the category "Other job-related reason" */

global se = 15

do "$do\accounting_HO"


postclose table_1



*=======================================
* Renter Growth Rates Without Reweight 
*=======================================

cap postclose table_1
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_renter_nonrwt", replace

*** Case 1: d!=1
* Measure: baseline
* Counterfactual: Renters Growth Gates - also for mobility for job

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 2			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"


* Measure: inter-county
* Counterfactual: Renters Growth Gates - also for mobility for job

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 2			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 1				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"


* Measure: HO correction
* Counterfactual: Renters Growth Gates - also for mobility for job

global POP		=	2				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   1				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 2			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 15

do "$do\accounting_HO"


* Measure: HO correction
* Counterfactual: Renters Percentage point change - also for mobility for job

global POP		=	2				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   1				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 12			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */

global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 1			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 16


do "$do\accounting_HO"


postclose table_1


*==========================================================
* Footnote on SE using the first column of the first table
*==========================================================
cap postclose table_1
postfile table_1 unemp_boost u2011 u2011c du2011 P_move P_move_c P_MJ P_MJ_c u2007 P_m_own_j_ue_c P_m_own_j_ue se_diff_u using "$results\table_se_FN", replace

*** Spec 1: Exact delta method using MJ in the Delta method.
* Measure: baseline
* Counterfactual: 2006 homeowners as control

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states*/
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */


global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 100						/* 1 to 14 are the specifications in tables 1 and 2 in the paper 
									   100 uses off-diagonal elements in the calculation (with MJ in the Delta) */ 

do "$do\accounting_HO"

*** Spec 2: se 101
* Measure: baseline
* Counterfactual: 2006 homeowners as control

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states*/
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 101						/* 1 to 14 are the specifications in tables 1 and 2 in the paper 
									   100 uses off-diagonal elements in the calculation (with MJ in the Delta) */ 

do "$do\accounting_HO"

*** Spec 3: se 102
* Measure: baseline
* Counterfactual: 2006 homeowners as control

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession 
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states*/
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 102						/* 1 to 14 are the specifications in tables 1 and 2 in the paper 
									   100 uses off-diagonal elements in the calculation (with MJ in the Delta) */ 

do "$do\accounting_HO"





*==========================================
* Std Error with Unemployment Cov Set to 0
*==========================================
*** Column A1
* Measure: baseline
* Counterfactual: 2006 homeowners as control

global POP		=	1				/* 1: Labor Force
									   2: 16-65 
									   */
									   
global HO_corr 	=   0				/* 0: Don't do the correction
									   1: Do the correction.   
									   2: Do the correction with the alternative measure (H_{t-1}\R_{t}+NH_{t})*/
									   
global unemp_boost = 2				/* Choose by which factor to boost unemployed probablity of move relative to the
										probability in the populaion. PSID 96-2006 is 1 to 2, so we take 2 as 
										upper bound */ 

global unemp_rent = 1.2				/* This is only used when a calculation for renters in levels is required. 
										(This type of factor will fall anyway when using growth rates.) */ 

global g_N	= 	1					/* g = Pr(MJ(t)|M(t))  which is at this point conditional or unconditional */
global g 	=   1					/* User provided 'g' for the counterfactual. Becomes useful for $counter_prob_MJ==2 only */

global use_g_for_N = 	0			/* 0: g from the data 
									   1: user provided g.
									   */ 

global counter_prob_M = 1			/* Note: We always counterfactual on M
									   1: use 2006 mobility rate. 	
									   2: Use renter's change in mobility rates in the recession (not reweighted)
									   3: Use homeowner's mobility from previous recessions 
									   4: Use historical high 
									   5: Reneter reweighted percentage points difference
									   6: Use renter's REWEIGHTED percent change in mobility rates in the recession 
									   7: Use renter's REWEIGHTED in LEVELS
									   8: Use matched mobility from the equivalent month in 2006 
									   9: Use mobility from HO in 5 worst hit states, and CF mobility from growth of renters in same states
									   10: Use mobility from HO in 5 worst hit states, and CF mobility from pre-recession in same states 
									   11: Use mobility from HO in 5 worst hit states, and CF mobility from growth of REWEIGHTED renters in same states
									   12: Use renter's percentage change in mobility rates in the recession (not reweighted) */
									   
global seasonality = 0				/* 0: Don't do seasonal adjustment
									   1: Do seasonal adjustment 
									   */

global counter_prob_MJ = 0			/* 0: Don't counterfactual on MJ
									   1: Counterfactual on MJ  using Renters from 07-09 recession */		

global use_county = 0				/* 0: Don't use across county data (baseline)
									   1: Use across county data for moves
									   2: Use the matched data scaled to match annual for mobility rates 
									   3: Use the matched data divided by 2 */

global destruction= 0				/* 0: no counterfactual on destruction rates
									   1: Counterfactual on destruction rates */

global se = 103						/* 1 to 14 are the specifications in tables 1 and 2 in the paper 
									   100: Calculate the covariance matrix between unemployment / lf / mobility /cf mobility / j, and does not assign any zeros on off diagonal
									   101: Calculate the covariance matrix between unemployment / lf / mobility /cf mobility / j, and assigns zeros on off diagonal 
											for mobility with everything, but keeping the block of cf mobility with non-zeros on the diagonal. This is equivalent to the REPWT spec, calculated using regression.
									   102: Same as 101, only assigning the covariance between mobility and cf mobility to zero AFTER applying the delta with js. 
									   103: Same as 101 but with the unemployment block turned to 0.*/ 

do "$do\accounting_HO"

postclose table_1




/* Turn to a nice table */

u "$results\table_1", clear
xpose, clear varname
order _v

save "$results\table_1_xls", replace		/* The excel file turns this to the paper format */ 

u "$results\table_3", clear
xpose, clear varname
order _v

save "$results\table_3_xls", replace		/* The excel file turns this to the paper format */ 

u "$results\table_seasonality", clear
xpose, clear varname
order _v

save "$results\table_seasonality_xls", replace		/* The excel file turns this to the paper format */ 

u "$results\table_matched", clear
xpose, clear varname
order _v

save "$results\table_matched_xls", replace		/* The excel file turns this to the paper format */ 

u "$results\table_nevada", clear
xpose, clear varname
order _v

save "$results\table_nevada_xls", replace		/* The excel file turns this to the paper format */ 


u "$results\table_sep", clear
xpose, clear varname
order _v

save "$results\table_sep_xls", replace		/* The excel file turns this to the paper format */ 

u "$results\table_se_FN", clear
xpose, clear varname
order _v

save "$results\table_se_FN_xls", replace		/* The excel file turns this to the paper format */ 


u "$results\table_renter_nonrwt", clear
xpose, clear varname
order _v

save "$results\table_renter_nonrwt_xls", replace		/* The excel file turns this to the paper format */ 


cd "$do"
