/* Monthly CPS data prepared for accounting exercise. This file generates monthly employment status and ownership data which 
	gets used as input into the accouting exercise. It does this for our baseline sample (labor force), and for the sample
	which gets used in the homeownership correction.
	
	The eventual dataset called data_for_accounting_monthly.dta just has one onsevation for each month from Jan 2006 to March 2012. */ 

/* NOTE: The year variable that is eventually stored is an April to March year.
In particular, April 2006 to March 2007 is labelled as 2007. This corresponds to the fact that the March
CPS 2007 mobility number refers to number of movers from April 2006 to March 2007.
There is a date variable which records the exact year month combination for every observation */

clear all
set mem 1500m
set more off
cd "$output"


***************************************
*** Data Preparation Baseline		***
***************************************
*** Sample selection
u cpsm_2006_2012, clear
keep if empst<5							/* 	We only work with the individuals in the labor force.  
												note that can also drop empst=-1, but these have weight==0 anyway */ 
													
keep year month weight own empst unemp_dur  

*** Use CPS Prepare the aggregate level sums and collapse by year, month
egen double POP = sum(weight), by(year month) 
label var POP "Labor force level"
egen double U = sum(weight) if empst==3|empst==4, by(year month)
label var U "Unemployment level"
egen double OWN = sum(weight) if own==1, by(year month)
label var OWN "Home owners level"
egen double RENT = sum(weight) if own>1, by(year month)
label var RENT "Home renters level"
egen double U_OWN = sum(weight) if own==1 & (empst==3|empst==4), by(year month)
label var U_OWN "Home owners who are unemployed level"
egen double U_s_temp= sum(weight) if unemp_dur<=4 &unemp_dur>=0 & own==1, by(year month)
egen double U_s_OWN	 = max(U_s_temp), by(year month)
drop U_s_temp
label var U_s_OWN "Home owners who are unemployed for 4 weeks or less (level)"
egen double U_RENT = sum(weight) if own>1 & (empst==3|empst==4), by(year month)
label var U_OWN "Home renters who are unemployed level"
collapse POP U OWN RENT U_OWN U_s_OWN U_RENT, by(year month)
sort year month

gen apr_to_march_yr = .		/* April 2006 to March 2007 is labelled as 2007. This corresponds to the fact that the March
								CPS 2007 mobility number refers to number of movers from April 2006 to March 2007 */

local counter = 2006
cou
forvalues i = 1/`r(N)' {
	if month[`i']==4 {
		local counter = `counter' + 1
	}
	di `counter'
	replace apr_to_march_yr = `counter' in `i'

}


/* Generates averages for April to March 
Note that 2006 and 2011 are not over 12 months but 3 and 9 months repectively */
foreach var of varlist  POP- U_RENT {

	egen double avg_`var' = mean(`var'),by(apr_to_march_yr)

}

order apr_to_march_yr

foreach var of varlist  POP- avg_U_RENT {
	ren `var' monthly_`var'
}

sort year month
save baseline_monthly, replace





*************************************************
*** Data Preparation Homeownership Correction ***
*************************************************
/* Note that table 5 is different in sample since it looks at the entire population 16-65 and not just on the labor force */ 
*** Sample selection
u cpsm_2006_2012, clear

keep if age>15 & age<66
keep year month weight own empst
*** Use CPS Prepare the aggregate level sums and collapse by year, month
egen double POP = sum(weight), by(year month) 
label var POP "Labor force level"
egen double LF = sum(weight) if empst<5, by(year month) 
label var LF "Labor force 16-65 level"
egen double U = sum(weight) if empst==3|empst==4, by(year month)
label var U "Unemployment level"
egen double OWN = sum(weight) if own==1, by(year month)
label var OWN "Home owners level"
egen double RENT = sum(weight) if own>1, by(year month)
label var RENT "Home renters level"
egen double U_OWN = sum(weight) if own==1 & (empst==3|empst==4), by(year month)
label var U_OWN "Home owners who are unemployed level"
egen double U_RENT = sum(weight) if own>1 & (empst==3|empst==4), by(year month)
label var U_OWN "Home renters who are unemployed level"
collapse POP LF U OWN RENT U_OWN U_RENT, by(year month)
sort year month


gen apr_to_march_yr = .		/* April 2006 to March 2007 is labelled as 2007. This corresponds to the fact that the March
								CPS 2007 mobility number refers to number of movers from April 2006 to March 2007 */

local counter = 2006
cou
forvalues i = 1/`r(N)' {
	if month[`i']==4 {
		local counter = `counter' + 1
	}
	di `counter'
	replace apr_to_march_yr = `counter' in `i'

}


/* Generates averages for April to March 
Note that 2006 and 2011 are not over 12 months but 3 and 9 months repectively */
foreach var of varlist  POP- U_RENT {

	egen double avg_`var' = mean(`var'),by(apr_to_march_yr)

}

order apr_to_march_yr

foreach var of varlist POP- avg_U_RENT {
	ren `var' age1665_monthly_`var'
} 
sort year month


save ho_corr_monthly, replace




********************
* Merging Datasets *
********************

u baseline_monthly, clear
merge year month using ho_corr_monthly
tab _m 
drop _m
sort year month


gen date = ym(year, month)
format date %tm
drop year
ren apr_to_march_yr year

save data_for_accounting_monthly, replace

erase ho_corr_monthly.dta
erase baseline_monthly.dta

cd "$do"


