*** This file is called by accounting_params

*** Some strings (used to call the correct projection of mobility rates)
*local proj_st "baseline_states"
local proj_st "baseline"
if $POP==2 {
	local proj_st "16_65"
}
if $POP==2 & $HO_corr==1 {
	local proj_st "HO_corr"
}

if $POP==2 & $HO_corr==2 {
	local proj_st "HO_corr2"
}

if $use_county == 1 {
	local proj_st "county"
	*local proj_st "county_noimpute"
}

************************************************
*** Calcualtions for the Accounting Exercise ***
************************************************
use data_for_accounting_monthly, clear
sort year
merge year using data_for_accounting
tab _merge
drop _merge
sort date

/* NOTE: The year in the monthly dataset was corresponding to April to March years. 
So when merging with the March CPS, the March 2007 CPS numbers get assigned to April 2006 to March 2007. */


*================================================================================
* Keeps the variables which will be used, depending on the population being used 
*================================================================================

if $POP == 1 {
	drop age1665*
	gen double LF=POP
	gen double monthly_LF = monthly_POP
}
if $POP == 2 {
	keep year month date age1665*
	order date
	foreach var of varlist age1665_monthly_POP-age1665_M_RENT_corr2 {
		local name=substr("`var'",9,.)
		ren `var' `name'
	}
} 


tsset date


*================================================================================
* Calculate the moving rates depending on whether a homeownership correction 
* is being done, and whether county data is being used
*================================================================================

*** Mobility Rate

gen P_M_OWN = M_OWN/monthly_avg_OWN/12				/* Note that these are monthly moving rates i.e. number of moves divided by average home ownership divided by 12 */
gen P_M_RENT = M_RENT/monthly_avg_RENT/12			/* Note that these are monthly moving rates i.e. number of moves divided by average rentership divided by 12 */
if $HO_corr == 1 {
	replace P_M_OWN =  M_OWM_corr/monthly_avg_OWN/12
	replace P_M_RENT =  M_RENT_corr/monthly_avg_RENT/12	
}

if $HO_corr == 2 {
	replace P_M_OWN =  M_OWM_corr2/monthly_avg_OWN/12
	replace P_M_RENT =  M_RENT_corr2/monthly_avg_RENT/12	
}

if $use_county == 1 {
	replace M_OWN	=M_OWN_OUT_COUNTY
	replace M_RENT	=M_RENT_OUT_COUNTY
	replace M_OWN_J	=M_OWN_J_OUT_COUNTY
	replace M_RENT_J=M_RENT_J_OUT_COUNTY
	replace P_M_OWN = M_OWN_OUT_COUNTY/monthly_avg_OWN/12
	replace P_M_RENT= M_RENT_OUT_COUNTY/monthly_avg_RENT/12	
}
 
if $use_county == 2 {
	sort date
	merge date using matched_mobility
	tsset date
	global unemp_boost = 1 	/* This is measured conditioning on unemployment already */ 

	/* Generate the scaling factor (note timing - matched data mobility is t to t+1) */ 
	tsset date
	gen P_M = (l1.matched_M_OWN + l1.matched_M_RENT)/(l1.matched_OWN + l1.matched_RENT)
	gen P_M_march = (M_OWN + M_RENT)/(monthly_avg_OWN + monthly_avg_RENT)
	egen annual_P_M = sum(P_M), by(year)
	replace annual_P_M =. if date<=tm(2006m3)
	gen scale_P_M = P_M_march/annual_P_M
	
	/* Generate mobility rates */ 
	replace P_M_OWN = scale_P_M*l1.matched_M_OWN_UE/l1.matched_U_OWN
	replace P_M_RENT= scale_P_M*l1.matched_M_RENT_UE/l1.matched_U_RENT	
}

if $use_county == 3 {
	sort date
	merge date using matched_mobility
	tsset date
	global unemp_boost = 1 	/* This is measured conditioning on unemployment already */ 

	/* Scaling factor is 0.5 */ 
	gen scale_P_M = 0.5
	
	/* Generate mobility rates */ 
	replace P_M_OWN = scale_P_M*l1.matched_M_OWN_UE/l1.matched_U_OWN
	replace P_M_RENT= scale_P_M*l1.matched_M_RENT_UE/l1.matched_U_RENT	
}
 
 

label var P_M_OWN "Probability of moving for owners" 
label var P_M_RENT "Probability of moving for renters" 

if $seasonality==1 {
	merge m:1 month using "$do\sipp_seasonality.dta"
	replace P_M_OWN = M_OWN/monthly_OWN*seas_adj
	replace P_M_RENT = M_RENT/monthly_RENT*seas_adj
	sort date
	drop _merge

}

gen P_M_OWN_UE = ${unemp_boost}*P_M_OWN		/* Unemployment boost */
label var P_M_OWN_UE "Probability of moving for unemployed owners" 


*** Mobility For Job 
gen g_N=M_OWN_J/M_OWN				/* g_N is the g used for the backing out of N. The value it takes depends on the value of $use_g_for_N */

if $use_g_for_N==1 {
		replace g_N=$g_N
}


gen P_M_OWN_J_UE=g_N*P_M_OWN_UE

*=============================================
* Calculate mobility rates for counterfactual
*=============================================
*** Mobility rate
if $counter_prob_M == 1 {
	gen P_M_OWN_hat_temp = P_M_OWN_UE if date>=tm(2006m4)&date<=tm(2007m3)
	egen P_M_OWN_hat=min(P_M_OWN_hat_temp),by(month)
}

else if $counter_prob_M==2 {
	gen d_P_M_RENT = P_M_RENT/l1.P_M_RENT
	gen P_M_OWN_hat = P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_hat = l1.P_M_OWN_hat*d_P_M_RENT if date>tm(2007m3)
}

else if $counter_prob_M==3 {
	sort year
	di "`proj_st'"
	merge year using P_M_OWN_hat_`proj_st'
	sort date
	gen double implied_moves_temp = P_M_OWN_hat*l1.OWN if month==4 
	/* Note that tsset is wrt to date. For month==4, l1.OWN is the number of owners from the March CPS for the previous year */
	egen double implied_moves = max(implied_moves_temp),by (year)
	replace  P_M_OWN_hat = implied_moves/monthly_avg_OWN/12
	replace P_M_OWN_hat=${unemp_boost}*P_M_OWN_hat
	drop if _m==2
	drop _m
	tsset date
}

else if $counter_prob_M==4 {
	sort year
	di "`proj_st'"
	merge year using P_M_OWN_hat_`proj_st'
	sort date
	gen double implied_moves_temp = max_P_M_OWN*l1.OWN if month==4 
	/* Note that tsset is wrt to date. For month==4, l1.OWN is the number of owners from the March CPS for the previous year */
	egen double implied_moves = max(implied_moves_temp),by (year)
	replace  P_M_OWN_hat = implied_moves/monthly_avg_OWN/12
	replace P_M_OWN_hat=${unemp_boost}*P_M_OWN_hat
	drop if _m==2
	drop _m
	tsset date
}

else if $counter_prob_M==5 {
	sort year
	di "`proj_st'"
	merge year using M_RENT_reweight_`proj_st'
	tsset date
	replace P_M_RENT=M_OWN_reweight/monthly_avg_OWN/12
	gen d_P_M_RENT = P_M_RENT - l1.P_M_RENT
	gen P_M_OWN_hat = P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_hat = l1.P_M_OWN_hat + d_P_M_RENT if date>tm(2007m3)
	
	gen P_M_RENT_J = M_OWN_J_reweight/monthly_avg_OWN/12
	gen d_P_M_RENT_J = P_M_RENT_J - l1.P_M_RENT_J
	gen P_M_OWN_J_UE_hat_rw = g_N*P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_J_UE_hat_rw = l1.P_M_OWN_J_UE_hat_rw + d_P_M_RENT_J if date>tm(2007m3)
}

else if $counter_prob_M==6 {
	sort year
	di "`proj_st'"
	merge year using M_RENT_reweight_`proj_st'
	tsset date
	replace P_M_RENT=M_OWN_reweight/monthly_avg_OWN/12
	gen d_P_M_RENT = P_M_RENT/l1.P_M_RENT
	gen P_M_OWN_hat = P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_hat = l1.P_M_OWN_hat*d_P_M_RENT if date>tm(2007m3)
	
	gen P_M_RENT_J = M_OWN_J_reweight/monthly_avg_OWN/12
	gen d_P_M_RENT_J = P_M_RENT_J/l1.P_M_RENT_J
	gen P_M_OWN_J_UE_hat_rw = g_N*P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_J_UE_hat_rw = l1.P_M_OWN_J_UE_hat_rw*d_P_M_RENT_J if date>tm(2007m3)

}

else if $counter_prob_M==7 {
	sort year
	di "`proj_st'"
	merge year using M_RENT_reweight_`proj_st'
	tsset date
	replace P_M_RENT=${unemp_rent}*M_OWN_reweight/monthly_avg_OWN/12
	gen P_M_OWN_hat = P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_hat = P_M_RENT if date>tm(2007m3)
	
	gen P_M_RENT_J = ${unemp_rent}*M_OWN_J_reweight/monthly_avg_OWN/12
	gen P_M_OWN_J_UE_hat_rw = g_N*P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_J_UE_hat_rw = P_M_RENT_J  if date>tm(2007m3)

}

else if $counter_prob_M==8 {
	gen P_M_OWN_hat_temp = P_M_OWN_UE if date<=tm(2007m3)	
	egen P_M_OWN_hat=min(P_M_OWN_hat_temp), by(month)
}

else if $counter_prob_M==9 {
	sort year
	di "`proj_st'"
	merge year using 5states_`proj_st'
	tsset date

	replace P_M_OWN_UE= ${unemp_boost}*fstates_P_M_OWN/12
	if ${use_g_for_N}==0{
		replace g_N=fstates_P_M_OWN_J/fstates_P_M_OWN
	}
	replace P_M_OWN_J_UE=g_N*P_M_OWN_UE		/* In this case we assume high leverage states mobility for homeowners before we start */ 
		
	replace P_M_RENT= fstates_P_M_RENT/12
	gen d_P_M_RENT = P_M_RENT/l1.P_M_RENT
	gen P_M_OWN_hat = P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_hat = l1.P_M_OWN_hat*d_P_M_RENT if date>tm(2007m3)
	
	gen P_M_RENT_J = fstates_P_M_RENT_J/12
	gen d_P_M_RENT_J = P_M_RENT_J/l1.P_M_RENT_J
	gen P_M_OWN_J_UE_hat_fstates = ${unemp_boost}*fstates_P_M_OWN_J/12 if date==tm(2007m3)
	replace P_M_OWN_J_UE_hat_fstates = l1.P_M_OWN_J_UE_hat_fstates*d_P_M_RENT_J if date>tm(2007m3)

}

else if $counter_prob_M==10 {
	sort year
	di "`proj_st'"
	merge year using fstates_`proj_st'
	tsset date

	replace P_M_OWN_UE= ${unemp_boost}*fstates_P_M_OWN/12
	if ${use_g_for_N}==0{
		replace g_N=fstates_P_M_OWN_J/fstates_P_M_OWN
	}
	replace P_M_OWN_J_UE=g_N*P_M_OWN_UE		/* In this case we assume high leverage mobility for homeowners before we start */ 

	gen P_M_OWN_hat_temp = P_M_OWN_UE if date==tm(2007m3)
	egen P_M_OWN_hat=min(P_M_OWN_hat_temp)


}

else if $counter_prob_M==11 {
	sort year
	di "`proj_st'"
	merge year using 5states_`proj_st'
	tsset date

	replace P_M_OWN_UE= ${unemp_boost}*fstates_P_M_OWN/12
	if ${use_g_for_N}==0{
		replace g_N=fstates_P_M_OWN_J/fstates_P_M_OWN
	}
	replace P_M_OWN_J_UE=g_N*P_M_OWN_UE		/* In this case we assume high leverage states mobility for homeowners before we start */ 
		
	replace P_M_RENT= fstates_P_M_RENT/12
	gen d_P_M_RENT = P_M_RENT/l1.P_M_RENT
	gen P_M_OWN_hat = P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_hat = l1.P_M_OWN_hat*d_P_M_RENT if date>tm(2007m3)
	
	gen P_M_RENT_J_temp = fstates_M_OWN_J_reweight/l1.fstates_OWN/12 if month==4
	egen P_M_RENT_J = min(P_M_RENT_J_temp),by(year)
	gen d_P_M_RENT_J = P_M_RENT_J/l1.P_M_RENT_J
	gen P_M_OWN_J_UE_hat_fstates = ${unemp_boost}*fstates_P_M_OWN_J/12 if date==tm(2007m3)
	replace P_M_OWN_J_UE_hat_fstates = l1.P_M_OWN_J_UE_hat_fstates*d_P_M_RENT_J if date>tm(2007m3)

}

else if $counter_prob_M==12 {
	gen d_P_M_RENT = P_M_RENT - l1.P_M_RENT
	gen P_M_OWN_hat = P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_hat = l1.P_M_OWN_hat + d_P_M_RENT if date>tm(2007m3)
	
	gen P_M_RENT_J = M_RENT_J/monthly_avg_RENT/12
	gen d_P_M_RENT_J = P_M_RENT_J - l1.P_M_RENT_J
	gen P_M_OWN_J_UE_hat_rw = g_N*P_M_OWN_UE if date==tm(2007m3)
	replace P_M_OWN_J_UE_hat_rw = l1.P_M_OWN_J_UE_hat_rw + d_P_M_RENT_J if date>tm(2007m3)
}


*** Mobility for Jobs
* Generate 'g' for renters growth rates 
gen g_rent = M_RENT_J/M_RENT
gen d_g_rent = g_rent/l1.g_rent


/* We also need to see if we want to do a counterfactual on the g. This depends on the valuetaken by $counter_prob_MJ */
gen g=g_N
if $counter_prob_MJ==1 {
		ren g g_old
		gen g=g_old if date==tm(2007m3)
		replace g=d_g_rent*l1.g if date>tm(2007m3)
}

ren P_M_OWN_hat P_M_OWN_UE_hat
gen P_M_OWN_J_UE_hat=g*P_M_OWN_UE_hat

if $counter_prob_MJ==1 & ($counter_prob_M==5 | $counter_prob_M==6|$counter_prob_M==7|$counter_prob_M==12) {
		replace P_M_OWN_J_UE_hat=P_M_OWN_J_UE_hat_rw
}

if $counter_prob_MJ==1 & ($counter_prob_M==9|$counter_prob_M==11) {
		replace P_M_OWN_J_UE = ${unemp_boost}*fstates_P_M_OWN_J/12 /* We also take non-counterfactual mobility from high leverage */ 
		replace P_M_OWN_J_UE_hat=P_M_OWN_J_UE_hat_fstates
}


***************************
*** Accounting Exercise ***
***************************

*** Back out N(t): N(t) = U(t) - U(t-1) + g*Pr(M(t)|H(t),UE(t-1),NE(t-1))*U_OWN_NE(t-1). 
gen double N = monthly_U - l1.monthly_U + P_M_OWN_J_UE*l1.monthly_U_OWN

*** Back out NN(t) : NN(t) = U_OWN_NE(t) - U_OWN_NE(t-1) + g*Pr(M(t)|H(t),UE(t-1),NE(t-1))*U_OWN_NE(t-1)
gen double NN = monthly_U_OWN - l1.monthly_U_OWN + P_M_OWN_J_UE*l1.monthly_U_OWN

*** Generate counterfactual U_OWN 
gen double U_OWN_hat = monthly_U_OWN if date == tm(2007m3)
replace U_OWN_hat = l1.U_OWN_hat - P_M_OWN_J_UE_hat*l1.U_OWN_hat + NN if date>tm(2007m3)

*** Generate counterfactual U by assigning P_M_OWN_NE=P_M_OWN_PE and using U_OWN_NE_hat 
gen double U_hat = monthly_U if date==tm(2007m3)
replace U_hat = l1.U_hat - P_M_OWN_J_UE_hat*l1.U_OWN_hat + N if date>tm(2007m3)

 
if $destruction==1 {

gen P_M_E = P_M_OWN_UE/$unemp_boost
gen P_M_E_hat = P_M_OWN_UE_hat/$unemp_boost
gen double monthly_E_OWN = monthly_OWN - monthly_U_OWN

gen g_E = M_OWN_NEWJ/M_OWN
if  $use_g_E==1{
	replace g_E = M_OWN_NEWJ_O/M_OWN
}

gen d_rate = (monthly_U_s_OWN/l1.monthly_E_OWN)*1.15
if  $use_d==1 {
	replace d_rate = 1
}

replace N 	= monthly_U 	   	- l1.monthly_U 		  	+ P_M_OWN_J_UE*l1.monthly_U_OWN 	+ g_E*P_M_E*l1.monthly_E_OWN*d_rate
replace NN  = monthly_U_OWN 	- l1.monthly_U_OWN 		+ P_M_OWN_J_UE*l1.monthly_U_OWN 	+ g_E*P_M_E*l1.monthly_E_OWN*d_rate

replace U_OWN_hat = l1.U_OWN_hat 						- P_M_OWN_J_UE_hat*l1.U_OWN_hat		- g_E*P_M_E_hat*(l1.monthly_OWN - l1.U_OWN_hat)*d_rate + NN if date>tm(2007m3)
replace U_hat = l1.U_hat 								- P_M_OWN_J_UE_hat*l1.U_OWN_hat	    - g_E*P_M_E_hat*(l1.monthly_OWN	- l1.U_OWN_hat)*d_rate + N if date>tm(2007m3)

/*


gen monthly_U_s_OWN_hat=monthly_U_s_OWN*(1-g*P_M_OWN_hat)/(1-g_N*P_M_OWN_UE)
drop N NN U_OWN_hat U_hat U_hat_intra

gen N = monthly_U - l1.monthly_U + g_N*P_M_OWN_UE*l1.monthly_U_OWN - monthly_U_s_OWN
gen NN = monthly_U_OWN - l1.monthly_U_OWN + g_N*P_M_OWN_UE*l1.monthly_U_OWN - monthly_U_s_OWN

gen U_OWN_hat = monthly_U_OWN if date == tm(2007m3)
replace U_OWN_hat = l1.U_OWN_hat - g*P_M_OWN_hat*l1.U_OWN_hat + monthly_U_s_OWN_hat + NN if date>tm(2007m3)

gen U_hat = monthly_U if date==tm(2007m3)
replace U_hat = l1.U_hat - g*P_M_OWN_hat*l1.U_OWN_hat + monthly_U_s_OWN_hat + N if date>tm(2007m3)
*/

}

 
*** Report results 
gen u = 100*monthly_U/monthly_LF
su u if date==tm(2012m3)
global u2011=r(mean)

su u if date==tm(2007m3)
global u2007=r(mean)

gen u_hat = 100*U_hat/monthly_LF
su u_hat if date==tm(2012m3)
global u2011c=r(mean)

gen dif_u_hat_u = u - u_hat
su dif_u_hat_u  if date==tm(2012m3)
global du2011=r(mean)

gen uOWN = 100*monthly_U_OWN/monthly_OWN
su uOWN if date==tm(2012m3)
global u2011OWN=r(mean)

gen uOWNc = 100*U_OWN_hat/monthly_OWN
su uOWNc if date==tm(2012m3)
global u2011OWNc=r(mean)

gen duOWN = uOWN  - uOWNc
su duOWN if date==tm(2012m3)
global du2011OWN =r(mean)



su P_M_OWN_UE
global P_move=r(mean) 


su P_M_OWN_UE_hat
global P_move_c=r(mean) 


su g
global P_MJ_c=r(mean)

su g_N
global P_MJ=r(mean)

su P_M_OWN_J_UE_hat if date>tm(2007m3)&date<=tm(2012m3)
global P_m_own_j_ue_c = r(mean)

su P_M_OWN_J_UE if date>tm(2007m3)&date<=tm(2012m3)
global P_m_own_j_ue = r(mean)


*=================
* Standard Errors 
*=================
if $se==1 | $se==2 | $se==5 | $se==6 | $se==9 | $se==10 | $se==11 | $se==13 | $se==14 {

	*** Read the covariance matrix for mobility
	preserve
	use "$output\cov_m_$se.dta", clear
	mkmat  cov*, matrix(m)
	restore

	*** Read the covariance matrix for counterfactual mobility
	preserve
	use "$output\cov_mc_$se.dta", clear
	mkmat  cov*, matrix(mc)
	restore

	*** Read the covariance matrix for job related mobility (j)
	preserve
	use "$output\cov_j_$se.dta", clear
	mkmat  cov*, matrix(j)
	restore

	*** Put together
	matrix zeros_small = J(5,5,0)
	matrix m_mc_j = m,zeros_small,zeros_small\zeros_small,mc,zeros_small\zeros_small,zeros_small,j
	
	*** Generate zeros for off-diagonal of mobility and unemployment
	matrix zeros = J(62,10,0)
	
	*** Read the covariance matrix for unemployment
	if $se==9|$se==10 {
		preserve
		use "$output\U_cov_HOCorr.dta", clear
		mkmat  big_cov1- big_cov62, matrix(U)
		restore
	}
	else {
		preserve
		use "$output\U_cov_baseline.dta", clear
		mkmat  big_cov1- big_cov62, matrix(U)
		restore
	}
	
	*** Apply delta method on m mc j to get m*j and mc*j
	preserve
	keep if month==3 & year>=2008
	keep P_M_OWN_UE P_M_OWN_UE_hat g year 
	
	mkmat g, matrix(j_temp)
	mkmat P_M_OWN_UE, matrix(m_tilde_temp)
	mkmat P_M_OWN_UE_hat, matrix(m_C_tilde_temp)
	matrix eye_small = I(5)
	
	foreach m in j m_tilde m_C_tilde {
		matrix `m' = diag(`m'_temp)
	}
	
	matrix Jac_mob 	= j, zeros_small, m_tilde\zeros_small, j, m_C_tilde
	matrix mob = Jac_mob*m_mc_j*Jac_mob'
	
	
	*** Construct the covariance matrix for all the variables that enter the delta method
	matrix X0 = U,zeros\zeros',mob
	restore 
	
	*** Run delta method (iterative procedure)
	local init_size = rowsof(X0)

	keep if year>=2008|(year==2007&month==3)
	keep  monthly_U_OWN  P_M_OWN_J_UE_hat P_M_OWN_J_UE  monthly_POP year month  U_OWN_hat

	* Iteration one
	local iter = 1
	local iter_1 = `iter'-1
	local B_rows `init_size'
	local B_cols `init_size'
	matrix B_11 = -( P_M_OWN_J_UE_hat[`iter'+1] - P_M_OWN_J_UE[`iter'+1])
	matrix B_12 = J(1,`B_cols'-1,0) 
	matrix B_12[1,1] = 1
	matrix B_12[1,`B_cols'-1 - 9] = monthly_U_OWN[`iter']
	matrix B_12[1,`B_cols'-1 - 4] = -monthly_U_OWN[`iter']
	matrix B_21 = J(`B_rows'-1,1,0)
	matrix B_22 = I(`B_rows'-1)
	matrix B`iter' = B_11,B_12\B_21,B_22
	matrix X`iter' = B`iter'*X`iter_1'*B`iter''

	* All other iterations
	gen P_M_Jac = -1 +  P_M_OWN_J_UE
	gen P_M_hat_Jac = 1 -  P_M_OWN_J_UE_hat

	while `iter'<=59 {
	*while `iter'<=2 {

		local iter = `iter' + 1
		*di `iter'
		local iter_1 = `iter'-1
		local B_rows `init_size' - `iter' + 1
		local B_cols `init_size'- `iter' + 2

		matrix drop B_11 B_12 B_21 B_22
		
		matrix B_11 = [P_M_hat_Jac[`iter'+1],P_M_Jac[`iter'+1] ]
		matrix B_12 = J(1,`B_cols' - 2,0) 
		matrix B_12[1,1] = 1
		local mob_index = floor((`iter'-.1)/12)
		matrix B_12[1,`B_cols' - 2 - 9 + `mob_index'] = monthly_U_OWN[`iter']
		matrix B_12[1,`B_cols' - 2 - 4 + `mob_index'] = -U_OWN_hat[`iter']
		matrix B_21 = J(`B_rows' - 1,2,0)
		matrix B_22 = I(`B_rows'-1)
		matrix B`iter' = B_11,B_12\B_21,B_22
		matrix X`iter' = B`iter'*X`iter_1'*B`iter''
	}	
		
	local LF = monthly_POP[`iter'+1]
	local U =  monthly_U_OWN[`iter'+1]
	local U_C =   U_OWN_hat[`iter'+1]
	
	matrix B_u_diff = [1/`LF', -1/`LF', -(`U_C'-`U')/`LF'^2,0,0,0,0,0,0,0,0,0,0]
	matrix var_diff_u = (B_u_diff*X`iter'*B_u_diff')
	global se_diff_u = sqrt(var_diff_u[1,1])
	di $se_diff_u

}


if  $se==12 {

	*** Read the covariance matrix for mobility for job (m*j)
	preserve
	use "$output\cov_mj_$se.dta", clear
	mkmat  cov*, matrix(mj)
	restore
	
	*** Read the covariance matrix for counterfactual mobility for job (mc*j)
	preserve
	use "$output\cov_mcj_$se.dta", clear
	mkmat  cov*, matrix(mcj)
	restore

	*** Put together
	matrix zeros_small = J(5,5,0)
	matrix mob = mj,zeros_small\zeros_small,mcj

	*** Generate zeros for off-diagonal of mobility and unemployment
	matrix zeros = J(62,10,0)
	
	*** Read the covariance matrix for mobility
	if $se==9|$se==10 {
		preserve
		use "$output\U_cov_HOCorr.dta", clear
		mkmat  big_cov1- big_cov62, matrix(U)
		restore
	}
	else {
		preserve
		use "$output\U_cov_baseline.dta", clear
		mkmat  big_cov1- big_cov62, matrix(U)
		restore
	}
	
	*** Construct the covariance matrix for all the variables that enter the delta method
	matrix X0 = U,zeros\zeros',mob
	
	*** Run delta method (iterative procedure)
	local init_size = rowsof(X0)

	keep if year>=2008|(year==2007&month==3)
	keep  monthly_U_OWN  P_M_OWN_J_UE_hat P_M_OWN_J_UE  monthly_POP year month  U_OWN_hat

	* Iteration one
	local iter = 1
	local iter_1 = `iter'-1
	local B_rows `init_size'
	local B_cols `init_size'
	matrix B_11 = -( P_M_OWN_J_UE_hat[`iter'+1] - P_M_OWN_J_UE[`iter'+1])
	matrix B_12 = J(1,`B_cols'-1,0) 
	matrix B_12[1,1] = 1
	matrix B_12[1,`B_cols'-1 - 9] = monthly_U_OWN[`iter']
	matrix B_12[1,`B_cols'-1 - 4] = -monthly_U_OWN[`iter']
	matrix B_21 = J(`B_rows'-1,1,0)
	matrix B_22 = I(`B_rows'-1)
	matrix B`iter' = B_11,B_12\B_21,B_22
	matrix X`iter' = B`iter'*X`iter_1'*B`iter''

	* All other iterations
	gen P_M_Jac = -1 +  P_M_OWN_J_UE
	gen P_M_hat_Jac = 1 -  P_M_OWN_J_UE_hat

	while `iter'<=59 {
	*while `iter'<=2 {

		local iter = `iter' + 1
		*di `iter'
		local iter_1 = `iter'-1
		local B_rows `init_size' - `iter' + 1
		local B_cols `init_size'- `iter' + 2

		matrix drop B_11 B_12 B_21 B_22
		
		matrix B_11 = [P_M_hat_Jac[`iter'+1],P_M_Jac[`iter'+1] ]
		matrix B_12 = J(1,`B_cols' - 2,0) 
		matrix B_12[1,1] = 1
		local mob_index = floor((`iter'-.1)/12)
		matrix B_12[1,`B_cols' - 2 - 9 + `mob_index'] = monthly_U_OWN[`iter']
		matrix B_12[1,`B_cols' - 2 - 4 + `mob_index'] = -U_OWN_hat[`iter']
		matrix B_21 = J(`B_rows' - 1,2,0)
		matrix B_22 = I(`B_rows'-1)
		matrix B`iter' = B_11,B_12\B_21,B_22
		matrix X`iter' = B`iter'*X`iter_1'*B`iter''
	}	
		
	local LF = monthly_POP[`iter'+1]
	local U =  monthly_U_OWN[`iter'+1]
	local U_C =   U_OWN_hat[`iter'+1]
	
	matrix B_u_diff = [1/`LF', -1/`LF', -(`U_C'-`U')/`LF'^2,0,0,0,0,0,0,0,0,0,0]
	matrix var_diff_u = (B_u_diff*X`iter'*B_u_diff')
	global se_diff_u = sqrt(var_diff_u[1,1])
	di $se_diff_u

}


if $se==3 | $se==4 | $se==7 | $se==8  {

*** Read the covariance matrix for mobility for job (m*j)
	preserve
	use "$output\cov_mj_$se.dta", clear
	mkmat  cov*, matrix(mj)
	restore
	
	*** Read the covariance matrix for counterfactual mobility for job (mc*j)
	preserve
	use "$output\cov_mcj_inputs_$se.dta", clear
	mkmat  cov*, matrix(mcj_inputs)
	restore
	
	if $se==3 | $se==7 {
		preserve 
		keep if month==4&year>2006
		tsset year
		gen P_M_RENT_J_07_temp = P_M_RENT_J if year==2007
		egen P_M_RENT_J_07 = min(P_M_RENT_J_07_temp)
		gen P_M_OWN_J_UE_07_temp = g_N*P_M_OWN_UE if year==2007
		egen P_M_OWN_J_UE_07 = min(P_M_OWN_J_UE_07_temp)
		keep if year>2007
		gen J_m = P_M_RENT_J/P_M_RENT_J_07
		mkmat J_m,matrix(J1)
		gen J_R_07 = - P_M_OWN_J_UE_07*P_M_RENT_J/(P_M_RENT_J_07^2)
		mkmat J_R_07,matrix(J2)
		gen J_R_08_12 = P_M_OWN_J_UE_07/P_M_RENT_J_07
		mkmat J_R_08_12,matrix(J3)
		matrix J3 = diag(J3)
		matrix J = J1,J2,J3
		restore
	}
	if $se==4 | $se==8 {
		matrix J1 = J(5,1,1)
		matrix J2 = J(5,1,-1)
		matrix J3 = I(5)
		matrix J = J1,J2,J3
	}
	matrix mcj = J*mcj_inputs*J'

	
	*** Put together
	matrix zeros_small = J(5,5,0)
	matrix mob = mj,zeros_small\zeros_small,mcj

	*** Generate zeros for off-diagonal of mobility and unemployment
	matrix zeros = J(62,10,0)
	
	*** Read the covariance matrix for mobility
	if $se==9|$se==10 {
		preserve
		use "$output\U_cov_HOCorr.dta", clear
		mkmat  big_cov1- big_cov62, matrix(U)
		restore
	}
	else {
		preserve
		use "$output\U_cov_baseline.dta", clear
		mkmat  big_cov1- big_cov62, matrix(U)
		restore
	}
	
	*** Construct the covariance matrix for all the variables that enter the delta method
	matrix X0 = U,zeros\zeros',mob
	
	*** Run delta method (iterative procedure)
	local init_size = rowsof(X0)

	keep if year>=2008|(year==2007&month==3)
	keep  monthly_U_OWN  P_M_OWN_J_UE_hat P_M_OWN_J_UE  monthly_POP year month  U_OWN_hat

	* Iteration one
	local iter = 1
	local iter_1 = `iter'-1
	local B_rows `init_size'
	local B_cols `init_size'
	matrix B_11 = -( P_M_OWN_J_UE_hat[`iter'+1] - P_M_OWN_J_UE[`iter'+1])
	matrix B_12 = J(1,`B_cols'-1,0) 
	matrix B_12[1,1] = 1
	matrix B_12[1,`B_cols'-1 - 9] = monthly_U_OWN[`iter']
	matrix B_12[1,`B_cols'-1 - 4] = -monthly_U_OWN[`iter']
	matrix B_21 = J(`B_rows'-1,1,0)
	matrix B_22 = I(`B_rows'-1)
	matrix B`iter' = B_11,B_12\B_21,B_22
	matrix X`iter' = B`iter'*X`iter_1'*B`iter''

	* All other iterations
	gen P_M_Jac = -1 +  P_M_OWN_J_UE
	gen P_M_hat_Jac = 1 -  P_M_OWN_J_UE_hat

	while `iter'<=59 {
	*while `iter'<=2 {

		local iter = `iter' + 1
		*di `iter'
		local iter_1 = `iter'-1
		local B_rows `init_size' - `iter' + 1
		local B_cols `init_size'- `iter' + 2

		matrix drop B_11 B_12 B_21 B_22
		
		matrix B_11 = [P_M_hat_Jac[`iter'+1],P_M_Jac[`iter'+1] ]
		matrix B_12 = J(1,`B_cols' - 2,0) 
		matrix B_12[1,1] = 1
		local mob_index = floor((`iter'-.1)/12)
		matrix B_12[1,`B_cols' - 2 - 9 + `mob_index'] = monthly_U_OWN[`iter']
		matrix B_12[1,`B_cols' - 2 - 4 + `mob_index'] = -U_OWN_hat[`iter']
		matrix B_21 = J(`B_rows' - 1,2,0)
		matrix B_22 = I(`B_rows'-1)
		matrix B`iter' = B_11,B_12\B_21,B_22
		matrix X`iter' = B`iter'*X`iter_1'*B`iter''
	}	
		
	local LF = monthly_POP[`iter'+1]
	local U =  monthly_U_OWN[`iter'+1]
	local U_C =   U_OWN_hat[`iter'+1]
	
	matrix B_u_diff = [1/`LF', -1/`LF', -(`U_C'-`U')/`LF'^2,0,0,0,0,0,0,0,0,0,0]
	matrix var_diff_u = (B_u_diff*X`iter'*B_u_diff')
	global se_diff_u = sqrt(var_diff_u[1,1])
	di $se_diff_u
}


if $se==100 {
	*** Read the large covariance matrix (unemployment / Lf / mobility / cf mobility / mobility for job)
	preserve
	use "$output\U_cov_footnote.dta", clear
	mkmat  big_cov*, matrix(v_a_tilde)
	restore
	
	*** Construct the Jacobian, to turn to a 72*72 cov matrix (X0)
	preserve
	keep if month==3 & year>=2008
	keep P_M_OWN_UE P_M_OWN_UE_hat g year 
	
	matrix eye_large = I(62)
	matrix zeros_large = J(62,15,0)
	mkmat g, matrix(j_temp)
	mkmat P_M_OWN_UE, matrix(m_tilde_temp)
	mkmat P_M_OWN_UE_hat, matrix(m_C_tilde_temp)
	matrix eye_small = I(5)
	matrix zeros_small = J(5,5,0)
	matrix zeros_med = J(62,10,0)
	
	foreach m in j m_tilde m_C_tilde {
		matrix `m' = diag(`m'_temp)
	}
	
	matrix Jac_mob 	= j, zeros_small, m_tilde\zeros_small, j, m_C_tilde
	matrix Jac 		= eye_large,zeros_large\zeros_med',Jac_mob
	
	*** Construct the covariance matrix for all the variables that enter the delta method
	matrix X0 = Jac*v_a_tilde*Jac'
	restore 
	
	*** Run delta method (iterative procedure)
	local init_size = rowsof(X0)

	keep if year>=2008|(year==2007&month==3)
	keep  monthly_U_OWN  P_M_OWN_J_UE_hat P_M_OWN_J_UE  monthly_POP year month  U_OWN_hat

	* Iteration one
	local iter = 1
	local iter_1 = `iter'-1
	local B_rows `init_size'
	local B_cols `init_size'
	matrix B_11 = -( P_M_OWN_J_UE_hat[`iter'+1] - P_M_OWN_J_UE[`iter'+1])
	matrix B_12 = J(1,`B_cols'-1,0) 
	matrix B_12[1,1] = 1
	matrix B_12[1,`B_cols'-1 - 9] = monthly_U_OWN[`iter']
	matrix B_12[1,`B_cols'-1 - 4] = -monthly_U_OWN[`iter']
	matrix B_21 = J(`B_rows'-1,1,0)
	matrix B_22 = I(`B_rows'-1)
	matrix B`iter' = B_11,B_12\B_21,B_22
	matrix X`iter' = B`iter'*X`iter_1'*B`iter''

	* All other iterations
	gen P_M_Jac = -1 +  P_M_OWN_J_UE
	gen P_M_hat_Jac = 1 -  P_M_OWN_J_UE_hat

	while `iter'<=59 {
	*while `iter'<=2 {

		local iter = `iter' + 1
		*di `iter'
		local iter_1 = `iter'-1
		local B_rows `init_size' - `iter' + 1
		local B_cols `init_size'- `iter' + 2

		matrix drop B_11 B_12 B_21 B_22
		
		matrix B_11 = [P_M_hat_Jac[`iter'+1],P_M_Jac[`iter'+1] ]
		matrix B_12 = J(1,`B_cols' - 2,0) 
		matrix B_12[1,1] = 1
		local mob_index = floor((`iter'-.1)/12)
		matrix B_12[1,`B_cols' - 2 - 9 + `mob_index'] = monthly_U_OWN[`iter']
		matrix B_12[1,`B_cols' - 2 - 4 + `mob_index'] = -U_OWN_hat[`iter']
		matrix B_21 = J(`B_rows' - 1,2,0)
		matrix B_22 = I(`B_rows'-1)
		matrix B`iter' = B_11,B_12\B_21,B_22
		matrix X`iter' = B`iter'*X`iter_1'*B`iter''
	}	
		
	local LF = monthly_POP[`iter'+1]
	local U =  monthly_U_OWN[`iter'+1]
	local U_C =   U_OWN_hat[`iter'+1]
	
	matrix B_u_diff = [1/`LF', -1/`LF', -(`U_C'-`U')/`LF'^2,0,0,0,0,0,0,0,0,0,0]
	matrix var_diff_u = (B_u_diff*X`iter'*B_u_diff')
	global se_diff_u = sqrt(var_diff_u[1,1])
	di $se_diff_u
}

if $se==101 {
	*** Read the large covariance matrix (unemployment / Lf / mobility / cf mobility / mobility for job)
	preserve
	use "$output\U_cov_footnote.dta", clear
	mkmat  big_cov*, matrix(v_a_tilde_temp)
	
	*** Assign off diagonals of mobility (other than the CF mobility block) to zero
	matrix U = v_a_tilde_temp[1..62,1..62]
	matrix m_temp = diag(vecdiag(v_a_tilde_temp[63..67,63..67]))
	matrix m_C_temp = v_a_tilde_temp[68..72,68..72]
	matrix j_temp = diag(vecdiag(v_a_tilde_temp[73..77,73..77]))
	
	matrix zeros_small = J(5,5,0)
	matrix mob = m_temp,zeros_small,zeros_small\ /// 
				 zeros_small,m_C_temp,zeros_small\ ///
				 zeros_small,zeros_small,j_temp
	matrix zeros_large = J(62,15,0)
	matrix v_a_tilde = U,zeros_large\zeros_large',mob
	restore
	
	*** Construct the Jacobian, to turn to a 72*72 cov matrix (X0)
	preserve
	keep if month==3 & year>=2008
	keep P_M_OWN_UE P_M_OWN_UE_hat g year 
	
	matrix eye_large = I(62)
	matrix zeros_large = J(62,15,0)
	mkmat g, matrix(j_temp)
	mkmat P_M_OWN_UE, matrix(m_tilde_temp)
	mkmat P_M_OWN_UE_hat, matrix(m_C_tilde_temp)
	matrix eye_small = I(5)
	matrix zeros_small = J(5,5,0)
	matrix zeros_med = J(62,10,0)
	
	foreach m in j m_tilde m_C_tilde {
		matrix `m' = diag(`m'_temp)
	}
	
	matrix Jac_mob 	= j, zeros_small, m_tilde\zeros_small, j, m_C_tilde
	matrix Jac 		= eye_large,zeros_large\zeros_med',Jac_mob
	
	*** Construct the covariance matrix for all the variables that enter the delta method 
	matrix X0 = Jac*v_a_tilde*Jac'
	restore 
	
	*** Run delta method (iterative procedure)
	local init_size = rowsof(X0)

	keep if year>=2008|(year==2007&month==3)
	keep  monthly_U_OWN  P_M_OWN_J_UE_hat P_M_OWN_J_UE  monthly_POP year month  U_OWN_hat

	* Iteration one
	local iter = 1
	local iter_1 = `iter'-1
	local B_rows `init_size'
	local B_cols `init_size'
	matrix B_11 = -( P_M_OWN_J_UE_hat[`iter'+1] - P_M_OWN_J_UE[`iter'+1])
	matrix B_12 = J(1,`B_cols'-1,0) 
	matrix B_12[1,1] = 1
	matrix B_12[1,`B_cols'-1 - 9] = monthly_U_OWN[`iter']
	matrix B_12[1,`B_cols'-1 - 4] = -monthly_U_OWN[`iter']
	matrix B_21 = J(`B_rows'-1,1,0)
	matrix B_22 = I(`B_rows'-1)
	matrix B`iter' = B_11,B_12\B_21,B_22
	matrix X`iter' = B`iter'*X`iter_1'*B`iter''

	* All other iterations
	gen P_M_Jac = -1 +  P_M_OWN_J_UE
	gen P_M_hat_Jac = 1 -  P_M_OWN_J_UE_hat

	while `iter'<=59 {
	*while `iter'<=2 {

		local iter = `iter' + 1
		*di `iter'
		local iter_1 = `iter'-1
		local B_rows `init_size' - `iter' + 1
		local B_cols `init_size'- `iter' + 2

		matrix drop B_11 B_12 B_21 B_22
		
		matrix B_11 = [P_M_hat_Jac[`iter'+1],P_M_Jac[`iter'+1] ]
		matrix B_12 = J(1,`B_cols' - 2,0) 
		matrix B_12[1,1] = 1
		local mob_index = floor((`iter'-.1)/12)
		matrix B_12[1,`B_cols' - 2 - 9 + `mob_index'] = monthly_U_OWN[`iter']
		matrix B_12[1,`B_cols' - 2 - 4 + `mob_index'] = -U_OWN_hat[`iter']
		matrix B_21 = J(`B_rows' - 1,2,0)
		matrix B_22 = I(`B_rows'-1)
		matrix B`iter' = B_11,B_12\B_21,B_22
		matrix X`iter' = B`iter'*X`iter_1'*B`iter''
	}	
		
	local LF = monthly_POP[`iter'+1]
	local U =  monthly_U_OWN[`iter'+1]
	local U_C =   U_OWN_hat[`iter'+1]
	
	matrix B_u_diff = [1/`LF', -1/`LF', -(`U_C'-`U')/`LF'^2,0,0,0,0,0,0,0,0,0,0]
	matrix var_diff_u = (B_u_diff*X`iter'*B_u_diff')
	global se_diff_u = sqrt(var_diff_u[1,1])
	di $se_diff_u
}


if $se==102 {
	*** Read the large covariance matrix (unemployment / Lf / mobility / cf mobility / mobility for job)
	preserve
	use "$output\U_cov_footnote.dta", clear
	mkmat  big_cov*, matrix(v_a_tilde_temp)
	
	*** Assign off diagonals of mobility (other than the CF mobility block) to zero
	matrix U = v_a_tilde_temp[1..62,1..62]
	matrix m_temp = diag(vecdiag(v_a_tilde_temp[63..67,63..67]))
	matrix m_C_temp = v_a_tilde_temp[68..72,68..72]
	matrix j_temp = diag(vecdiag(v_a_tilde_temp[73..77,73..77]))
	
	matrix zeros_small = J(5,5,0)
	matrix mob = m_temp,zeros_small,zeros_small\ /// 
				 zeros_small,m_C_temp,zeros_small\ ///
				 zeros_small,zeros_small,j_temp
	matrix zeros_large = J(62,15,0)
	matrix v_a_tilde = U,zeros_large\zeros_large',mob
	restore
	
	*** Construct the Jacobian, to turn to a 72*72 cov matrix (X0)
	preserve
	keep if month==3 & year>=2008
	keep P_M_OWN_UE P_M_OWN_UE_hat g year 
	
	matrix eye_large = I(62)
	matrix zeros_large = J(62,15,0)
	mkmat g, matrix(j_temp)
	mkmat P_M_OWN_UE, matrix(m_tilde_temp)
	mkmat P_M_OWN_UE_hat, matrix(m_C_tilde_temp)
	matrix eye_small = I(5)
	matrix zeros_small = J(5,5,0)
	matrix zeros_med = J(62,10,0)
	
	foreach m in j m_tilde m_C_tilde {
		matrix `m' = diag(`m'_temp)
	}
	
	matrix Jac_mob 	= j, zeros_small, m_tilde\zeros_small, j, m_C_tilde
	matrix Jac 		= eye_large,zeros_large\zeros_med',Jac_mob
	
	*** Construct the covariance matrix for all the variables that enter the delta method - assign the correlations of M and MC to zero
	matrix X0 = Jac*v_a_tilde*Jac'
	matrix zeros = J(67,5,0)
	matrix X0_1 = X0[1..67,1..67]
	matrix X0_2 = X0[68..72,68..72]
	matrix X0 = X0_1,zeros\zeros',X0_2
	restore 
	
	*** Run delta method (iterative procedure)
	local init_size = rowsof(X0)

	keep if year>=2008|(year==2007&month==3)
	keep  monthly_U_OWN  P_M_OWN_J_UE_hat P_M_OWN_J_UE  monthly_POP year month  U_OWN_hat

	* Iteration one
	local iter = 1
	local iter_1 = `iter'-1
	local B_rows `init_size'
	local B_cols `init_size'
	matrix B_11 = -( P_M_OWN_J_UE_hat[`iter'+1] - P_M_OWN_J_UE[`iter'+1])
	matrix B_12 = J(1,`B_cols'-1,0) 
	matrix B_12[1,1] = 1
	matrix B_12[1,`B_cols'-1 - 9] = monthly_U_OWN[`iter']
	matrix B_12[1,`B_cols'-1 - 4] = -monthly_U_OWN[`iter']
	matrix B_21 = J(`B_rows'-1,1,0)
	matrix B_22 = I(`B_rows'-1)
	matrix B`iter' = B_11,B_12\B_21,B_22
	matrix X`iter' = B`iter'*X`iter_1'*B`iter''

	* All other iterations
	gen P_M_Jac = -1 +  P_M_OWN_J_UE
	gen P_M_hat_Jac = 1 -  P_M_OWN_J_UE_hat

	while `iter'<=59 {
	*while `iter'<=2 {

		local iter = `iter' + 1
		*di `iter'
		local iter_1 = `iter'-1
		local B_rows `init_size' - `iter' + 1
		local B_cols `init_size'- `iter' + 2

		matrix drop B_11 B_12 B_21 B_22
		
		matrix B_11 = [P_M_hat_Jac[`iter'+1],P_M_Jac[`iter'+1] ]
		matrix B_12 = J(1,`B_cols' - 2,0) 
		matrix B_12[1,1] = 1
		local mob_index = floor((`iter'-.1)/12)
		matrix B_12[1,`B_cols' - 2 - 9 + `mob_index'] = monthly_U_OWN[`iter']
		matrix B_12[1,`B_cols' - 2 - 4 + `mob_index'] = -U_OWN_hat[`iter']
		matrix B_21 = J(`B_rows' - 1,2,0)
		matrix B_22 = I(`B_rows'-1)
		matrix B`iter' = B_11,B_12\B_21,B_22
		matrix X`iter' = B`iter'*X`iter_1'*B`iter''
	}	
		
	local LF = monthly_POP[`iter'+1]
	local U =  monthly_U_OWN[`iter'+1]
	local U_C =   U_OWN_hat[`iter'+1]
	
	matrix B_u_diff = [1/`LF', -1/`LF', -(`U_C'-`U')/`LF'^2,0,0,0,0,0,0,0,0,0,0]
	matrix var_diff_u = (B_u_diff*X`iter'*B_u_diff')
	global se_diff_u = sqrt(var_diff_u[1,1])
	di $se_diff_u
}



if $se==103 {
	*** Read the large covariance matrix (unemployment / Lf / mobility / cf mobility / mobility for job)
	preserve
	use "$output\U_cov_footnote.dta", clear
	mkmat  big_cov*, matrix(v_a_tilde)
	restore
	
	*** Construct the Jacobian, to turn to a 72*72 cov matrix (X0)
	preserve
	keep if month==3 & year>=2008
	keep P_M_OWN_UE P_M_OWN_UE_hat g year 
	
	matrix eye_large = I(62)
	matrix zeros_large = J(62,15,0)
	mkmat g, matrix(j_temp)
	mkmat P_M_OWN_UE, matrix(m_tilde_temp)
	mkmat P_M_OWN_UE_hat, matrix(m_C_tilde_temp)
	matrix eye_small = I(5)
	matrix zeros_small = J(5,5,0)
	matrix zeros_med = J(62,10,0)
	
	foreach m in j m_tilde m_C_tilde {
		matrix `m' = diag(`m'_temp)
	}
	
	matrix Jac_mob 	= j, zeros_small, m_tilde\zeros_small, j, m_C_tilde
	matrix Jac 		= eye_large,zeros_large\zeros_med',Jac_mob
	
	
	
	*** Construct the covariance matrix for all the variables that enter the delta method
	matrix X0 = Jac*v_a_tilde*Jac'
	matrix X0_1 = J(62,62,0)
	matrix X0_2 = J(62,10,0)
	matrix X0_3 = X0[63..72,63..72]
	matrix X0 = X0_1,X0_2\X0_2',X0_3
	restore 
	
	*** Run delta method (iterative procedure)
	local init_size = rowsof(X0)

	keep if year>=2008|(year==2007&month==3)
	keep  monthly_U_OWN  P_M_OWN_J_UE_hat P_M_OWN_J_UE  monthly_POP year month  U_OWN_hat

	* Iteration one
	local iter = 1
	local iter_1 = `iter'-1
	local B_rows `init_size'
	local B_cols `init_size'
	matrix B_11 = -( P_M_OWN_J_UE_hat[`iter'+1] - P_M_OWN_J_UE[`iter'+1])
	matrix B_12 = J(1,`B_cols'-1,0) 
	matrix B_12[1,1] = 1
	matrix B_12[1,`B_cols'-1 - 9] = monthly_U_OWN[`iter']
	matrix B_12[1,`B_cols'-1 - 4] = -monthly_U_OWN[`iter']
	matrix B_21 = J(`B_rows'-1,1,0)
	matrix B_22 = I(`B_rows'-1)
	matrix B`iter' = B_11,B_12\B_21,B_22
	matrix X`iter' = B`iter'*X`iter_1'*B`iter''

	* All other iterations
	gen P_M_Jac = -1 +  P_M_OWN_J_UE
	gen P_M_hat_Jac = 1 -  P_M_OWN_J_UE_hat

	while `iter'<=59 {
	*while `iter'<=2 {

		local iter = `iter' + 1
		*di `iter'
		local iter_1 = `iter'-1
		local B_rows `init_size' - `iter' + 1
		local B_cols `init_size'- `iter' + 2

		matrix drop B_11 B_12 B_21 B_22
		
		matrix B_11 = [P_M_hat_Jac[`iter'+1],P_M_Jac[`iter'+1] ]
		matrix B_12 = J(1,`B_cols' - 2,0) 
		matrix B_12[1,1] = 1
		local mob_index = floor((`iter'-.1)/12)
		matrix B_12[1,`B_cols' - 2 - 9 + `mob_index'] = monthly_U_OWN[`iter']
		matrix B_12[1,`B_cols' - 2 - 4 + `mob_index'] = -U_OWN_hat[`iter']
		matrix B_21 = J(`B_rows' - 1,2,0)
		matrix B_22 = I(`B_rows'-1)
		matrix B`iter' = B_11,B_12\B_21,B_22
		matrix X`iter' = B`iter'*X`iter_1'*B`iter''
	}	
		
	local LF = monthly_POP[`iter'+1]
	local U =  monthly_U_OWN[`iter'+1]
	local U_C =   U_OWN_hat[`iter'+1]
	
	matrix B_u_diff = [1/`LF', -1/`LF', -(`U_C'-`U')/`LF'^2,0,0,0,0,0,0,0,0,0,0]
	matrix var_diff_u = (B_u_diff*X`iter'*B_u_diff')
	global se_diff_u = sqrt(var_diff_u[1,1])
	di $se_diff_u
}



post table_1 (${unemp_boost}) (${u2011}) (${u2011c}) (${du2011}) (${P_move}) (${P_move_c}) (${P_MJ}) (${P_MJ_c}) (${u2007}) (${P_m_own_j_ue_c}) (${P_m_own_j_ue}) (${se_diff_u})
global se_diff_u 99

