#include "SolutionPrinter.h"

#pragma region Private Declarations

void PrintStdSol(SolutionSet * Set, EconParams * econParams, printLevel level);
void PrintStdSolSetEq();
void PrintStdSolSetNum(SolutionSet * Set, EconParams * econParams);

void PrintCorASol(SolutionSet * Set, int part, EconParams * econParams, printLevel level);
void PrintCorASolSetEq(int part);
void PrintCorASolSetNum(SolutionSet * Set, int part, EconParams * econParams);

void PrintCorH2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level);
void PrintCorH2SolSetEq(int part);
void PrintCorH2SolSetNum(SolutionSet * Set, int part, EconParams * econParams);

void PrintCorH1Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level);
void PrintCorH1SolSetEq(int part);
void PrintCorH1SolSetNum(SolutionSet * Set, int part, EconParams * econParams);

void PrintCorA_H2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level);
void PrintCorA_H2SolSetEq(int part);
void PrintCorA_H2SolSetNum(SolutionSet * Set, int part, EconParams * econParams);

void PrintCorA_H1Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level);
void PrintCorA_H1SolSetEq(int part);
void PrintCorA_H1SolSetNum(SolutionSet * Set, int part, EconParams * econParams);

void PrintCorH1_H2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level);
void PrintCorH1_H2SolSetEq(int part);
void PrintCorH1_H2SolSetNum(SolutionSet * Set, int part, EconParams * econParams);

void PrintCorA_H1_H2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level);
void PrintCorA_H1_H2SolSetEq(int part);
void PrintCorA_H1_H2SolSetNum(SolutionSet * Set, int part, EconParams * econParams);

#pragma endregion


#pragma region Templates

const char * fConstNum = "%s: %f (exogenic)\n";
const char * fConstNum_eq = "eq%s = %s;\n";

const char * fCalcParam2_Generic = "%s: %f = ((%f / %f) * (%f / %f) * (%f ^ (1 / %f))) ^ %f\n";
const char * fCalcParam2_Generic_eq = "eq%s = ((%s / %s) * (%s / %s) * (%s ^ (1 / %s))) ^ %s;\n";

const char * fTfromL = "%s: %f = %i - %f\n";
const char * fTfromL_eq = "eq%s = %s - %s;\n";

const char * fCalcH = "%s: %f = %i - %f - %f\n";
const char * fCalcH_eq = "eq%s = %s - %s - %s;\n";

const char * fCalcY = "Y: %f = %f + (%f * %f) + (%f * %f)\n";
const char * fCalcY_eq = "eqY = %s + (%s * %s) + (%s * %s);\n";

const char * fCalcAtY = "atY: %f = %f * (%f ^ (1 - %f)) - %f\n";
const char * fCalcAtY_eq = "eqatY = %s * (%s ^ (1 - %s)) - %s;\n";

const char * fCalcMY = "mY: %f = %f * (1 - %f) * (%f ^ ( - %f))\n";
const char * fCalcMY_eq = "eqmY = %s * (1 - %s) * (%s ^ ( - %s));\n";

const char * fCalcMU_Generic = "%s: %f = (%f * (%f ^ (1 - 1 / %f)) + %f * (%f ^ (1 - 1 / %f))) ^ (-%f)\n";
const char * fCalcMU_Generic_eq = "eq%s = (%s * (%s ^ (1 - 1 / %s)) + %s * (%s ^ (1 - 1 / %s))) ^ (-%s);\n";

const char * fCalcC_Generic = "C: %f = (%f * %f * %f / (%f * %f * (%f ^ (-1 / %f)))) ^ %f\n";
const char * fCalcC_Generic_eq = "eqC = (%s * %s * %s / (%s * %s * (%s ^ (-1 / %s)))) ^ %s;\n";

const char * fCalcCfromA = "C: %f =  %f + %f - %f / (1 + %f)\n";
const char * fCalcCfromA_eq = "eqC =  %s + %s - %s / (1 + %s);\n";

const char * fCalcA = "A: %f = (1 + %f)*(%f + %f - %f)\n";
const char * fCalcA_eq = "eqA = (1 + %s)*(%s + %s - %s);\n";

const char * fCalcLamParam_Generic = "%s: %f = (((%f / %f) * (%f / %f + %f)) ^ (%f)) * (%f ^ (%f / %f))\n";
const char * fCalcLamParam_Generic_eq = "eq%s = (((%s / %s) * (%s / %s + %s)) ^ (%s)) * (%s ^ (%s / %s));\n";

const char * fCalcVFromAll = "V: %f = %f * (1 / (1 - 1 / %f)) * %f ^ (1 - 1 / %f) ...\n - (1 / (1 - %f)) * (%f * %f ^ (1 - 1 / %f) + %f * %f ^ (1 - 1 / %f)) ^ (1 - %f)...\n - (1 / (1 - %f)) * (%f * %f ^ (1 - 1 / %f) + %f * %f ^ (1 - 1 / %f)) ^ (1 - %f)...\n + %f * %f\n";
const char * fCalcVFromAll_eq = "eqV = %s * (1 / (1 - 1 / %s)) * %s ^ (1 - 1 / %s) ...\n - (1 / (1 - %s)) * (%s * %s ^ (1 - 1 / %s) + %s * %s ^ (1 - 1 / %s)) ^ (1 - %s)...\n - (1 / (1 - %s)) * (%s * %s ^ (1 - 1 / %s) + %s * %s ^ (1 - 1 / %s)) ^ (1 - %s)...\n + %s * %s;\n";


#pragma endregion

void PrintSolutionPart(SolutionSet * Set, EconParams * econParams, printLevel level, int part)
{
	switch (Set->CorIndex)
	{
	case sol_corner::std_cor:
		PrintStdSol(Set, econParams, level);
		break;

	case sol_corner::cor_A:
		PrintCorASol(Set, part, econParams, level);
		break;

	case sol_corner::cor_h2:
		PrintCorH2Sol(Set, part, econParams, level);
		break;

	case sol_corner::cor_h1:
		PrintCorH1Sol(Set, part, econParams, level);
		break;

	case sol_corner::cor_A_h2:
		PrintCorA_H2Sol(Set, part, econParams, level);
		break;

	case sol_corner::cor_A_h1:
		PrintCorA_H1Sol(Set, part, econParams, level);
		break;

	case sol_corner::cor_h1_h2:
		PrintCorH1_H2Sol(Set, part, econParams, level);
		break;

	case sol_corner::cor_A_h1_h2:
		PrintCorA_H1_H2Sol(Set, part, econParams, level);
		break;

	default:
		break;
	}
}

void mexPrintSolutionSet(SolutionSet * Set, int j, int k, int l, int m)
{
	bool isInvalid = (Set->V == -DBL_MAX);
	mexPrintf("\n~~~~~ CORNER[%i] for (%i,%i,%i,%i) ~~~~~\n", Set->CorIndex, j, k, l, m);

	mexPrintf("The Solver Status was: %i\n", Set->status);
	if (isInvalid)
	{
		mexPrintf("The solution found is invalid.\n", Set->L1, Set->T1, Set->L2, Set->T2);
	}
	else
	{
		mexPrintf("L1: %f, T1: %f, L2: %f, T2: %f\n", Set->L1, Set->T1, Set->L2, Set->T2);
		mexPrintf("H1: %f, H2: %f, Y:  %f, mY: %f\n", Set->H1, Set->H2, Set->Y, Set->mY);
		mexPrintf("A:  %f, C:  %f,\nF:  %f \nV:  %f\n", Set->A, Set->C, Set->F, Set->V);
		mexPrintf("EUFunc(A): %f\n", Set->EU);
	}

	mexPrintf("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
}

// CalcFullSolutionSetStd:

void PrintStdSol(SolutionSet * Set, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintStdSolSetEq();
		return;
	case printLevel::enum_printLevel_numeric:
		PrintStdSolSetNum(Set, econParams);
		return;

	default:
		return;
	}
}

void PrintStdSolSetEq()
{
	//L1 = LT1.x0;
	mexPrintf(fConstNum_eq, "L1", "LT1.x0");
	//T1 = LT1.x1;
	mexPrintf(fConstNum_eq, "T1", "LT1.x1");
	//CalcL2fromL1(L1);
	mexPrintf(fCalcParam2_Generic_eq, "L2", "w1", "w2", "mu_l2_tild", "mu_l1_tild", "eqL1", "psi_l1", "psi_l2");
	//CalcT2fromT1(T1);
	mexPrintf(fCalcParam2_Generic_eq, "T2", "w1", "w2", "mu_t2_tild", "mu_t1_tild", "eqT1", "psi_t1", "psi_t2");
	//CalcH(L1, T1);
	mexPrintf(fCalcH_eq, "H1", "L_bar", "eqL1", "eqT1");
	//CalcH(L2, T2);
	mexPrintf(fCalcH_eq, "H2", "L_bar", "eqL2", "eqT2");
	//CalcY(H1, H2)
	mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
	//CalcAtY(Y);
	mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu", "tau_gamma");
	//CalcMY(Y);
	mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
	//CalcMUL(L1, L2);
	mexPrintf(fCalcMU_Generic_eq, "MUL", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l");
	//CalcCfromL1(mY, MUL, L1);
	mexPrintf(fCalcC_Generic_eq, "mu_0", "eqmY", "w1", "eqMUL", "mu_l1_tild", "eqL1", "psi_l1", "eta_cp");
	//CalcA(atY, C);
	mexPrintf(fCalcA_eq, "r", "Aprev", "eqatY", "eqC");
	//CalcVFromAll(C, L1, L2, T1, T2, A)
	mexPrintf(fCalcVFromAll_eq,
		"mu_0", "eta_cp", "eqC", "eta_cp",
		"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
		"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
		"beta", "eqEV");
}

void PrintStdSolSetNum(SolutionSet * Set, EconParams * econParams)
{
	//L1 = LT1.x0;
	mexPrintf(fConstNum, "L1", Set->L1);

	//T1 = LT1.x1;
	mexPrintf(fConstNum, "T1", Set->T1);

	//CalcL2fromL1(L1);
	mexPrintf(fCalcParam2_Generic, "L2", Set->L2,
		econParams->w1, econParams->w2,
		econParams->mu_l2_tild, econParams->mu_l1_tild,
		Set->L1, econParams->psi_l1, econParams->psi_l2);

	//CalcT2fromT1(T1);
	mexPrintf(fCalcParam2_Generic, "T2", Set->T2,
		econParams->w1, econParams->w2,
		econParams->mu_t2_tild, econParams->mu_t1_tild,
		Set->T1, econParams->psi_t1, econParams->psi_t2);

	//CalcH(L1, T1);
	mexPrintf(fCalcH, "H1", Set->H1, econParams->L_bar, Set->L1, Set->T1);

	//CalcH(L2, T2);
	mexPrintf(fCalcH, "H2", Set->H2, econParams->L_bar, Set->L2, Set->T2);

	//CalcY(H1, H2)
	mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);

	//CalcAtY(Y);
	mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);

	//CalcMY(Y);
	mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);

	//CalcMUL(L1, L2);
	mexPrintf(fCalcMU_Generic, "MUL", Set->MUL, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l);

	//CalcCfromL1(mY, MUL, L1);
	mexPrintf(fCalcC_Generic, Set->C, econParams->mu_0, Set->mY, econParams->w1, Set->MUL, econParams->mu_l1_tild, Set->L1, econParams->psi_l1, econParams->eta_cp);

	//CalcA(atY, C);
	mexPrintf(fCalcA, Set->A, econParams->r, econParams->Aprev, Set->atY, Set->C);

	//CalcVFromAll(C, L1, L2, T1, T2, EV)
	mexPrintf(fCalcVFromAll,
		Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
		econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
		econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
		econParams->beta, Set->EV);
}

// CalcFullSolutionSetA:

void PrintCorASol(SolutionSet * Set, int part, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintCorASolSetEq(part);
		return;
	case printLevel::enum_printLevel_numeric:
		PrintCorASolSetNum(Set, part, econParams);
		return;

	default:
		return;
	}
}

void PrintCorASolSetEq(int part)
{
	if (part == 1)
	{
		//L1 = LT1.x0;
		mexPrintf(fConstNum_eq, "L1", "LT1.x0");
		//T1 = LT1.x1;
		mexPrintf(fConstNum_eq, "T1", "LT1.x1");
	}
	if (part == 2)
	{
		//CalcL2fromL1(L1);
		mexPrintf(fCalcParam2_Generic_eq, "L2", "w1", "w2", "mu_l2_tild", "mu_l1_tild", "eqL1", "psi_l1", "psi_l2");
		//CalcT2fromT1(T1);
		mexPrintf(fCalcParam2_Generic_eq, "T2", "w1", "w2", "mu_t2_tild", "mu_t1_tild", "eqT1", "psi_t1", "psi_t2");
		//CalcH(L1, T1);
		mexPrintf(fCalcH_eq, "H1", "L_bar", "eqL1", "eqT1");
		//CalcH(L2, T2);
		mexPrintf(fCalcH_eq, "H2", "L_bar", "eqL2", "eqT2");
		//CalcY(H1, H2)
		mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
		//CalcAtY(Y);
		mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu", "tau_gamma");
		//CalcMY(Y);
		mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
		//A = amin;
		mexPrintf(fConstNum_eq, "A", "amin");
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA_eq, "Aprev", "eqatY", "eqA", "r");
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll_eq,
			"mu_0", "eta_cp", "eqC", "eta_cp",
			"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
			"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
			"beta", "eqEV");
	}
}

void PrintCorASolSetNum(SolutionSet * Set, int part, EconParams * econParams)
{
	if (part == 1)
	{
		//L1 = LT1.x0;
		mexPrintf(fConstNum, "L1", Set->L1);
		//T1 = LT1.x1;
		mexPrintf(fConstNum, "T1", Set->T1);
	}
	if (part == 2)
	{
		//CalcL2fromL1(L1);
		mexPrintf(fCalcParam2_Generic, "L2", Set->L2,
			econParams->w1, econParams->w2,
			econParams->mu_l2_tild, econParams->mu_l1_tild,
			Set->L1, econParams->psi_l1, econParams->psi_l2);
		//CalcT2fromT1(T1);
		mexPrintf(fCalcParam2_Generic, "T2", Set->T2,
			econParams->w1, econParams->w2,
			econParams->mu_t2_tild, econParams->mu_t1_tild,
			Set->T1, econParams->psi_t1, econParams->psi_t2);
		//CalcH(L1, T1);
		mexPrintf(fCalcH, "H1", Set->H1, econParams->L_bar, Set->L1, Set->T1);
		//CalcH(L2, T2);
		mexPrintf(fCalcH, "H2", Set->H2, econParams->L_bar, Set->L2, Set->T2);
		//CalcY(H1, H2)
		mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);
		//CalcAtY(Y);
		mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);
		//CalcMY(Y);
		mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);
		//A = amin;
		mexPrintf(fConstNum, "A", Set->A);
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA, Set->C, econParams->Aprev, Set->atY, Set->A, econParams->r);
		//CalcVFromAll(C, L1, L2, T1, T2, EV)
		mexPrintf(fCalcVFromAll,
			Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
			econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
			econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
			econParams->beta, Set->EV);
	}
}

// CalcFullSolutionSetH2:

void PrintCorH2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintCorH2SolSetEq(part);
		return;
	case printLevel::enum_printLevel_numeric:
		PrintCorH2SolSetNum(Set, part, econParams);
		return;

	default:
		return;
	}
}

void PrintCorH2SolSetEq(int part)
{
	if (part == 1)
	{
		//L1 = L1LAM2.x0;
		mexPrintf(fConstNum_eq, "L1", "L1LAM2.x0");
		//LAM2 = L1LAM2.x1;
		mexPrintf(fConstNum_eq, "LAM2", "L1LAM2.x1");
		//CalcL2fromL1Lam2(L1, Lam2)
		mexPrintf(fCalcLamParam_Generic_eq, "L2", "mu_l1_tild", "mu_l2_tild", "w2", "w1", "eqLAM2", "(-psi_l2)", "eqL1", "psi_l2", "psi_l1");
		//CalcT2fromT1(T1);
		mexPrintf(fTfromL_eq, "T2", "L_bar", "eqL2");
	}
	if (part == 2)
	{
		//CalcT1fromT2Lam2(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic_eq, "T1", "mu_t1_tild", "mu_t2_tild", "w2", "w1", "eqLAM2", "psi_t1", "eqT2", "psi_t1", "psi_t2");
		//CalcH(L1, T1);
		mexPrintf(fCalcH_eq, "H1", "L_bar", "eqL1", "eqT1");
		//CalcH(L2, T2);
		mexPrintf(fCalcH_eq, "H2", "L_bar", "eqL2", "eqT2");
		//CalcY(H1, H2)
		mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu","tau_gamma");
		//CalcMY(Y);
		mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
		//CalcMUL(L1, L2);
		mexPrintf(fCalcMU_Generic_eq, "MUL", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l");
		//CalcCfromL1(mY, MUL, L1);
		mexPrintf(fCalcC_Generic_eq, "mu_0", "eqmY", "w1", "eqMUL", "mu_l1_tild", "eqL1", "psi_l1", "eta_cp");
		//CalcA(atY, C);
		mexPrintf(fCalcA_eq, "r", "Aprev", "eqatY", "eqC");
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll_eq,
			"mu_0", "eta_cp", "eqC", "eta_cp",
			"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
			"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
			"beta", "eqEV");
	}
}

void PrintCorH2SolSetNum(SolutionSet * Set, int part, EconParams * econParams)
{
	if (part == 1)
	{
		//L1 = L1LAM2.x0;
		mexPrintf(fConstNum, "L1", Set->L1);
		//LAM2 = L1LAM2.x1;
		mexPrintf(fConstNum, "LAM2", Set->LAM2);
		//CalcL2fromL1Lam2(L1, Lam2)
		mexPrintf(fCalcLamParam_Generic, "L2", Set->L2,
			econParams->mu_l1_tild, econParams->mu_l2_tild,
			econParams->w2, econParams->w1, Set->LAM2, (-econParams->psi_l2),
			Set->L1, econParams->psi_l2, econParams->psi_l1);
		//CalcT2fromT1(T1);
		mexPrintf(fTfromL, "T2", Set->T2, econParams->L_bar, Set->L2);
	}
	if (part == 2)
	{
		//CalcT1fromT2Lam2(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic, "T1", Set->T1,
			econParams->mu_t1_tild, econParams->mu_t2_tild,
			econParams->w2, econParams->w1, Set->LAM2, (econParams->psi_t1),
			Set->T2, econParams->psi_t1, econParams->psi_t2);
		//CalcH(L1, T1);
		mexPrintf(fCalcH, "H1", Set->H1, econParams->L_bar, Set->L1, Set->T1);
		//CalcH(L2, T2);
		mexPrintf(fCalcH, "H2", Set->H2, econParams->L_bar, Set->L2, Set->T2);
		//CalcY(H1, H2)
		mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);
		//CalcMY(Y);
		mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);
		//CalcMUL(L1, L2);
		mexPrintf(fCalcMU_Generic, "MUL", Set->MUL, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l);
		//CalcCfromL1(mY, MUL, L1);
		mexPrintf(fCalcC_Generic, Set->C, econParams->mu_0, Set->mY, econParams->w1, Set->MUL, econParams->mu_l1_tild, Set->L1, econParams->psi_l1, econParams->eta_cp);
		//CalcA(atY, C);
		mexPrintf(fCalcA, Set->A, econParams->r, econParams->Aprev, Set->atY, Set->C);
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll,
			Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
			econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
			econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
			econParams->beta, Set->EV);
	}
}

// CalcFullSolutionSetH1:

void PrintCorH1Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintCorH1SolSetEq(part);
		return;
	case printLevel::enum_printLevel_numeric:
		PrintCorH1SolSetNum(Set, part, econParams);
		return;

	default:
		return;
	}
}

void PrintCorH1SolSetEq(int part)
{
	if (part == 1)
	{
		//L2 = L2LAM1.x0;
		mexPrintf(fConstNum_eq, "L2", "L2LAM1.x0");
		//LAM1 = L2LAM1.x1;
		mexPrintf(fConstNum_eq, "LAM1", "L2LAM1.x1");
		//CalcL1fromL2Lam1(L2, Lam1)
		mexPrintf(fCalcLamParam_Generic_eq, "L1", "mu_l2_tild", "mu_l1_tild", "w1", "w2", "eqLAM1", "(-psi_l1)", "eqL2", "psi_l1", "psi_l2");
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL_eq, "T1", "L_bar", "eqL1");
	}
	if (part == 2)
	{
		//CalcT2fromT1Lam1(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic_eq, "T2", "mu_t2_tild", "mu_t1_tild", "w1", "w2", "eqLAM1", "psi_t2", "eqT1", "psi_t2", "psi_t1");
		//CalcH(L1, T1);
		mexPrintf(fCalcH_eq, "H1", "L_bar", "eqL1", "eqT1");
		//CalcH(L2, T2);
		mexPrintf(fCalcH_eq, "H2", "L_bar", "eqL2", "eqT2");
		//CalcY(H1, H2)
		mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu", "tau_gamma");
		//CalcMY(Y);
		mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
		//CalcMUL(L1, L2);
		mexPrintf(fCalcMU_Generic_eq, "MUL", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l");
		//CalcCfromL1(mY, MUL, L1);
		mexPrintf(fCalcC_Generic_eq, "mu_0", "eqmY", "w1", "eqMUL", "mu_l1_tild", "eqL1", "psi_l1", "eta_cp");
		//CalcA(atY, C);
		mexPrintf(fCalcA_eq, "r", "Aprev", "eqatY", "eqC");
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll_eq,
			"mu_0", "eta_cp", "eqC", "eta_cp",
			"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
			"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
			"beta", "eqEV");
	}
}

void PrintCorH1SolSetNum(SolutionSet * Set, int part, EconParams * econParams)
{
	if (part == 1)
	{
		//L2 = L2LAM1.x0;
		mexPrintf(fConstNum, "L2", Set->L2);
		//LAM1 = L2LAM1.x1;
		mexPrintf(fConstNum, "LAM1", Set->LAM1);
		//CalcL1fromL2Lam1(L2, Lam1)
		mexPrintf(fCalcLamParam_Generic, "L1", Set->L1,
			econParams->mu_l2_tild, econParams->mu_l1_tild,
			econParams->w1, econParams->w2, Set->LAM1, (-econParams->psi_l1),
			Set->L2, econParams->psi_l1, econParams->psi_l2);
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL, "T1", Set->T1, econParams->L_bar, Set->L1);
	}
	if (part == 2)
	{
		//CalcT2fromT1Lam1(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic, "T2", Set->T2,
			econParams->mu_t2_tild, econParams->mu_t1_tild,
			econParams->w1, econParams->w2, Set->LAM1, (econParams->psi_t2),
			Set->T1, econParams->psi_t2, econParams->psi_t1);
		//CalcH(L1, T1);
		mexPrintf(fCalcH, "H1", Set->H1, econParams->L_bar, Set->L1, Set->T1);
		//CalcH(L2, T2);
		mexPrintf(fCalcH, "H2", Set->H2, econParams->L_bar, Set->L2, Set->T2);
		//CalcY(H1, H2)
		mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);
		//CalcMY(Y);
		mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);
		//CalcMUL(L1, L2);
		mexPrintf(fCalcMU_Generic, "MUL", Set->MUL, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l);
		//CalcCfromL2(mY, MUL, L1);
		mexPrintf(fCalcC_Generic, Set->C, econParams->mu_0, Set->mY, econParams->w2, Set->MUL, econParams->mu_l2_tild, Set->L2, econParams->psi_l2, econParams->eta_cp);
		//CalcA(atY, C);
		mexPrintf(fCalcA, Set->A, econParams->r, econParams->Aprev, Set->atY, Set->C);
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll,
			Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
			econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
			econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
			econParams->beta, Set->EV);
	}
}

// CalcFullSolutionSetA_H2:

void PrintCorA_H2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintCorA_H2SolSetEq(part);
		return;
	case printLevel::enum_printLevel_numeric:
		PrintCorA_H2SolSetNum(Set, part, econParams);
		return;

	default:
		return;
	}
}

void PrintCorA_H2SolSetEq(int part)
{
	if (part == 1)
	{
		//L1 = L1LAM2.x0;
		mexPrintf(fConstNum_eq, "L1", "L1LAM2.x0");
		//LAM2 = L1LAM2.x1;
		mexPrintf(fConstNum_eq, "LAM2", "L1LAM2.x1");
		//CalcL2fromL1Lam2(L1, Lam2)
		mexPrintf(fCalcLamParam_Generic_eq, "L2", "mu_l1_tild", "mu_l2_tild", "w2", "w1", "eqLAM2", "(-psi_l2)", "eqL1", "psi_l2", "psi_l1");
		//CalcT2fromT1(T1);
		mexPrintf(fTfromL_eq, "T2", "L_bar", "eqL2");
	}
	if (part == 2)
	{
		//CalcT1fromT2Lam2(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic_eq, "T1", "mu_t1_tild", "mu_t2_tild", "w2", "w1", "eqLAM2", "psi_t1", "eqT2", "psi_t1", "psi_t2");
		//CalcH(L1, T1);
		mexPrintf(fCalcH_eq, "H1", "L_bar", "eqL1", "eqT1");
		//CalcH(L2, T2);
		mexPrintf(fCalcH_eq, "H2", "L_bar", "eqL2", "eqT2");
		//CalcY(H1, H2)
		mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu", "tau_gamma");
		//CalcMY(Y);
		mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
		//A = amin;
		mexPrintf(fConstNum_eq, "A", "amin");
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA_eq, "Aprev", "eqatY", "eqA", "r");
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll_eq,
			"mu_0", "eta_cp", "eqC", "eta_cp",
			"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
			"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
			"beta", "eqEV");
	}
}

void PrintCorA_H2SolSetNum(SolutionSet * Set, int part, EconParams * econParams)
{
	if (part == 1)
	{
		//L1 = L1LAM2.x0;
		mexPrintf(fConstNum, "L1", Set->L1);
		//LAM2 = L1LAM2.x1;
		mexPrintf(fConstNum, "LAM2", Set->LAM2);
		//CalcL2fromL1Lam2(L1, Lam2)
		mexPrintf(fCalcLamParam_Generic, "L2", Set->L2,
			econParams->mu_l1_tild, econParams->mu_l2_tild,
			econParams->w2, econParams->w1, Set->LAM2, (-econParams->psi_l2),
			Set->L1, econParams->psi_l2, econParams->psi_l1);
		//CalcT2fromT1(T1);
		mexPrintf(fTfromL, "T2", Set->T2, econParams->L_bar, Set->L2);
	}
	if (part == 2)
	{
		//CalcT1fromT2Lam2(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic, "T1", Set->T1,
			econParams->mu_t1_tild, econParams->mu_t2_tild,
			econParams->w2, econParams->w1, Set->LAM2, (econParams->psi_t1),
			Set->T2, econParams->psi_t1, econParams->psi_t2);
		//CalcH(L1, T1);
		mexPrintf(fCalcH, "H1", Set->H1, econParams->L_bar, Set->L1, Set->T1);
		//CalcH(L2, T2);
		mexPrintf(fCalcH, "H2", Set->H2, econParams->L_bar, Set->L2, Set->T2);
		//CalcY(H1, H2)
		mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);
		//CalcMY(Y);
		mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);
		//A = amin;
		mexPrintf(fConstNum, "A", Set->A);
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA, Set->C, econParams->Aprev, Set->atY, Set->A, econParams->r);
		//CalcVFromAll(C, L1, L2, T1, T2, EV)
		mexPrintf(fCalcVFromAll,
			Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
			econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
			econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
			econParams->beta, Set->EV);
	}
}

// CalcFullSolutionSetA_H1:

void PrintCorA_H1Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintCorA_H1SolSetEq(part);
		return;
	case printLevel::enum_printLevel_numeric:
		PrintCorA_H1SolSetNum(Set, part, econParams);
		return;

	default:
		return;
	}
}

void PrintCorA_H1SolSetEq(int part)
{
	if (part == 1)
	{
		//L2 = L2LAM1.x0;
		mexPrintf(fConstNum_eq, "L2", "L2LAM1.x0");
		//LAM1 = L2LAM1.x1;
		mexPrintf(fConstNum_eq, "LAM1", "L2LAM1.x1");
		//CalcL1fromL2Lam1(L2, Lam1)
		mexPrintf(fCalcLamParam_Generic_eq, "L1", "mu_l2_tild", "mu_l1_tild", "w1", "w2", "eqLAM1", "(-psi_l1)", "eqL2", "psi_l1", "psi_l2");
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL_eq, "T1", "L_bar", "eqL1");
	}
	if (part == 2)
	{
		//CalcT2fromT1Lam1(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic_eq, "T2", "mu_t2_tild", "mu_t1_tild", "w1", "w2", "eqLAM1", "psi_t2", "eqT1", "psi_t2", "psi_t1");
		//CalcH(L1, T1);
		mexPrintf(fCalcH_eq, "H1", "L_bar", "eqL1", "eqT1");
		//CalcH(L2, T2);
		mexPrintf(fCalcH_eq, "H2", "L_bar", "eqL2", "eqT2");
		//CalcY(H1, H2)
		mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu", "tau_gamma");
		//CalcMY(Y);
		mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
		//A = amin;
		mexPrintf(fConstNum_eq, "A", "amin");
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA_eq, "Aprev", "eqatY", "eqA", "r");
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll_eq,
			"mu_0", "eta_cp", "eqC", "eta_cp",
			"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
			"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
			"beta", "eqEV");
	}
}

void PrintCorA_H1SolSetNum(SolutionSet * Set, int part, EconParams * econParams)
{
	if (part == 1)
	{
		//L2 = L2LAM1.x0;
		mexPrintf(fConstNum, "L2", Set->L2);
		//LAM1 = L2LAM1.x1;
		mexPrintf(fConstNum, "LAM1", Set->LAM1);
		//CalcL1fromL2Lam1(L2, Lam1)
		mexPrintf(fCalcLamParam_Generic, "L1", Set->L1,
			econParams->mu_l2_tild, econParams->mu_l1_tild,
			econParams->w1, econParams->w2, Set->LAM1, (-econParams->psi_l1),
			Set->L2, econParams->psi_l1, econParams->psi_l2);
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL, "T1", Set->T1, econParams->L_bar, Set->L1);
	}
	if (part == 2)
	{
		//CalcT2fromT1Lam1(L2, Lam2)
		mexPrintf(fCalcLamParam_Generic, "T2", Set->T2,
			econParams->mu_t2_tild, econParams->mu_t1_tild,
			econParams->w1, econParams->w2, Set->LAM1, (econParams->psi_t2),
			Set->T1, econParams->psi_t2, econParams->psi_t1);
		//CalcH(L1, T1);
		mexPrintf(fCalcH, "H1", Set->H1, econParams->L_bar, Set->L1, Set->T1);
		//CalcH(L2, T2);
		mexPrintf(fCalcH, "H2", Set->H2, econParams->L_bar, Set->L2, Set->T2);
		//CalcY(H1, H2)
		mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);
		//CalcMY(Y);
		mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);
		//A = amin;
		mexPrintf(fConstNum, "A", Set->A);
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA, Set->C, econParams->Aprev, Set->atY, Set->A, econParams->r);
		//CalcVFromAll(C, L1, L2, T1, T2, EV)
		mexPrintf(fCalcVFromAll,
			Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
			econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
			econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
			econParams->beta, Set->EV);
	}
}

// CalcFullSolutionSetH1_H2:

void PrintCorH1_H2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintCorH1_H2SolSetEq(part);
		return;
	case printLevel::enum_printLevel_numeric:
		PrintCorH1_H2SolSetNum(Set, part, econParams);
		return;

	default:
		return;
	}
}

void PrintCorH1_H2SolSetEq(int part)
{
	if (part == 1)
	{
		//L1 = L1L2A.x0;
		mexPrintf(fConstNum_eq, "L1", "L1L2A.x0");
		//L2 = L1L2A.x1;
		mexPrintf(fConstNum_eq, "L2", "L1L2A.x1");
		//A = L1L2A.x2;
		mexPrintf(fConstNum_eq, "A", "L1L2A.x2");
		//CalcT2fromT1(T1);
		mexPrintf(fTfromL_eq, "T2", "L_bar", "eqL2");
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL_eq, "T1", "L_bar", "eqL1");
	}
	if (part == 2)
	{
		//H1 = 0;
		mexPrintf(fConstNum_eq, "H1", "0");
		//H2 = 0;
		mexPrintf(fConstNum_eq, "H2", "0");
		//CalcY(H1, H2)
		mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu", "tau_gamma");
		//CalcMY(Y);
		mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
		//A = amin;
		mexPrintf(fConstNum_eq, "A", "amin");
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA_eq, "Aprev", "eqatY", "eqA", "r");
	}
	if (part == 4)
	{
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll_eq,
			"mu_0", "eta_cp", "eqC", "eta_cp",
			"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
			"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
			"beta", "eqEV");
	}
}

void PrintCorH1_H2SolSetNum(SolutionSet * Set, int part, EconParams * econParams)
{
	if (part == 1)
	{
		//L1 = L1L2A.x0;
		mexPrintf(fConstNum, "L1", Set->L1);
		//L2 = L1L2A.x1;
		mexPrintf(fConstNum, "L2", Set->L2);
		//A = L1L2A.x2;
		mexPrintf(fConstNum, "A", Set->A);
		//CalcT2fromT1(T1);
		mexPrintf(fTfromL, "T2", Set->T2, econParams->L_bar, Set->L2);
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL, "T1", Set->T1, econParams->L_bar, Set->L1);
	}
	if (part == 2)
	{
		//H1 = 0;
		mexPrintf(fConstNum, "H1", Set->H1);
		//H2 = 0;
		mexPrintf(fConstNum, "H2", Set->H2);
		//CalcY(H1, H2)
		mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);
		//CalcMY(Y);
		mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);
		//A = amin;
		mexPrintf(fConstNum, "A", Set->A);
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA, Set->C, econParams->Aprev, Set->atY, Set->A, econParams->r);
	}
	if (part == 4)
	{
		//CalcVFromAll(C, L1, L2, T1, T2, EV)
		mexPrintf(fCalcVFromAll,
			Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
			econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
			econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
			econParams->beta, Set->EV);
	}
}

// CalcFullSolutionSetA_H1_H2:

void PrintCorA_H1_H2Sol(SolutionSet * Set, int part, EconParams * econParams, printLevel level)
{
	switch (level)
	{
	case printLevel::enum_printLevel_equation:
		PrintCorA_H1_H2SolSetEq(part);
		return;
	case printLevel::enum_printLevel_numeric:
		PrintCorA_H1_H2SolSetNum(Set, part, econParams);
		return;

	default:
		return;
	}
}

void PrintCorA_H1_H2SolSetEq(int part)
{
	if (part == 1)
	{
		//L1 = L1L2A.x0;
		mexPrintf(fConstNum_eq, "L1", "L1L2.x0");
		//L2 = L1L2A.x1;
		mexPrintf(fConstNum_eq, "L2", "L1L2.x1");
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL_eq, "T1", "L_bar", "eqL1");
		//CalcT2fromL2(T1);
		mexPrintf(fTfromL_eq, "T2", "L_bar", "eqL2");
	}
	if (part == 2)
	{
		//H1 = 0;
		mexPrintf(fConstNum_eq, "H1", "0");
		//H2 = 0;
		mexPrintf(fConstNum_eq, "H2", "0");
		//CalcY(H1, H2)
		mexPrintf(fCalcY_eq, "tau_b", "w1", "eqH1", "w2", "eqH2");
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY_eq, "chi", "eqY", "taumu", "tau_gamma");
		//CalcMY(Y);
		mexPrintf(fCalcMY_eq, "chi", "taumu", "eqY", "taumu");
		//A = amin;
		mexPrintf(fConstNum_eq, "A", "amin");
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA_eq, "Aprev", "eqatY", "eqA", "r");
		//CalcVFromAll(C, L1, L2, T1, T2, A)
		mexPrintf(fCalcVFromAll_eq,
			"mu_0", "eta_cp", "eqC", "eta_cp",
			"rhou_l", "mu_l1", "eqL1", "psi_l1", "mu_l2", "eqL2", "psi_l2", "rhou_l",
			"rhou_t", "mu_t1", "eqT1", "psi_t1", "mu_t2", "eqT2", "psi_t2", "rhou_t",
			"beta", "eqEV");
	}
}

void PrintCorA_H1_H2SolSetNum(SolutionSet * Set, int part, EconParams * econParams)
{
	if (part == 1)
	{
		//L1 = L1L2A.x0;
		mexPrintf(fConstNum, "L1", Set->L1);
		//L2 = L1L2A.x1;
		mexPrintf(fConstNum, "L2", Set->L2);
		//CalcT1fromL1(T1);
		mexPrintf(fTfromL, "T1", Set->T1, econParams->L_bar, Set->L1);
		//CalcT2fromL2(T1);
		mexPrintf(fTfromL, "T2", Set->T2, econParams->L_bar, Set->L2);
	}
	if (part == 2)
	{
		//H1 = 0;
		mexPrintf(fConstNum, "H1", Set->H1);
		//H2 = 0;
		mexPrintf(fConstNum, "H2", Set->H2);
		//CalcY(H1, H2)
		mexPrintf(fCalcY, Set->Y, econParams->tau_b, econParams->w1, Set->H1, econParams->w2, Set->H2);
	}
	if (part == 3)
	{
		//CalcAtY(Y);
		mexPrintf(fCalcAtY, Set->atY, econParams->chi, Set->Y, econParams->taumu, econParams->tau_gamma);
		//CalcMY(Y);
		mexPrintf(fCalcMY, Set->mY, econParams->chi, econParams->taumu, Set->Y, econParams->taumu);
		//A = amin;
		mexPrintf(fConstNum, "A", Set->A);
		//CalcCfromA(A, atY);
		mexPrintf(fCalcCfromA, Set->C, econParams->Aprev, Set->atY, Set->A, econParams->r);
		//CalcVFromAll(C, L1, L2, T1, T2, EV)
		mexPrintf(fCalcVFromAll,
			Set->V, econParams->mu_0, econParams->eta_cp, Set->C, econParams->eta_cp,
			econParams->rhou_l, econParams->mu_l1, Set->L1, econParams->psi_l1, econParams->mu_l2, Set->L2, econParams->psi_l2, econParams->rhou_l,
			econParams->rhou_t, econParams->mu_t1, Set->T1, econParams->psi_t1, econParams->mu_t2, Set->T2, econParams->psi_t2, econParams->rhou_t,
			econParams->beta, Set->EV);
	}
}

