#include <sys/times.h>
#include <unistd.h>
#include "global.h"
#include "init.h"
#include "relaxation.h"
#include "subset.h"
#include "feasibility.h"

int num_demand_points;
int num_service_set_size;
int num_service_points;
int seed = 345345;
int points_to_add = 1, *farthest;

double radius_lower_limit;

static void print_current_best_solution(relaxation_matrix *r, subset *s,
				 double solution_value)
{
#ifdef VERBOSE
  int i;
#endif

  printf("Current best solution: %lf\n", solution_value);
#ifdef VERBOSE
  for (i=0; i<s->size; i++)
    {
      printf("x: %d\n", r->circles[s->indices[i]].index);
      printf("r: %6lf\n\n", r->circles[s->indices[i]].r);
    }
#endif
  fflush(stdout);
}

int main(int argc, char *argv[])
{
  subset reduced, solution_reduced;
  relaxation_matrix r;
  int max_problem = 0;
  int k, count = 0;
  struct tms before, after;
  int ticks_per_second = sysconf (_SC_CLK_TCK);

  if((argc < 3) || (argc > 5))
    {
      fprintf(stderr, "usage: %s filename num_service_points "
	      "[points_to_add] [guess]\n",
	      argv[0]);
      exit(0);
    }
  
  num_service_points = atoi(argv[2]);

  srandom(seed);

  if (argc >= 4)
    points_to_add = atoi(argv[3]);
  farthest = (int *)malloc(points_to_add*sizeof(int));

  if (argc >= 5)
      sscanf(argv[4], "%lf", &radius_lower_limit);
  else
    radius_lower_limit = EPSILON;
  
  init_point_array(argv[1]);

  times(&before);
  
  subset_init(&solution_reduced);
  
  k = num_service_points;
  
  subset_init(&reduced);
  subset_random(&reduced, 1);
  relaxation_matrix_init(&r, &reduced);
  
  while (1)
    {
      max_problem = reduced.size > max_problem ? reduced.size : max_problem;
      count ++;
      if( feasibility(&r, k, &solution_reduced) )
      {
        if (check_feasibility(&r, &solution_reduced, radius_lower_limit))
	  goto Cont;
	else
	{
	  int i;

	  for (i=0; i<points_to_add; i++)
	  {
		  if (farthest[i] >= 0)
			  subset_add(&reduced, farthest[i]);
	  }
	  relaxation_matrix_zero(&r);
	  relaxation_matrix_init(&r, &reduced);
	}
      }
      else
      {
        radius_lower_limit = find_next_bound(&reduced);
	relaxation_matrix_zero(&r);
	relaxation_matrix_init(&r, &reduced);
      }
      print_current_best_solution(&r, &solution_reduced, radius_lower_limit);
    }
 Cont:
  printf("Okay, we have solved the problem for %d service points\n", k);
  printf("------------------------------------------------------\n");

  times(&after);
  printf("\nResult: %lf  Elapsed Time: %lf seconds  max_problem: %d"
	 "  no_sub_problems: %d\n",
	 radius_lower_limit,
	 ((double)after.tms_utime - (double)before.tms_utime) /
	 ticks_per_second +
	 ((double)after.tms_stime - (double)before.tms_stime) /
	 ticks_per_second, max_problem, count);
  fflush(stdout);
  
  return 1;
}
