#include <sys/times.h>
#include <unistd.h>
#include "global.h"
#include "init.h"
#include "relaxation.h"
#include "subset.h"
#include "feasibility.h"

int num_demand_points;
int num_service_set_size;
int num_service_points;
int seed = 345345;
int points_to_add = 1, *farthest;

double coverage_distance = DBL_MAX;

void print_current_best_solution(relaxation_matrix *r, subset *s,
				 double solution_value)
{
#ifdef VERBOSE
  int i;
#endif

  printf("Current best solution: %lf\n", solution_value);
#ifdef VERBOSE
  for (i=0; i<s->size; i++)
    {
      printf("x: %d\n", r->circles[s->indices[i]].index);
      printf("r: %6lf\n\n", r->circles[s->indices[i]].r);
    }
#endif
  fflush(stdout);
}

int main(int argc, char *argv[])
{
  subset reduced, solution_reduced;
  relaxation_matrix r;
  int max_problem = 0;
  double solution_value, new_solution_value;
  int k, count = 0;
  struct tms before, after;
  int ticks_per_second = sysconf (_SC_CLK_TCK);
  double upper_bound = DBL_MAX,
	 lower_bound = 0, half;

  if((argc < 3) || (argc > 6))
    {
      fprintf(stderr, "usage: %s filename num_service_points "
	      "[points_to_add] [upper_bound] [lower_bound]\n",
	      argv[0]);
      exit(0);
    }
  
  num_service_points = atoi(argv[2]);

  srandom(seed);

  if (argc >= 4)
    points_to_add = atoi(argv[3]);
  farthest = (int *)malloc(points_to_add * sizeof(int));

  if (argc >= 5)
      sscanf(argv[4], "%lf", &upper_bound);
  else
    upper_bound = DBL_MAX;

  if (argc >= 6)
	  sscanf(argv[5], "%lf", &lower_bound);
  else
	  lower_bound = 0;
  
  init_point_array(argv[1]);

  times(&before);
  
  subset_init(&solution_reduced);
  
  k = num_service_points;
  
  subset_init(&reduced);
  subset_random(&reduced, 1);
  relaxation_matrix_init(&r, &reduced);
  
  while (upper_bound - lower_bound > 0.00000001)
    {
      int feasible;

      max_problem = reduced.size > max_problem ? reduced.size : max_problem;
      half = (upper_bound + lower_bound) / 2.0;
      coverage_distance = half;
      relaxation_matrix_zero(&r);
      relaxation_matrix_init(&r, &reduced);
      feasible = feasibility(&r, k, &solution_reduced);
      count++;

      if (feasible)
      {
More:
	      solution_value = get_solution_value(&r, &solution_reduced);
	      if (check_feasibility(&r, &solution_reduced,
				      &new_solution_value))
	      {
		      upper_bound = new_solution_value;
		      printf("upper_bound: %lf\n", upper_bound);
		      fflush(stdout);
		      coverage_distance = new_solution_value / (1 + EPSILON);
		      relaxation_matrix_zero(&r);
		      relaxation_matrix_init(&r, &reduced);
		      feasible = feasibility(&r, k, &solution_reduced);
		      count++;
		      if (!feasible)
			      goto Cont;
		      else
			      goto More;
	      }
	      else
	      {
		      int i;

		      for (i=0; i<points_to_add; i++)
		      {
			      if (farthest[i] >= 0)
				      subset_add(&reduced, farthest[i]);
		      }
	      }
      }
      else
      {
	      lower_bound = half;
	      printf("lower_bound: %lf\n", lower_bound);
	      fflush(stdout);
      }
    }
 Cont:
  printf("Okay, we have solved the problem for %d service points\n", k);
  printf("------------------------------------------------------\n");

  times(&after);
  printf("\nResult: %lf  Elapsed Time: %lf seconds  max_problem: %d"
	 "  no_sub_problems: %d\n",
	 coverage_distance,
	 ((double)after.tms_utime - (double)before.tms_utime) /
	 ticks_per_second +
	 ((double)after.tms_stime - (double)before.tms_stime) /
	 ticks_per_second, max_problem, count);
  fflush(stdout);
  
  return 1;
}
