function []=dft_demo()
% Demonstration of properties of the Discrete Fourier Transform (DFT).
% The image is denoted by A and the DFT of the image A is denoted by B.


% Demonstration of effect of the "fftshift" command on the amplitude and 
% on the spectrum of the image.
%--------------------------------------------------------------------------
N1=101;
N2=101;
M1=19;
M2=19;
A=zeros(N1,N2);
A((N1-M1)/2+1:(N1+M1)/2,(N2-M2)/2+1:(N2+M2)/2)=255;
B=fft2(A);
magnitude_B=abs(B);
log_magnitude_B=20*log10(magnitude_B+1); % 1 - in order 
                                           % to prevent log(0)
B_fftshift=fftshift(B);
magnitude_B_fftshift=abs(B_fftshift);
log_magnitude_B_fftshift=20*log10(magnitude_B_fftshift+1);

figure;
subplot(1,3,1);
imshow(A,[]);
title('The original image A');
subplot(1,3,2);
imshow(log_magnitude_B,[]);
title('|DFT(A)| [dB]');
subplot(1,3,3);
imshow(log_magnitude_B_fftshift,[]);
title('|DFT(A)| [dB] after ''fftshift''');

% Linearity - 3 demonstrations
% (two horizontal boxes, two vertical boxes and four boxes)
%--------------------------------------------------------------------------
% Two horizontal boxes
delta2=40;
B=fft2(A);

A2_hor=zeros(N1,N2);
A2_hor((N1-M1)/2+1:(N1+M1)/2,(N2-delta2-M2)/2:(N2-delta2+M2)/2+1)=255;
A2_hor((N1-M1)/2+1:(N1+M1)/2,(N2+delta2-M2)/2:(N2+delta2+M2)/2+1)=255;
B2_hor=fft2(A2_hor);

figure;
subplot(2,2,1);
imshow(A,[]);
title('One box');
subplot(2,2,2);
imshow(A2_hor,[]);
title('Two horizontal boxes');
subplot(2,2,3);
imshow(20*log10(abs(fftshift(B))+1),[]);
title('|DFT\{One box\}| [dB]');
subplot(2,2,4);
imshow(20*log10(abs(fftshift(B2_hor))+1),[]);
title('|DFT\{Two horizontal boxes\}| [dB]');

% Two vertical boxes
delta1=40;
B=fft2(A);

A2_ver=zeros(N1,N2);
A2_ver((N1-delta1-M1)/2+1:(N1-delta1+M1)/2,(N2-M2)/2:(N2+M2)/2+1)=255;
A2_ver((N1+delta1-M1)/2+1:(N1+delta1+M1)/2,(N2-M2)/2:(N2+M2)/2+1)=255;
B2_ver=fft2(A2_ver);

figure;
subplot(2,2,1);
imshow(A,[]);
title('One box');
subplot(2,2,2);
imshow(A2_ver,[]);
title('Two vertical boxes');
subplot(2,2,3);
imshow(20*log10(abs(fftshift(B))+1),[]);
title('|DFT\{One box\}| [dB]');
subplot(2,2,4);
imshow(20*log10(abs(fftshift(B2_ver))+1),[]);
title('|DFT\{Two vertical boxes\}| [dB]');

% Four boxes
delta1=40;
delta2=40;
B=fft2(A);

A4=zeros(N1,N2);
A4((N1-delta1-M1)/2+1:(N1-delta1+M1)/2,(N2-delta2-M2)/2:(N2-delta2+M2)/2+1)=255;
A4((N1+delta1-M1)/2+1:(N1+delta1+M1)/2,(N2-delta2-M2)/2:(N2-delta2+M2)/2+1)=255;
A4((N1-delta1-M1)/2+1:(N1-delta1+M1)/2,(N2+delta2-M2)/2:(N2+delta2+M2)/2+1)=255;
A4((N1+delta1-M1)/2+1:(N1+delta1+M1)/2,(N2+delta2-M2)/2:(N2+delta2+M2)/2+1)=255;
B4=fft2(A4);

figure;
subplot(2,2,1);
imshow(A,[]);
title('One box');
subplot(2,2,2);
imshow(A4,[]);
title('Four boxes');
subplot(2,2,3);
imshow(20*log10(abs(fftshift(B))+1),[]);
title('|DFT\{One box\}| [dB]');
subplot(2,2,4);
imshow(20*log10(abs(fftshift(B4))+1),[]);
title('|DFT\{Four boxes\}| [dB]');

% Scaling (reduction and enlargement)
%--------------------------------------------------------------------------
N1=101;
N2=101;
M1=19;
M2=19;
A=zeros(N1,N2);
A((N1-M1)/2+1:(N1+M1)/2,(N2-M2)/2+1:(N2+M2)/2)=255;
B=fft2(A);

% Reduction
M1_small=9;
M2_small=9;
A_small=zeros(N1,N2);
A_small((N1-M1_small)/2+1:(N1+M1_small)/2,...
        (N2-M2_small)/2+1:(N2+M2_small)/2)=255;
B_of_small_A=fft2(A_small);

% Enlargement
M1_large=39;
M2_large=39;
A_large=zeros(N1,N2);
A_large((N1-M1_large)/2+1:(N1+M1_large)/2,...
        (N2-M2_large)/2+1:(N2+M2_large)/2)=255;
B_of_large_A=fft2(A_large);

figure;
subplot(2,3,1);
imshow(A,[]);
title('The original image A');
subplot(2,3,2);
imshow(A_small,[]);
title('Reduced image');
subplot(2,3,3);
imshow(A_large,[]);
title('Enlarged image');
subplot(2,3,4);
imshow(20*log10(abs(fftshift(B))+1),[]);
title('|DFT(A)| [dB]');
subplot(2,3,5);
imshow(20*log10(abs(fftshift(B_of_small_A))+1),[]);
title('|DFT\{Reduced A\}| [dB]');
subplot(2,3,6);
imshow(20*log10(abs(fftshift(B_of_large_A))+1),[]);
title('|DFT\{Enlarged A\}| [dB]');

% Rotation
%--------------------------------------------------------------------------
theta=45; % in degrees
A_rotated=imrotate(A,theta,'bicubic','crop');

B=fft2(A);
B_of_rotated_A=fft2(A_rotated);

figure;
subplot(2,2,1);
imshow(A,[]);
title('The original image A');
subplot(2,2,2);
imshow(A_rotated,[]);
title('Rotated image');
subplot(2,2,3);
imshow(20*log10(abs(fftshift(B))+1),[]);
title('|DFT(A)| [dB]');
subplot(2,2,4);
imshow(20*log10(abs(fftshift(B_of_rotated_A))+1),[]);
title('|DFT\{Rotated A\}| [dB]');

% Exchange between magnitude and phase
%--------------------------------------------------------------------------
% Load the images of Lena and Barbara
lena=imread('lena.gif');
barbara=imread('barbara.gif');

% Exchange between magnitude and phase
lena_spectrum=fft2(im2double(lena));
lena_magnitude=abs(lena_spectrum);
lena_phase=angle(lena_spectrum);
barbara_spectrum=fft2(im2double(barbara));
barbara_magnitude=abs(barbara_spectrum);
barbara_phase=angle(barbara_spectrum);

% Magnitude of Lena, phase of Barbara
lena_barbara_spectrum=lena_magnitude.*exp(1i*barbara_phase);
lena_barbara=ifft2(lena_barbara_spectrum);

% Magnitude of Barbara, phase of Lena
barbara_lena_spectrum=barbara_magnitude.*exp(1i*lena_phase);
barbara_lena=ifft2(barbara_lena_spectrum);

figure;
subplot(2,2,1);
imshow(lena,[]);
title('Lena');
subplot(2,2,2);
imshow(barbara,[]);
title('Barbara');
subplot(2,2,3);
imshow(abs(barbara_lena),[]);
title('Magnitude of Barbara, phase of Lena');
subplot(2,2,4);
imshow(abs(lena_barbara),[]);
title('Magnitude of Lena, phase of Barbara');